// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/AddNodeAction.h"
#include "Actions/ActionRegistry.h"

#include "Engine/Blueprint.h"
#include "EdGraph/EdGraph.h"
#include "EdGraph/EdGraphNode.h"
#include "EdGraph/EdGraphPin.h"
#include "EdGraphSchema_K2.h"
#include "K2Node_CallFunction.h"
#include "K2Node_IfThenElse.h"
#include "K2Node_ExecutionSequence.h"
#include "K2Node_CustomEvent.h"
#include "K2Node_Event.h"
#include "K2Node_VariableGet.h"
#include "K2Node_VariableSet.h"
#include "K2Node_SpawnActorFromClass.h"
#include "K2Node_DynamicCast.h"
#include "K2Node_MakeArray.h"
#include "K2Node_Self.h"
#include "K2Node_Timeline.h"
#include "K2Node_MacroInstance.h"
#include "Kismet2/BlueprintEditorUtils.h"

#include "Kismet/KismetSystemLibrary.h"
#include "Kismet/KismetMathLibrary.h"
#include "Kismet/GameplayStatics.h"
#include "Kismet/KismetStringLibrary.h"
#include "Kismet/KismetArrayLibrary.h"

#include "GameFramework/Actor.h"
#include "GameFramework/Pawn.h"
#include "GameFramework/Character.h"

#include "AssetRegistry/AssetRegistryModule.h"
#include "Engine/BlueprintGeneratedClass.h"

// ─── Helpers ─────────────────────────────────────────────────────────────────

static UBlueprint* FindBlueprintByName(const FString& NameOrPath)
{
	// Try loading by path first
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}

	// Search asset registry
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);

	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

static UEdGraph* FindGraph(UBlueprint* BP, const FString& GraphName)
{
	TArray<UEdGraph*> Graphs;
	BP->GetAllGraphs(Graphs);
	for (UEdGraph* G : Graphs)
	{
		if (G->GetName().Equals(GraphName, ESearchCase::IgnoreCase))
		{
			return G;
		}
	}
	// Default to first UbergraphPage (EventGraph)
	if (BP->UbergraphPages.Num() > 0)
	{
		return BP->UbergraphPages[0];
	}
	return nullptr;
}

static UClass* ResolveClass(const FString& ClassName)
{
	static TMap<FString, UClass*> LookupCache;
	if (LookupCache.Num() == 0)
	{
		LookupCache.Add(TEXT("KismetSystemLibrary"), UKismetSystemLibrary::StaticClass());
		LookupCache.Add(TEXT("KismetMathLibrary"), UKismetMathLibrary::StaticClass());
		LookupCache.Add(TEXT("GameplayStatics"), UGameplayStatics::StaticClass());
		LookupCache.Add(TEXT("KismetStringLibrary"), UKismetStringLibrary::StaticClass());
		LookupCache.Add(TEXT("KismetArrayLibrary"), UKismetArrayLibrary::StaticClass());
		LookupCache.Add(TEXT("Actor"), AActor::StaticClass());
		LookupCache.Add(TEXT("Pawn"), APawn::StaticClass());
		LookupCache.Add(TEXT("Character"), ACharacter::StaticClass());
	}

	// Direct lookup
	if (UClass** Found = LookupCache.Find(ClassName))
	{
		return *Found;
	}

	// Try TObjectIterator fallback
	for (TObjectIterator<UClass> It; It; ++It)
	{
		FString Name = (*It)->GetName();
		if (Name.Equals(ClassName, ESearchCase::IgnoreCase) ||
			Name.Equals(TEXT("U") + ClassName, ESearchCase::IgnoreCase) ||
			Name.Equals(TEXT("A") + ClassName, ESearchCase::IgnoreCase))
		{
			LookupCache.Add(ClassName, *It);
			return *It;
		}
	}

	// Try Blueprint-generated class: AI sends 'BP_AITurret' but the generated class is 'BP_AITurret_C'
	FString ClassNameWithC = ClassName;
	if (!ClassNameWithC.EndsWith(TEXT("_C")))
	{
		ClassNameWithC += TEXT("_C");
	}
	for (TObjectIterator<UClass> It; It; ++It)
	{
		if ((*It)->GetName().Equals(ClassNameWithC, ESearchCase::IgnoreCase))
		{
			LookupCache.Add(ClassName, *It);
			return *It;
		}
	}

	// Try loading the Blueprint asset and getting its generated class
	{
		FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
		TArray<FAssetData> Assets;
		AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
		for (const FAssetData& Asset : Assets)
		{
			if (Asset.AssetName.ToString().Equals(ClassName, ESearchCase::IgnoreCase))
			{
				UBlueprint* BP = Cast<UBlueprint>(Asset.GetAsset());
				if (BP && BP->GeneratedClass)
				{
					LookupCache.Add(ClassName, BP->GeneratedClass);
					return BP->GeneratedClass;
				}
			}
		}
	}

	return nullptr;
}

static FString PinTypeToString(const FEdGraphPinType& PinType)
{
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Exec) return TEXT("exec");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Boolean) return TEXT("bool");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Int) return TEXT("int");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Int64) return TEXT("int64");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Float) return TEXT("float");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Real) return TEXT("double");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Name) return TEXT("name");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_String) return TEXT("string");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Text) return TEXT("text");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Byte) return TEXT("byte");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Struct)
	{
		if (PinType.PinSubCategoryObject.IsValid())
		{
			return FString::Printf(TEXT("struct:%s"), *PinType.PinSubCategoryObject->GetName());
		}
		return TEXT("struct");
	}
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Object ||
		PinType.PinCategory == UEdGraphSchema_K2::PC_Interface)
	{
		if (PinType.PinSubCategoryObject.IsValid())
		{
			return FString::Printf(TEXT("object:%s"), *PinType.PinSubCategoryObject->GetName());
		}
		return TEXT("object");
	}
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Class) return TEXT("class");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_SoftObject) return TEXT("soft_object");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Enum)
	{
		if (PinType.PinSubCategoryObject.IsValid())
		{
			return FString::Printf(TEXT("enum:%s"), *PinType.PinSubCategoryObject->GetName());
		}
		return TEXT("enum");
	}
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Wildcard) return TEXT("wildcard");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_Delegate) return TEXT("delegate");
	if (PinType.PinCategory == UEdGraphSchema_K2::PC_MCDelegate) return TEXT("multicast_delegate");
	return PinType.PinCategory.ToString();
}

// ─── Schema ──────────────────────────────────────────────────────────────────

TSharedPtr<FJsonObject> FAddNodeAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeInt = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("integer"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or full content path (e.g. 'BP_HealthSystem' or '/Game/Blueprints/BP_HealthSystem')")));
	Props->SetObjectField(TEXT("graph_name"), MakeStr(TEXT("Graph name (default: 'EventGraph')")));
	Props->SetObjectField(TEXT("node_type"), MakeStr(TEXT("Node type: CallFunction, Branch, Sequence, ForLoop, ForEachLoop, WhileLoop, DoOnce, FlipFlop, Gate, CustomEvent, Event, VariableGet, VariableSet, SpawnActor, Delay, PrintString, SetTimer, GetPlayerController, MakeArray, Cast, Self")));
	Props->SetObjectField(TEXT("function_class"), MakeStr(TEXT("For CallFunction: class containing the function (e.g. 'KismetSystemLibrary', 'GameplayStatics', 'Actor')")));
	Props->SetObjectField(TEXT("function_name"), MakeStr(TEXT("For CallFunction: function name (e.g. 'PrintString', 'Delay')")));
	Props->SetObjectField(TEXT("event_name"), MakeStr(TEXT("For Event node_type: event name (e.g. 'ReceiveBeginPlay', 'ReceiveTick', 'ReceiveActorBeginOverlap')")));
	Props->SetObjectField(TEXT("custom_event_name"), MakeStr(TEXT("For CustomEvent: the name of the custom event")));
	Props->SetObjectField(TEXT("variable_name"), MakeStr(TEXT("For VariableGet/VariableSet: name of the Blueprint variable")));
	Props->SetObjectField(TEXT("cast_to_class"), MakeStr(TEXT("For Cast: class to cast to")));
	Props->SetObjectField(TEXT("position_x"), MakeInt(TEXT("X position in the graph (default: 0)")));
	Props->SetObjectField(TEXT("position_y"), MakeInt(TEXT("Y position in the graph (default: 0)")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("node_type")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

// ─── Execute ─────────────────────────────────────────────────────────────────

FBlueprintActionResult FAddNodeAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString NodeType = Params->GetStringField(TEXT("node_type"));
	const FString GraphName = Params->HasField(TEXT("graph_name")) ? Params->GetStringField(TEXT("graph_name")) : TEXT("EventGraph");
	const int32 PosX = Params->HasField(TEXT("position_x")) ? (int32)Params->GetNumberField(TEXT("position_x")) : 0;
	const int32 PosY = Params->HasField(TEXT("position_y")) ? (int32)Params->GetNumberField(TEXT("position_y")) : 0;

	UBlueprint* BP = FindBlueprintByName(BPName);
	if (!BP) return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));

	UEdGraph* Graph = FindGraph(BP, GraphName);
	if (!Graph) return FBlueprintActionResult::Failure(FString::Printf(TEXT("Graph '%s' not found in '%s'"), *GraphName, *BPName));

	UEdGraphNode* NewNode = nullptr;

	// ── Shortcut types (these are really CallFunction nodes) ────────────
	if (NodeType.Equals(TEXT("PrintString"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("Delay"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("SetTimer"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("SetTimerByEvent"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("GetPlayerController"), ESearchCase::IgnoreCase))
	{
		UClass* FuncClass = nullptr;
		FName FuncName;

		if (NodeType.Equals(TEXT("PrintString"), ESearchCase::IgnoreCase))
		{
			FuncClass = UKismetSystemLibrary::StaticClass();
			FuncName = TEXT("PrintString");
		}
		else if (NodeType.Equals(TEXT("Delay"), ESearchCase::IgnoreCase))
		{
			FuncClass = UKismetSystemLibrary::StaticClass();
			FuncName = TEXT("Delay");
		}
		else if (NodeType.Equals(TEXT("SetTimer"), ESearchCase::IgnoreCase))
		{
			FuncClass = UKismetSystemLibrary::StaticClass();
			FuncName = TEXT("K2_SetTimer");
		}
		else if (NodeType.Equals(TEXT("SetTimerByEvent"), ESearchCase::IgnoreCase))
		{
			FuncClass = UKismetSystemLibrary::StaticClass();
			FuncName = TEXT("K2_SetTimerDelegate");
		}
		else if (NodeType.Equals(TEXT("GetPlayerController"), ESearchCase::IgnoreCase))
		{
			FuncClass = UGameplayStatics::StaticClass();
			FuncName = TEXT("GetPlayerController");
		}

		UK2Node_CallFunction* FuncNode = NewObject<UK2Node_CallFunction>(Graph);
		FuncNode->FunctionReference.SetExternalMember(FuncName, FuncClass);
		FuncNode->AllocateDefaultPins();
		NewNode = FuncNode;
	}
	// ── CallFunction ──────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("CallFunction"), ESearchCase::IgnoreCase))
	{
		const FString FuncClassName = Params->GetStringField(TEXT("function_class"));
		FString FuncName = Params->GetStringField(TEXT("function_name"));

		if (FuncClassName.IsEmpty() || FuncName.IsEmpty())
		{
			return FBlueprintActionResult::Failure(TEXT("CallFunction requires function_class and function_name"));
		}

		// Auto-resolve common function name aliases (AI-friendly → UE5 internal)
		static TMap<FString, FString> FuncNameAliases = {
			{TEXT("GetActorLocation"), TEXT("K2_GetActorLocation")},
			{TEXT("SetActorLocation"), TEXT("K2_SetActorLocation")},
			{TEXT("GetActorRotation"), TEXT("K2_GetActorRotation")},
			{TEXT("SetActorRotation"), TEXT("K2_SetActorRotation")},
			{TEXT("SetActorTransform"), TEXT("K2_SetActorTransform")},
			{TEXT("GetActorTransform"), TEXT("GetTransform")},
			{TEXT("K2_GetActorTransform"), TEXT("GetTransform")},
			{TEXT("DestroyActor"), TEXT("K2_DestroyActor")},
			{TEXT("SetLifeSpan"), TEXT("SetLifeSpan")},
			{TEXT("SetTimerByFunctionName"), TEXT("K2_SetTimer")},
			{TEXT("GetWorldLocation"), TEXT("K2_GetComponentLocation")},
			{TEXT("GetWorldRotation"), TEXT("K2_GetComponentRotation")},
			{TEXT("SetWorldLocation"), TEXT("K2_SetWorldLocation")},
			{TEXT("SetWorldRotation"), TEXT("K2_SetWorldRotation")},
			{TEXT("AttachToComponent"), TEXT("K2_AttachToComponent")},
			{TEXT("DetachFromComponent"), TEXT("K2_DetachFromComponent")},
			{TEXT("GetComponentLocation"), TEXT("K2_GetComponentLocation")},
			{TEXT("GetComponentRotation"), TEXT("K2_GetComponentRotation")},
		};
		if (const FString* AliasedName = FuncNameAliases.Find(FuncName))
		{
			FuncName = *AliasedName;
		}

		UClass* FuncClass = ResolveClass(FuncClassName);
		if (!FuncClass)
		{
			return FBlueprintActionResult::Failure(FString::Printf(TEXT("Could not resolve class '%s'"), *FuncClassName));
		}

		UK2Node_CallFunction* FuncNode = NewObject<UK2Node_CallFunction>(Graph);
		FuncNode->FunctionReference.SetExternalMember(FName(*FuncName), FuncClass);
		FuncNode->AllocateDefaultPins();

		if (FuncNode->Pins.Num() == 0)
		{
			FuncNode->MarkAsGarbage();
			return FBlueprintActionResult::Failure(
				FString::Printf(TEXT("Function '%s::%s' not found or has no pins. Use get_class_functions to discover available functions."), *FuncClassName, *FuncName));
		}

		NewNode = FuncNode;
	}
	// ── Branch ────────────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("Branch"), ESearchCase::IgnoreCase))
	{
		UK2Node_IfThenElse* Node = NewObject<UK2Node_IfThenElse>(Graph);
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	// ── Sequence ──────────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("Sequence"), ESearchCase::IgnoreCase))
	{
		UK2Node_ExecutionSequence* Node = NewObject<UK2Node_ExecutionSequence>(Graph);
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	// ── ForLoop / ForEachLoop / WhileLoop (macro instances) ──────────────
	else if (NodeType.Equals(TEXT("ForLoop"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("ForEachLoop"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("WhileLoop"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("DoOnce"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("FlipFlop"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("Gate"), ESearchCase::IgnoreCase) ||
		NodeType.Equals(TEXT("DoN"), ESearchCase::IgnoreCase))
	{
		// Load the StandardMacros Blueprint from Engine content
		UBlueprint* MacroBP = LoadObject<UBlueprint>(nullptr,
			TEXT("/Engine/EditorBlueprintResources/StandardMacros.StandardMacros"));
		if (!MacroBP)
		{
			return FBlueprintActionResult::Failure(TEXT("Could not load StandardMacros Blueprint from Engine content"));
		}

		// Map node_type name to StandardMacros graph name
		// Most names match directly, but "DoN" maps to "Do N" in the Blueprint
		FString MacroName = NodeType;
		if (NodeType.Equals(TEXT("DoN"), ESearchCase::IgnoreCase))
		{
			MacroName = TEXT("Do N");
		}

		// Find the macro graph by name
		UEdGraph* MacroGraph = nullptr;
		for (UEdGraph* MacroCandidate : MacroBP->MacroGraphs)
		{
			if (MacroCandidate && MacroCandidate->GetName().Equals(MacroName, ESearchCase::IgnoreCase))
			{
				MacroGraph = MacroCandidate;
				break;
			}
		}
		if (!MacroGraph)
		{
			return FBlueprintActionResult::Failure(
				FString::Printf(TEXT("Macro '%s' not found in StandardMacros Blueprint"), *NodeType));
		}

		UK2Node_MacroInstance* Node = NewObject<UK2Node_MacroInstance>(Graph);
		Node->SetMacroGraph(MacroGraph);
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	// ── CustomEvent ───────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("CustomEvent"), ESearchCase::IgnoreCase))
	{
		const FString EventName = Params->GetStringField(TEXT("custom_event_name"));
		if (EventName.IsEmpty())
		{
			return FBlueprintActionResult::Failure(TEXT("CustomEvent requires custom_event_name"));
		}

		UK2Node_CustomEvent* Node = NewObject<UK2Node_CustomEvent>(Graph);
		Node->CustomFunctionName = FName(*EventName);
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	// ── Event (BeginPlay, Tick, etc.) ─────────────────────────────────────
	else if (NodeType.Equals(TEXT("Event"), ESearchCase::IgnoreCase))
	{
		const FString EventName = Params->GetStringField(TEXT("event_name"));
		if (EventName.IsEmpty())
		{
			return FBlueprintActionResult::Failure(TEXT("Event requires event_name (e.g. 'ReceiveBeginPlay', 'ReceiveTick')"));
		}

		// Check if event already exists in graph
		for (UEdGraphNode* ExistingNode : Graph->Nodes)
		{
			UK2Node_Event* ExistingEvent = Cast<UK2Node_Event>(ExistingNode);
			if (ExistingEvent && ExistingEvent->EventReference.GetMemberName().ToString().Equals(EventName, ESearchCase::IgnoreCase))
			{
				// Event already exists - track it and return its info
				FString NodeId = Registry->TrackNode(ExistingEvent);
				TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
				Data->SetStringField(TEXT("node_id"), NodeId);
				Data->SetStringField(TEXT("info"), TEXT("Event already exists, returning existing node"));
				Data->SetArrayField(TEXT("pins"), SerializePins(ExistingEvent));
				return FBlueprintActionResult::Success(
					FString::Printf(TEXT("Event '%s' already exists (node %s)"), *EventName, *NodeId), Data);
			}
		}

		UK2Node_Event* Node = NewObject<UK2Node_Event>(Graph);

		// Use the parent class (not SkeletonGeneratedClass) so inherited events
		// like ReceiveBeginPlay and ReceiveActorBeginOverlap resolve correctly.
		// SkeletonGeneratedClass may not expose inherited functions if it hasn't
		// been fully compiled/regenerated, causing AllocateDefaultPins to produce
		// zero pins.
		UClass* EventClass = BP->ParentClass ? (UClass*)BP->ParentClass : AActor::StaticClass();
		Node->EventReference.SetExternalMember(FName(*EventName), EventClass);
		Node->bOverrideFunction = true;

		// Add node to graph BEFORE allocating pins so the node has full context
		Node->NodePosX = PosX;
		Node->NodePosY = PosY;
		Node->CreateNewGuid();
		Graph->AddNode(Node, /*bFromUI=*/false, /*bSelectNewNode=*/false);
		Node->PostPlacedNewNode();
		Node->AllocateDefaultPins();

		// If pins are still empty (shouldn't happen now), try ReconstructNode
		if (Node->Pins.Num() == 0)
		{
			Node->ReconstructNode();
		}

		Graph->NotifyGraphChanged();
		FBlueprintEditorUtils::MarkBlueprintAsModified(BP);

		// Track and build result (skip the generic placement code below)
		FString NodeId = Registry->TrackNode(Node);

		TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
		Data->SetStringField(TEXT("node_id"), NodeId);
		Data->SetStringField(TEXT("node_title"), Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString());
		Data->SetBoolField(TEXT("is_pure"), false);
		Data->SetArrayField(TEXT("pins"), SerializePins(Node));

		return FBlueprintActionResult::Success(
			FString::Printf(TEXT("Added node '%s' (id: %s)"), *Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString(), *NodeId),
			Data);
	}
	// ── VariableGet ───────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("VariableGet"), ESearchCase::IgnoreCase))
	{
		const FString VarName = Params->GetStringField(TEXT("variable_name"));
		if (VarName.IsEmpty())
		{
			return FBlueprintActionResult::Failure(TEXT("VariableGet requires variable_name"));
		}

		UK2Node_VariableGet* Node = NewObject<UK2Node_VariableGet>(Graph);
		Node->VariableReference.SetSelfMember(FName(*VarName));
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	// ── VariableSet ───────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("VariableSet"), ESearchCase::IgnoreCase))
	{
		const FString VarName = Params->GetStringField(TEXT("variable_name"));
		if (VarName.IsEmpty())
		{
			return FBlueprintActionResult::Failure(TEXT("VariableSet requires variable_name"));
		}

		UK2Node_VariableSet* Node = NewObject<UK2Node_VariableSet>(Graph);
		Node->VariableReference.SetSelfMember(FName(*VarName));
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	// ── SpawnActor ────────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("SpawnActor"), ESearchCase::IgnoreCase))
	{
		UK2Node_SpawnActorFromClass* Node = NewObject<UK2Node_SpawnActorFromClass>(Graph);
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	// ── Cast ──────────────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("Cast"), ESearchCase::IgnoreCase))
	{
		const FString CastClass = Params->GetStringField(TEXT("cast_to_class"));
		if (CastClass.IsEmpty())
		{
			return FBlueprintActionResult::Failure(TEXT("Cast requires cast_to_class"));
		}

		UClass* TargetClass = ResolveClass(CastClass);
		if (!TargetClass)
		{
			return FBlueprintActionResult::Failure(FString::Printf(TEXT("Could not resolve cast class '%s'"), *CastClass));
		}

		UK2Node_DynamicCast* Node = NewObject<UK2Node_DynamicCast>(Graph);
		Node->TargetType = TargetClass;
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	// ── MakeArray ─────────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("MakeArray"), ESearchCase::IgnoreCase))
	{
		UK2Node_MakeArray* Node = NewObject<UK2Node_MakeArray>(Graph);
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	// ── Self ──────────────────────────────────────────────────────────────
	else if (NodeType.Equals(TEXT("Self"), ESearchCase::IgnoreCase))
	{
		UK2Node_Self* Node = NewObject<UK2Node_Self>(Graph);
		Node->AllocateDefaultPins();
		NewNode = Node;
	}
	else
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Unknown node_type '%s'. Valid types: CallFunction, Branch, Sequence, ForLoop, ForEachLoop, WhileLoop, DoOnce, FlipFlop, Gate, DoN, Event, CustomEvent, VariableGet, VariableSet, SpawnActor, Cast, MakeArray, Self, Delay, PrintString, SetTimer, SetTimerByEvent, GetPlayerController"), *NodeType));
	}

	if (!NewNode)
	{
		return FBlueprintActionResult::Failure(TEXT("Failed to create node"));
	}

	// Place the node in the graph
	NewNode->NodePosX = PosX;
	NewNode->NodePosY = PosY;
	NewNode->CreateNewGuid();
	NewNode->PostPlacedNewNode();

	Graph->AddNode(NewNode, /*bFromUI=*/false, /*bSelectNewNode=*/false);
	Graph->NotifyGraphChanged();

	FBlueprintEditorUtils::MarkBlueprintAsModified(BP);

	// Track and build result
	FString NodeId = Registry->TrackNode(NewNode);

	// Determine if this is a pure node (no exec pins = pure, means no execution flow)
	bool bIsPure = true;
	for (UEdGraphPin* Pin : NewNode->Pins)
	{
		if (Pin->PinType.PinCategory == UEdGraphSchema_K2::PC_Exec)
		{
			bIsPure = false;
			break;
		}
	}

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("node_id"), NodeId);
	Data->SetStringField(TEXT("node_title"), NewNode->GetNodeTitle(ENodeTitleType::FullTitle).ToString());
	Data->SetBoolField(TEXT("is_pure"), bIsPure);
	if (bIsPure)
	{
		Data->SetStringField(TEXT("note"), TEXT("Pure node — NO exec pins. Connect data pins only, not execution flow."));
	}
	Data->SetArrayField(TEXT("pins"), SerializePins(NewNode));

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Added node '%s' (id: %s)"), *NewNode->GetNodeTitle(ENodeTitleType::FullTitle).ToString(), *NodeId),
		Data);
}

// ─── Pin serialization ───────────────────────────────────────────────────────

TArray<TSharedPtr<FJsonValue>> FAddNodeAction::SerializePins(UEdGraphNode* Node) const
{
	TArray<TSharedPtr<FJsonValue>> PinArray;
	if (!Node) return PinArray;

	for (UEdGraphPin* Pin : Node->Pins)
	{
		// Skip hidden pins (WorldContextObject, self for static functions, etc.)
		if (Pin->bHidden) continue;

		TSharedPtr<FJsonObject> PinObj = MakeShared<FJsonObject>();
		PinObj->SetStringField(TEXT("name"), Pin->PinName.ToString());
		PinObj->SetStringField(TEXT("direction"), Pin->Direction == EGPD_Input ? TEXT("Input") : TEXT("Output"));
		PinObj->SetStringField(TEXT("type"), PinTypeToString(Pin->PinType));

		if (!Pin->DefaultValue.IsEmpty())
		{
			PinObj->SetStringField(TEXT("default"), Pin->DefaultValue);
		}

		PinArray.Add(MakeShared<FJsonValueObject>(PinObj));
	}

	return PinArray;
}
