// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/AddFunctionAction.h"
#include "Actions/ActionRegistry.h"

#include "Kismet2/BlueprintEditorUtils.h"
#include "Engine/Blueprint.h"
#include "EdGraph/EdGraph.h"
#include "EdGraphSchema_K2.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "Engine/BlueprintGeneratedClass.h"

// ── Lightweight pin serializer (mirrors AddNodeAction::SerializePins) ─────────
static FString PinTypeToStr(const FEdGraphPinType& PinType)
{
	FString Base = PinType.PinCategory.ToString();
	if (PinType.PinSubCategoryObject.IsValid())
	{
		Base += TEXT(":") + PinType.PinSubCategoryObject->GetName();
	}
	if (PinType.ContainerType == EPinContainerType::Array) Base = TEXT("Array<") + Base + TEXT(">");
	if (PinType.ContainerType == EPinContainerType::Set) Base = TEXT("Set<") + Base + TEXT(">");
	if (PinType.ContainerType == EPinContainerType::Map) Base = TEXT("Map<") + Base + TEXT(">");
	return Base;
}

static TArray<TSharedPtr<FJsonValue>> SerializeNodePins(UEdGraphNode* Node)
{
	TArray<TSharedPtr<FJsonValue>> PinArray;
	if (!Node) return PinArray;

	for (UEdGraphPin* Pin : Node->Pins)
	{
		if (Pin->bHidden) continue;

		TSharedPtr<FJsonObject> PinObj = MakeShared<FJsonObject>();
		PinObj->SetStringField(TEXT("name"), Pin->PinName.ToString());
		PinObj->SetStringField(TEXT("direction"), Pin->Direction == EGPD_Input ? TEXT("Input") : TEXT("Output"));
		PinObj->SetStringField(TEXT("type"), PinTypeToStr(Pin->PinType));

		if (!Pin->DefaultValue.IsEmpty())
		{
			PinObj->SetStringField(TEXT("default"), Pin->DefaultValue);
		}

		PinArray.Add(MakeShared<FJsonValueObject>(PinObj));
	}

	return PinArray;
}

static UBlueprint* FindBlueprintByNameForFunc(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

TSharedPtr<FJsonObject> FAddFunctionAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeBool = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("boolean"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("function_name"), MakeStr(TEXT("Name of the function to create")));
	Props->SetObjectField(TEXT("is_pure"), MakeBool(TEXT("If true, function is pure (no exec pins, no side effects). Default: false.")));
	Props->SetObjectField(TEXT("access_specifier"), MakeStr(TEXT("Access level: 'Public', 'Protected', or 'Private'. Default: 'Public'.")));
	Props->SetObjectField(TEXT("category"), MakeStr(TEXT("Category for organizing in the function list")));
	Props->SetObjectField(TEXT("description"), MakeStr(TEXT("Description/tooltip for the function")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("function_name")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FAddFunctionAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString FuncName = Params->GetStringField(TEXT("function_name"));
	const bool bIsPure = Params->HasField(TEXT("is_pure")) && Params->GetBoolField(TEXT("is_pure"));

	UBlueprint* BP = FindBlueprintByNameForFunc(BPName);
	if (!BP) return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));

	// Check if function already exists
	for (UEdGraph* FGraph : BP->FunctionGraphs)
	{
		if (FGraph && FGraph->GetName().Equals(FuncName, ESearchCase::IgnoreCase))
		{
			return FBlueprintActionResult::Failure(
				FString::Printf(TEXT("Function '%s' already exists in '%s'"), *FuncName, *BPName));
		}
	}

	// Create function graph
	UEdGraph* NewGraph = FBlueprintEditorUtils::CreateNewGraph(
		BP,
		FName(*FuncName),
		UEdGraph::StaticClass(),
		UEdGraphSchema_K2::StaticClass()
	);

	if (!NewGraph)
	{
		return FBlueprintActionResult::Failure(TEXT("Failed to create function graph"));
	}

	FBlueprintEditorUtils::AddFunctionGraph<UClass>(BP, NewGraph, /*bIsUserCreated=*/true, nullptr);

	// Set pure flag
	if (bIsPure)
	{
		// Mark function as pure via function entry node
		// Function flags are set on the UFunction after compile, but we can mark via metadata
		NewGraph->GetSchema()->CreateDefaultNodesForGraph(*NewGraph);
	}

	FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("function_name"), FuncName);
	Data->SetStringField(TEXT("graph_name"), NewGraph->GetName());
	Data->SetStringField(TEXT("blueprint"), BP->GetPathName());

	// Track and serialize all default nodes (entry node, return node, etc.)
	TArray<TSharedPtr<FJsonValue>> DefaultNodes;
	FString EntryNodeId;
	for (UEdGraphNode* Node : NewGraph->Nodes)
	{
		FString NodeId = Registry->TrackNode(Node);
		TSharedPtr<FJsonObject> NodeObj = MakeShared<FJsonObject>();
		NodeObj->SetStringField(TEXT("node_id"), NodeId);
		NodeObj->SetStringField(TEXT("title"), Node->GetNodeTitle(ENodeTitleType::FullTitle).ToString());
		NodeObj->SetArrayField(TEXT("pins"), SerializeNodePins(Node));
		DefaultNodes.Add(MakeShared<FJsonValueObject>(NodeObj));

		// Remember the entry node ID for the success message
		if (EntryNodeId.IsEmpty())
		{
			EntryNodeId = NodeId;
		}
	}
	Data->SetArrayField(TEXT("default_nodes"), DefaultNodes);

	FString Message = FString::Printf(
		TEXT("Added function '%s' to '%s'. Entry node: %s. Use graph_name='%s' in add_node to add nodes, then connect_nodes to wire entry's 'then' output to your first node."),
		*FuncName, *BPName, *EntryNodeId, *NewGraph->GetName());

	return FBlueprintActionResult::Success(Message, Data);
}
