// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/AddComponentAction.h"

#include "Engine/Blueprint.h"
#include "Engine/SCS_Node.h"
#include "Engine/SimpleConstructionScript.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "Components/StaticMeshComponent.h"
#include "Components/SphereComponent.h"
#include "Components/BoxComponent.h"
#include "Components/CapsuleComponent.h"
#include "Components/ArrowComponent.h"
#include "Components/AudioComponent.h"
#include "Components/PointLightComponent.h"
#include "Components/SpotLightComponent.h"
#include "Camera/CameraComponent.h"
#include "GameFramework/SpringArmComponent.h"
#include "GameFramework/ProjectileMovementComponent.h"
#include "GameFramework/RotatingMovementComponent.h"
#include "Components/WidgetComponent.h"
#include "Components/SceneComponent.h"
#include "Components/ChildActorComponent.h"
#include "Components/DecalComponent.h"
#include "Components/BillboardComponent.h"

// ── Helpers ──────────────────────────────────────────────────────────────────

static UBlueprint* FindBPForComponent(const FString& NameOrPath)
{
	if (NameOrPath.StartsWith(TEXT("/")))
	{
		return LoadObject<UBlueprint>(nullptr, *NameOrPath);
	}
	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	TArray<FAssetData> Assets;
	AssetReg.Get().GetAssetsByClass(UBlueprint::StaticClass()->GetClassPathName(), Assets);
	for (const FAssetData& Asset : Assets)
	{
		if (Asset.AssetName.ToString().Equals(NameOrPath, ESearchCase::IgnoreCase))
		{
			return Cast<UBlueprint>(Asset.GetAsset());
		}
	}
	return nullptr;
}

static UClass* FindComponentClass(const FString& ClassName)
{
	// Try exact match first, then with/without "Component" suffix
	TArray<FString> Candidates;
	Candidates.Add(ClassName);
	if (!ClassName.EndsWith(TEXT("Component")))
	{
		Candidates.Add(ClassName + TEXT("Component"));
	}
	// Also try with U prefix
	Candidates.Add(TEXT("U") + ClassName);
	if (!ClassName.EndsWith(TEXT("Component")))
	{
		Candidates.Add(TEXT("U") + ClassName + TEXT("Component"));
	}

	for (TObjectIterator<UClass> It; It; ++It)
	{
		UClass* Class = *It;
		if (!Class->IsChildOf(UActorComponent::StaticClass())) continue;

		FString Name = Class->GetName();
		for (const FString& Candidate : Candidates)
		{
			if (Name.Equals(Candidate, ESearchCase::IgnoreCase))
			{
				return Class;
			}
		}
	}

	return nullptr;
}

// ── Schema ───────────────────────────────────────────────────────────────────

TSharedPtr<FJsonObject> FAddComponentAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};
	auto MakeBool = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("boolean"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Blueprint name or content path")));
	Props->SetObjectField(TEXT("component_class"), MakeStr(TEXT("Component class name (e.g. 'StaticMeshComponent', 'SphereComponent', 'ProjectileMovementComponent')")));
	Props->SetObjectField(TEXT("component_name"), MakeStr(TEXT("Display name for the component (e.g. 'ProjectileMesh', 'CollisionSphere')")));
	Props->SetObjectField(TEXT("parent_component"), MakeStr(TEXT("Name of the parent component to attach to. If omitted, attaches to the root.")));
	Props->SetObjectField(TEXT("is_root"), MakeBool(TEXT("If true, makes this the root component of the Blueprint")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("component_class")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

// ── Execute ──────────────────────────────────────────────────────────────────

FBlueprintActionResult FAddComponentAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	const FString BPName = Params->GetStringField(TEXT("blueprint"));
	const FString CompClassName = Params->GetStringField(TEXT("component_class"));
	const FString CompName = Params->HasField(TEXT("component_name"))
		? Params->GetStringField(TEXT("component_name"))
		: CompClassName.Replace(TEXT("Component"), TEXT(""));
	const bool bIsRoot = Params->HasField(TEXT("is_root")) && Params->GetBoolField(TEXT("is_root"));
	const FString ParentName = Params->HasField(TEXT("parent_component"))
		? Params->GetStringField(TEXT("parent_component"))
		: TEXT("");

	// Find blueprint
	UBlueprint* BP = FindBPForComponent(BPName);
	if (!BP)
	{
		return FBlueprintActionResult::Failure(FString::Printf(TEXT("Blueprint '%s' not found"), *BPName));
	}

	// Find component class
	UClass* CompClass = FindComponentClass(CompClassName);
	if (!CompClass)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Component class '%s' not found. Try: StaticMeshComponent, SphereComponent, BoxComponent, "
				"ProjectileMovementComponent, CameraComponent, SpringArmComponent, AudioComponent, "
				"PointLightComponent, WidgetComponent, ArrowComponent"), *CompClassName));
	}

	// Ensure blueprint has an SCS
	USimpleConstructionScript* SCS = BP->SimpleConstructionScript;
	if (!SCS)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Blueprint '%s' does not support components (no SimpleConstructionScript)"), *BPName));
	}

	// Check for name collision — return success (idempotent) if component already exists
	FName VariableName(*CompName);
	for (USCS_Node* Existing : SCS->GetAllNodes())
	{
		if (Existing->GetVariableName() == VariableName)
		{
			TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
			Data->SetStringField(TEXT("component_name"), CompName);
			Data->SetStringField(TEXT("status"), TEXT("already_exists"));
			if (Existing->ComponentTemplate)
			{
				Data->SetStringField(TEXT("component_class"), Existing->ComponentTemplate->GetClass()->GetName());
			}
			return FBlueprintActionResult::Success(
				FString::Printf(TEXT("Component '%s' already exists in '%s' (no changes made)"), *CompName, *BPName),
				Data);
		}
	}

	// Create the SCS node
	USCS_Node* NewNode = SCS->CreateNode(CompClass, FName(*CompName));
	if (!NewNode)
	{
		return FBlueprintActionResult::Failure(
			FString::Printf(TEXT("Failed to create component node for class '%s'"), *CompClassName));
	}

	if (bIsRoot)
	{
		// Replace default scene root
		USCS_Node* OldRoot = SCS->GetDefaultSceneRootNode();
		if (OldRoot && CompClass->IsChildOf(USceneComponent::StaticClass()))
		{
			// Move children from old root to new root
			const TArray<USCS_Node*> OldChildren = OldRoot->GetChildNodes();
			for (USCS_Node* Child : OldChildren)
			{
				OldRoot->RemoveChildNode(Child);
				NewNode->AddChildNode(Child);
			}
			SCS->RemoveNode(OldRoot);
		}
		SCS->AddNode(NewNode);
	}
	else if (!ParentName.IsEmpty())
	{
		// Find parent and attach
		USCS_Node* ParentNode = nullptr;
		for (USCS_Node* Existing : SCS->GetAllNodes())
		{
			if (Existing->GetVariableName().ToString().Equals(ParentName, ESearchCase::IgnoreCase) ||
				(Existing->ComponentTemplate && Existing->ComponentTemplate->GetName().Equals(ParentName, ESearchCase::IgnoreCase)))
			{
				ParentNode = Existing;
				break;
			}
		}
		if (ParentNode)
		{
			ParentNode->AddChildNode(NewNode);
		}
		else
		{
			// Parent not found, attach to root
			SCS->AddNode(NewNode);
		}
	}
	else
	{
		// Attach to default root
		USCS_Node* RootNode = SCS->GetDefaultSceneRootNode();
		if (RootNode && CompClass->IsChildOf(USceneComponent::StaticClass()))
		{
			RootNode->AddChildNode(NewNode);
		}
		else
		{
			SCS->AddNode(NewNode);
		}
	}

	FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(BP);

	// Build result
	TSharedPtr<FJsonObject> Data = MakeShared<FJsonObject>();
	Data->SetStringField(TEXT("component_name"), NewNode->GetVariableName().ToString());
	Data->SetStringField(TEXT("component_class"), CompClass->GetName());
	Data->SetStringField(TEXT("blueprint"), BPName);
	Data->SetBoolField(TEXT("is_root"), bIsRoot);

	return FBlueprintActionResult::Success(
		FString::Printf(TEXT("Added component '%s' (%s) to '%s'"),
			*NewNode->GetVariableName().ToString(), *CompClass->GetName(), *BPName),
		Data);
}
