// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/AddCommentBoxAction.h"
#include "Actions/ActionRegistry.h"
#include "EdGraphNode_Comment.h"
#include "Kismet2/BlueprintEditorUtils.h"
#include "Engine/Blueprint.h"
#include "Dom/JsonObject.h"
#include "Dom/JsonValue.h"

TSharedPtr<FJsonObject> FAddCommentBoxAction::GetParameterSchema() const
{
	auto MakeStr = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("string"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	auto MakeNum = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("number"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	auto MakeInt = [](const FString& Desc) -> TSharedPtr<FJsonObject> {
		TSharedPtr<FJsonObject> P = MakeShared<FJsonObject>();
		P->SetStringField(TEXT("type"), TEXT("integer"));
		P->SetStringField(TEXT("description"), Desc);
		return P;
	};

	TSharedPtr<FJsonObject> Schema = MakeShared<FJsonObject>();
	Schema->SetStringField(TEXT("type"), TEXT("object"));

	TSharedPtr<FJsonObject> Props = MakeShared<FJsonObject>();
	Props->SetObjectField(TEXT("blueprint"), MakeStr(TEXT("Name of the target Blueprint (e.g. 'BP_ThirdPersonCharacter')")));
	Props->SetObjectField(TEXT("comment"), MakeStr(TEXT("Text label displayed on the comment box")));
	Props->SetObjectField(TEXT("graph"), MakeStr(TEXT("Optional. Name of the graph (default: 'EventGraph')")));
	Props->SetObjectField(TEXT("color"), MakeStr(TEXT("Optional. Color as 'R,G,B,A' floats 0-1 range. Default: white (1,1,1,1)")));

	// node_ids array
	TSharedPtr<FJsonObject> NodeIdsSchema = MakeShared<FJsonObject>();
	NodeIdsSchema->SetStringField(TEXT("type"), TEXT("array"));
	NodeIdsSchema->SetStringField(TEXT("description"), TEXT("Optional. Array of tracked node IDs to auto-wrap. The box will be sized to contain all listed nodes with padding."));
	TSharedPtr<FJsonObject> ItemsSchema = MakeShared<FJsonObject>();
	ItemsSchema->SetStringField(TEXT("type"), TEXT("string"));
	NodeIdsSchema->SetObjectField(TEXT("items"), ItemsSchema);
	Props->SetObjectField(TEXT("node_ids"), NodeIdsSchema);

	Props->SetObjectField(TEXT("position_x"), MakeNum(TEXT("X position (only used if node_ids not provided)")));
	Props->SetObjectField(TEXT("position_y"), MakeNum(TEXT("Y position (only used if node_ids not provided)")));
	Props->SetObjectField(TEXT("width"), MakeNum(TEXT("Box width (only used if node_ids not provided, default: 400)")));
	Props->SetObjectField(TEXT("height"), MakeNum(TEXT("Box height (only used if node_ids not provided, default: 300)")));
	Props->SetObjectField(TEXT("font_size"), MakeInt(TEXT("Optional. Font size for the comment text (default: 18)")));

	Schema->SetObjectField(TEXT("properties"), Props);

	TArray<TSharedPtr<FJsonValue>> Required;
	Required.Add(MakeShared<FJsonValueString>(TEXT("blueprint")));
	Required.Add(MakeShared<FJsonValueString>(TEXT("comment")));
	Schema->SetArrayField(TEXT("required"), Required);

	return Schema;
}

FBlueprintActionResult FAddCommentBoxAction::Execute(const TSharedPtr<FJsonObject>& Params)
{
	FBlueprintActionResult Result;

	// --- Required params ---
	FString BlueprintName;
	if (!Params->TryGetStringField(TEXT("blueprint"), BlueprintName) || BlueprintName.IsEmpty())
	{
		Result.bSuccess = false;
		Result.Message = TEXT("Missing required parameter: 'blueprint'");
		return Result;
	}

	FString CommentText;
	if (!Params->TryGetStringField(TEXT("comment"), CommentText) || CommentText.IsEmpty())
	{
		Result.bSuccess = false;
		Result.Message = TEXT("Missing required parameter: 'comment'");
		return Result;
	}

	// --- Optional params ---
	FString GraphName = TEXT("EventGraph");
	Params->TryGetStringField(TEXT("graph"), GraphName);

	int32 FontSize = 18;
	if (Params->HasField(TEXT("font_size")))
	{
		FontSize = static_cast<int32>(Params->GetNumberField(TEXT("font_size")));
	}

	// Color
	FLinearColor BoxColor(1.f, 1.f, 1.f, 1.f);
	FString ColorStr;
	if (Params->TryGetStringField(TEXT("color"), ColorStr) && !ColorStr.IsEmpty())
	{
		TArray<FString> Parts;
		ColorStr.ParseIntoArray(Parts, TEXT(","));
		if (Parts.Num() >= 3)
		{
			BoxColor.R = FCString::Atof(*Parts[0]);
			BoxColor.G = FCString::Atof(*Parts[1]);
			BoxColor.B = FCString::Atof(*Parts[2]);
			BoxColor.A = Parts.Num() >= 4 ? FCString::Atof(*Parts[3]) : 1.f;
		}
	}

	// --- Find Blueprint ---
	UBlueprint* Blueprint = nullptr;
	for (TObjectIterator<UBlueprint> It; It; ++It)
	{
		if (It->GetName() == BlueprintName)
		{
			Blueprint = *It;
			break;
		}
	}

	if (!Blueprint)
	{
		Result.bSuccess = false;
		Result.Message = FString::Printf(TEXT("Blueprint '%s' not found. Make sure it is open in the editor."), *BlueprintName);
		return Result;
	}

	// --- Find Graph ---
	UEdGraph* TargetGraph = nullptr;
	for (UEdGraph* Graph : Blueprint->UbergraphPages)
	{
		if (Graph->GetName() == GraphName)
		{
			TargetGraph = Graph;
			break;
		}
	}
	if (!TargetGraph)
	{
		for (UEdGraph* Graph : Blueprint->FunctionGraphs)
		{
			if (Graph->GetName() == GraphName)
			{
				TargetGraph = Graph;
				break;
			}
		}
	}

	if (!TargetGraph)
	{
		Result.bSuccess = false;
		Result.Message = FString::Printf(TEXT("Graph '%s' not found in Blueprint '%s'."), *GraphName, *BlueprintName);
		return Result;
	}

	// --- Calculate bounds ---
	float PosX = 0, PosY = 0, BoxWidth = 400, BoxHeight = 300;
	constexpr float Padding = 100.f;  // padding around nodes

	const TArray<TSharedPtr<FJsonValue>>* NodeIdsArray = nullptr;
	if (Params->TryGetArrayField(TEXT("node_ids"), NodeIdsArray) && NodeIdsArray && NodeIdsArray->Num() > 0)
	{
		// Auto-calculate bounds from tracked nodes
		float MinX = TNumericLimits<float>::Max();
		float MinY = TNumericLimits<float>::Max();
		float MaxX = TNumericLimits<float>::Lowest();
		float MaxY = TNumericLimits<float>::Lowest();
		int32 FoundCount = 0;

		for (const TSharedPtr<FJsonValue>& IdVal : *NodeIdsArray)
		{
			FString NodeId = IdVal->AsString();
			UEdGraphNode* Node = Registry->FindTrackedNode(NodeId);
			if (!Node)
			{
				Result.bSuccess = false;
				Result.Message = FString::Printf(TEXT("Tracked node '%s' not found. Use get_selected_nodes first."), *NodeId);
				return Result;
			}

			float NX = static_cast<float>(Node->NodePosX);
			float NY = static_cast<float>(Node->NodePosY);
			// Estimate node size: most BP nodes are roughly 300x150
			constexpr float EstNodeWidth = 300.f;
			constexpr float EstNodeHeight = 150.f;

			MinX = FMath::Min(MinX, NX);
			MinY = FMath::Min(MinY, NY);
			MaxX = FMath::Max(MaxX, NX + EstNodeWidth);
			MaxY = FMath::Max(MaxY, NY + EstNodeHeight);
			FoundCount++;
		}

		if (FoundCount == 0)
		{
			Result.bSuccess = false;
			Result.Message = TEXT("No valid tracked nodes found from node_ids.");
			return Result;
		}

		PosX = MinX - Padding;
		PosY = MinY - Padding - 40.f;  // extra top padding for title bar
		BoxWidth = (MaxX - MinX) + (Padding * 2);
		BoxHeight = (MaxY - MinY) + (Padding * 2) + 40.f;
	}
	else
	{
		// Use explicit position/size
		if (Params->HasField(TEXT("position_x")))
			PosX = static_cast<float>(Params->GetNumberField(TEXT("position_x")));
		if (Params->HasField(TEXT("position_y")))
			PosY = static_cast<float>(Params->GetNumberField(TEXT("position_y")));
		if (Params->HasField(TEXT("width")))
			BoxWidth = static_cast<float>(Params->GetNumberField(TEXT("width")));
		if (Params->HasField(TEXT("height")))
			BoxHeight = static_cast<float>(Params->GetNumberField(TEXT("height")));
	}

	// --- Create the comment box node ---
	UEdGraphNode_Comment* CommentNode = NewObject<UEdGraphNode_Comment>(TargetGraph);
	CommentNode->NodeComment = CommentText;
	CommentNode->NodePosX = static_cast<int32>(PosX);
	CommentNode->NodePosY = static_cast<int32>(PosY);
	CommentNode->NodeWidth = static_cast<int32>(BoxWidth);
	CommentNode->NodeHeight = static_cast<int32>(BoxHeight);
	CommentNode->CommentColor = BoxColor;
	CommentNode->FontSize = FontSize;
	CommentNode->MoveMode = ECommentBoxMode::GroupMovement;
	CommentNode->bColorCommentBubble = false;
	CommentNode->CommentDepth = -1;  // behind nodes

	CommentNode->CreateNewGuid();
	CommentNode->PostPlacedNewNode();
	CommentNode->AllocateDefaultPins();

	TargetGraph->AddNode(CommentNode, /*bFromUI=*/false, /*bSelectNewNode=*/false);
	TargetGraph->NotifyGraphChanged();

	FBlueprintEditorUtils::MarkBlueprintAsStructurallyModified(Blueprint);

	// Track the comment node so it can be referenced later
	FString CommentNodeId = Registry->TrackNode(CommentNode);

	// Build result
	Result.bSuccess = true;
	Result.Message = FString::Printf(
		TEXT("Created comment box '%s' in %s.%s at (%d, %d) size %dx%d. node_id: %s"),
		*CommentText,
		*BlueprintName,
		*GraphName,
		CommentNode->NodePosX,
		CommentNode->NodePosY,
		CommentNode->NodeWidth,
		CommentNode->NodeHeight,
		*CommentNodeId
	);

	return Result;
}
