// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "Actions/ActionRegistry.h"
#include "AI/IAIProvider.h"
#include "EdGraph/EdGraphNode.h"
#include "Dom/JsonValue.h"

// Include all action headers
#include "Actions/CreateBlueprintAction.h"
#include "Actions/AddNodeAction.h"
#include "Actions/ConnectNodesAction.h"
#include "Actions/AddVariableAction.h"
#include "Actions/AddFunctionAction.h"
#include "Actions/CompileBlueprintAction.h"
#include "Actions/GetBlueprintInfoAction.h"
#include "Actions/ListAssetsAction.h"
#include "Actions/DeleteNodeAction.h"
#include "Actions/SetPinValueAction.h"
#include "Actions/GetProjectStructureAction.h"
#include "Actions/RenameVariableAction.h"
#include "Actions/SetVariableDefaultsAction.h"
#include "Actions/OpenBlueprintAction.h"
#include "Actions/AddComponentAction.h"
#include "Actions/SetComponentPropertyAction.h"
#include "Actions/RemoveComponentAction.h"
#include "Actions/DisconnectNodesAction.h"
#include "Actions/GetClassFunctionsAction.h"
#include "Actions/SaveMemoryAction.h"
#include "Actions/RemoveMemoryAction.h"
#include "Actions/GetOpenBlueprintAction.h"
#include "Actions/GetSelectedNodesAction.h"
#include "Actions/SetNodeCommentAction.h"
#include "Actions/AddCommentBoxAction.h"
#include "Actions/SearchBlueprintNodesAction.h"
#include "Actions/GetComponentPropertiesAction.h"

FActionRegistry::FActionRegistry()
{
	RegisterBuiltInActions();
}

void FActionRegistry::RegisterBuiltInActions()
{
	// Core Blueprint creation
	RegisterAction(MakeShared<FCreateBlueprintAction>(this));
	RegisterAction(MakeShared<FAddNodeAction>(this));
	RegisterAction(MakeShared<FConnectNodesAction>(this));
	RegisterAction(MakeShared<FAddVariableAction>());
	RegisterAction(MakeShared<FAddFunctionAction>(this));
	RegisterAction(MakeShared<FCompileBlueprintAction>());

	// Context / read operations
	RegisterAction(MakeShared<FGetBlueprintInfoAction>(this));
	RegisterAction(MakeShared<FListAssetsAction>());
	RegisterAction(MakeShared<FGetProjectStructureAction>());

	// Edit operations
	RegisterAction(MakeShared<FDeleteNodeAction>(this));
	RegisterAction(MakeShared<FSetPinValueAction>(this));
	RegisterAction(MakeShared<FRenameVariableAction>());
	RegisterAction(MakeShared<FSetVariableDefaultsAction>());

	// Component operations
	RegisterAction(MakeShared<FAddComponentAction>());
	RegisterAction(MakeShared<FSetComponentPropertyAction>());
	RegisterAction(MakeShared<FRemoveComponentAction>());

	// Connection operations
	RegisterAction(MakeShared<FDisconnectNodesAction>(this));

	// Discovery operations
	RegisterAction(MakeShared<FGetClassFunctionsAction>());
	RegisterAction(MakeShared<FSearchBlueprintNodesAction>(this));
	RegisterAction(MakeShared<FGetComponentPropertiesAction>());

	// Editor operations
	RegisterAction(MakeShared<FOpenBlueprintAction>());
	RegisterAction(MakeShared<FGetOpenBlueprintAction>());
	RegisterAction(MakeShared<FGetSelectedNodesAction>(this));
	RegisterAction(MakeShared<FSetNodeCommentAction>(this));
	RegisterAction(MakeShared<FAddCommentBoxAction>(this));

	// Memory operations
	RegisterAction(MakeShared<FSaveMemoryAction>());
	RegisterAction(MakeShared<FRemoveMemoryAction>());
}

void FActionRegistry::RegisterAction(TSharedPtr<IBlueprintAction> Action)
{
	if (Action.IsValid())
	{
		ActionMap.Add(Action->GetName(), Action);
		Actions.Add(Action);
	}
}

TSharedPtr<IBlueprintAction> FActionRegistry::FindAction(const FString& Name) const
{
	const TSharedPtr<IBlueprintAction>* Found = ActionMap.Find(Name);
	return Found ? *Found : nullptr;
}

TArray<TSharedPtr<FJsonObject>> FActionRegistry::BuildToolDefinitions(const TSharedPtr<IAIProvider>& Provider) const
{
	TArray<TSharedPtr<FJsonObject>> ToolDefs;
	if (!Provider.IsValid()) return ToolDefs;

	for (const auto& Action : Actions)
	{
		TSharedPtr<FJsonObject> ToolDef = Provider->FormatToolDefinition(
			Action->GetName(),
			Action->GetDescription(),
			Action->GetParameterSchema()
		);
		if (ToolDef.IsValid())
		{
			ToolDefs.Add(ToolDef);
		}
	}

	return ToolDefs;
}

FString FActionRegistry::GenerateToolDocumentation() const
{
	// Ultra-compact format to minimize prompt size for bridge/proxy providers.
	// One line per tool: name(required_params; optional_params) - description
	FString Doc;
	Doc += TEXT("Tools:\n");

	for (const auto& Action : Actions)
	{
		Doc += FString::Printf(TEXT("- %s"), *Action->GetName());

		TSharedPtr<FJsonObject> Schema = Action->GetParameterSchema();
		if (Schema.IsValid())
		{
			// Collect required and optional param names
			TSet<FString> RequiredSet;
			const TArray<TSharedPtr<FJsonValue>>* RequiredArr;
			if (Schema->TryGetArrayField(TEXT("required"), RequiredArr))
			{
				for (const auto& V : *RequiredArr)
				{
					RequiredSet.Add(V->AsString());
				}
			}

			const TSharedPtr<FJsonObject>* PropsObj;
			if (Schema->TryGetObjectField(TEXT("properties"), PropsObj))
			{
				FString ParamList;
				for (const auto& Pair : (*PropsObj)->Values)
				{
					if (!ParamList.IsEmpty()) ParamList += TEXT(", ");
					ParamList += Pair.Key;
					if (!RequiredSet.Contains(Pair.Key))
					{
						ParamList += TEXT("?"); // marks optional
					}
				}
				if (!ParamList.IsEmpty())
				{
					Doc += FString::Printf(TEXT("(%s)"), *ParamList);
				}
			}
		}

		Doc += FString::Printf(TEXT(" - %s\n"), *Action->GetDescription());
	}

	Doc += TEXT("\n");
	return Doc;
}

// ── Node tracking ────────────────────────────────────────────────────────────

FString FActionRegistry::TrackNode(UEdGraphNode* Node)
{
	if (!Node) return TEXT("");

	// Check if this node pointer is already tracked — return existing ID to prevent duplicates
	for (const auto& Pair : TrackedNodes)
	{
		if (Pair.Value.IsValid() && Pair.Value.Get() == Node)
		{
			return Pair.Key;
		}
	}

	FString NodeId = FString::Printf(TEXT("N%d"), NextNodeId++);
	TrackedNodes.Add(NodeId, Node);
	return NodeId;
}

UEdGraphNode* FActionRegistry::FindTrackedNode(const FString& NodeId) const
{
	const TWeakObjectPtr<UEdGraphNode>* Found = TrackedNodes.Find(NodeId);
	if (Found && Found->IsValid())
	{
		return Found->Get();
	}
	return nullptr;
}

void FActionRegistry::ClearSessionState()
{
	TrackedNodes.Empty();
	NextNodeId = 1;
	CreatedBlueprints.Empty();
}

void FActionRegistry::TrackCreatedBlueprint(const FString& Name, const FString& Path)
{
	// Avoid duplicates
	for (const auto& Pair : CreatedBlueprints)
	{
		if (Pair.Key.Equals(Name, ESearchCase::IgnoreCase))
		{
			return;
		}
	}
	CreatedBlueprints.Add(TPair<FString, FString>(Name, Path));
}

FString FActionRegistry::GetTrackedNodesInfo() const
{
	if (TrackedNodes.Num() == 0)
	{
		return FString();
	}

	// Group nodes by owning blueprint name
	TMap<FString, TArray<FString>> NodesByBlueprint;
	for (const auto& Pair : TrackedNodes)
	{
		if (!Pair.Value.IsValid()) continue;
		UEdGraphNode* Node = Pair.Value.Get();

		// Walk up: Node → UEdGraph → UBlueprint
		FString BPName = TEXT("Unknown");
		if (UEdGraph* Graph = Node->GetGraph())
		{
			UObject* Outer = Graph->GetOuter();
			if (Outer)
			{
				BPName = Outer->GetName();
			}
		}

		FString NodeTitle = Node->GetNodeTitle(ENodeTitleType::ListView).ToString();
		if (NodeTitle.IsEmpty()) NodeTitle = Node->GetClass()->GetName();
		FString Entry = FString::Printf(TEXT("%s=%s"), *Pair.Key, *NodeTitle);
		NodesByBlueprint.FindOrAdd(BPName).Add(Entry);
	}

	FString Result;
	for (const auto& BPPair : NodesByBlueprint)
	{
		Result += BPPair.Key + TEXT(": ") + FString::Join(BPPair.Value, TEXT(", ")) + TEXT("\n");
	}
	return Result;
}
