// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "AI/OpenRouterProvider.h"
#include "Serialization/JsonSerializer.h"

static const FString OpenRouterEndpoint = TEXT("https://openrouter.ai/api/v1/chat/completions");

FOpenRouterProvider::FOpenRouterProvider(const FString& InApiKey, const FString& InModel)
	: FOpenAIProvider(InApiKey, InModel, OpenRouterEndpoint)
	, RouterApiKey(InApiKey)
{
}

void FOpenRouterProvider::SendRequest(
	const FAICompletionRequest& Request,
	FOnAIResponse OnResponse,
	FOnAIError OnError)
{
	// OpenRouter accepts 'max_tokens' (unlike direct OpenAI which now requires
	// 'max_completion_tokens'). Override to build the body with max_tokens
	// for maximum compatibility across the many upstream providers OpenRouter proxies.
	TMap<FString, FString> Headers;
	Headers.Add(TEXT("Authorization"), FString::Printf(TEXT("Bearer %s"), *RouterApiKey));

	TSharedRef<FJsonObject> Body = MakeShared<FJsonObject>();
	Body->SetStringField(TEXT("model"), Request.Model.IsEmpty() ? Model : Request.Model);
	Body->SetNumberField(TEXT("max_tokens"), Request.MaxTokens);
	Body->SetNumberField(TEXT("temperature"), Request.Temperature);
	Body->SetArrayField(TEXT("messages"), BuildMessages(Request.Messages));

	if (Request.Tools.Num() > 0)
	{
		TArray<TSharedPtr<FJsonValue>> ToolsArray;
		for (const auto& Tool : Request.Tools)
		{
			ToolsArray.Add(MakeShared<FJsonValueObject>(Tool));
		}
		Body->SetArrayField(TEXT("tools"), ToolsArray);
	}

	SendHttpRequest(Endpoint, Headers, Body, OnResponse, OnError);
}

FString FOpenRouterProvider::GetModelsEndpoint() const
{
	return TEXT("https://openrouter.ai/api/v1/models");
}

TMap<FString, FString> FOpenRouterProvider::GetAuthHeaders() const
{
	TMap<FString, FString> Headers;
	Headers.Add(TEXT("Authorization"), FString::Printf(TEXT("Bearer %s"), *RouterApiKey));
	return Headers;
}
