// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "AI/OpenAIProvider.h"
#include "Serialization/JsonSerializer.h"

static const FString DefaultOpenAIEndpoint = TEXT("https://api.openai.com/v1/chat/completions");

FOpenAIProvider::FOpenAIProvider(const FString& InApiKey, const FString& InModel, const FString& InEndpoint)
	: ApiKey(InApiKey)
	, Model(InModel)
	, Endpoint(InEndpoint.IsEmpty() ? DefaultOpenAIEndpoint : InEndpoint)
{
	// Auto-detect Responses API from the endpoint URL
	bUseResponsesAPI = Endpoint.Contains(TEXT("/responses"));
	if (bUseResponsesAPI)
	{
		UE_LOG(LogTemp, Log, TEXT("BlueprintAI: OpenAI provider using Responses API format (endpoint: %s)"), *Endpoint);
	}
}

void FOpenAIProvider::SendRequest(
	const FAICompletionRequest& Request,
	FOnAIResponse OnResponse,
	FOnAIError OnError)
{
	TMap<FString, FString> Headers;
	Headers.Add(TEXT("Authorization"), FString::Printf(TEXT("Bearer %s"), *ApiKey));

	TSharedRef<FJsonObject> Body = MakeShared<FJsonObject>();
	const FString ActiveModel = Request.Model.IsEmpty() ? Model : Request.Model;
	Body->SetStringField(TEXT("model"), ActiveModel);

	if (bUseResponsesAPI)
	{
		// ── Responses API format (/v1/responses) ──
		Body->SetNumberField(TEXT("max_output_tokens"), Request.MaxTokens);

		// Extract system prompt → top-level "instructions", rest → "input" array
		FString Instructions;
		Body->SetArrayField(TEXT("input"), BuildResponsesInput(Request.Messages, Instructions));
		if (!Instructions.IsEmpty())
		{
			Body->SetStringField(TEXT("instructions"), Instructions);
		}

		// Tools in Responses API use same format as Chat Completions
		if (Request.Tools.Num() > 0)
		{
			TArray<TSharedPtr<FJsonValue>> ToolsArray;
			for (const auto& Tool : Request.Tools)
			{
				ToolsArray.Add(MakeShared<FJsonValueObject>(Tool));
			}
			Body->SetArrayField(TEXT("tools"), ToolsArray);
		}
	}
	else
	{
		// ── Chat Completions API format (/v1/chat/completions) ──
		// Newer OpenAI models (GPT-4o, o1, etc.) require 'max_completion_tokens'
		// instead of 'max_tokens'. Use the new parameter for direct OpenAI calls.
		Body->SetNumberField(TEXT("max_completion_tokens"), Request.MaxTokens);

		// NOTE: We intentionally omit "temperature" for direct OpenAI calls.
		// Many newer models (o1, o3, o4, chatgpt-4o-latest, etc.) reject any
		// temperature value other than the default (1.0). Since it's impossible
		// to reliably detect all such models by name, we let OpenAI use its
		// default for all models. This avoids HTTP 400 errors.

		// Messages
		Body->SetArrayField(TEXT("messages"), BuildMessages(Request.Messages));

		// Tools
		if (Request.Tools.Num() > 0)
		{
			TArray<TSharedPtr<FJsonValue>> ToolsArray;
			for (const auto& Tool : Request.Tools)
			{
				ToolsArray.Add(MakeShared<FJsonValueObject>(Tool));
			}
			Body->SetArrayField(TEXT("tools"), ToolsArray);
		}
	}

	SendHttpRequest(Endpoint, Headers, Body, OnResponse, OnError);
}

TSharedPtr<FJsonObject> FOpenAIProvider::FormatToolDefinition(
	const FString& Name,
	const FString& Description,
	const TSharedPtr<FJsonObject>& ParamSchema) const
{
	// OpenAI format:
	// { "type": "function", "function": { "name": "...", "description": "...", "parameters": { JSON Schema } } }
	TSharedPtr<FJsonObject> FuncObj = MakeShared<FJsonObject>();
	FuncObj->SetStringField(TEXT("name"), Name);
	FuncObj->SetStringField(TEXT("description"), Description);
	if (ParamSchema.IsValid())
	{
		FuncObj->SetObjectField(TEXT("parameters"), ParamSchema);
	}

	TSharedPtr<FJsonObject> Tool = MakeShared<FJsonObject>();
	Tool->SetStringField(TEXT("type"), TEXT("function"));
	Tool->SetObjectField(TEXT("function"), FuncObj);
	return Tool;
}

FAICompletionResponse FOpenAIProvider::ParseResponse(const FString& ResponseBody)
{
	FAICompletionResponse Result;

	TSharedPtr<FJsonObject> Json = StringToJson(ResponseBody);
	if (!Json.IsValid())
	{
		Result.ErrorMessage = TEXT("Failed to parse OpenAI response JSON");
		return Result;
	}

	// Check for error (Responses API includes "error": null for success — only treat as error if it's a real object)
	if (Json->HasField(TEXT("error")))
	{
		const TSharedPtr<FJsonObject>* ErrorObj;
		if (Json->TryGetObjectField(TEXT("error"), ErrorObj) && ErrorObj && (*ErrorObj).IsValid())
		{
			FString ErrMsg = (*ErrorObj)->GetStringField(TEXT("message"));

			// Friendly error: if the model requires a different endpoint
			if (ErrMsg.Contains(TEXT("not a chat model")) || ErrMsg.Contains(TEXT("not supported in the v1/chat/completions")))
			{
				ErrMsg = FString::Printf(
					TEXT("The model '%s' is not compatible with the Chat Completions API. "
						 "Try changing the Endpoint URL in Project Settings → Plugins → Blueprint AI Assistant → OpenAI to: "
						 "https://api.openai.com/v1/responses"),
					*Model);
			}
			else if (ErrMsg.Contains(TEXT("not supported in the v1/responses")))
			{
				ErrMsg = FString::Printf(
					TEXT("The model '%s' is not compatible with the Responses API. "
						 "Try changing the Endpoint URL in Project Settings → Plugins → Blueprint AI Assistant → OpenAI to: "
						 "https://api.openai.com/v1/chat/completions"),
					*Model);
			}

			Result.ErrorMessage = ErrMsg;
			return Result;
		}
		// If "error" field exists but is null → not an error, continue parsing
	}

	// Dispatch to the correct parser
	if (bUseResponsesAPI)
	{
		return ParseResponsesAPIResponse(Json);
	}

	// ── Chat Completions API response parsing ──
	Result.bSuccess = true;

	// Parse usage
	const TSharedPtr<FJsonObject>* Usage;
	if (Json->TryGetObjectField(TEXT("usage"), Usage))
	{
		Result.InputTokens = (*Usage)->GetIntegerField(TEXT("prompt_tokens"));
		Result.OutputTokens = (*Usage)->GetIntegerField(TEXT("completion_tokens"));
	}

	// Parse choices[0].message
	const TArray<TSharedPtr<FJsonValue>>* Choices;
	if (Json->TryGetArrayField(TEXT("choices"), Choices) && Choices->Num() > 0)
	{
		const TSharedPtr<FJsonObject>& Choice = (*Choices)[0]->AsObject();
		const TSharedPtr<FJsonObject>* MessageObj;
		if (Choice.IsValid() && Choice->TryGetObjectField(TEXT("message"), MessageObj))
		{
			// Text content
			Result.TextContent = (*MessageObj)->GetStringField(TEXT("content"));

			// Tool calls
			const TArray<TSharedPtr<FJsonValue>>* ToolCallsArray;
			if ((*MessageObj)->TryGetArrayField(TEXT("tool_calls"), ToolCallsArray))
			{
				for (const auto& TCVal : *ToolCallsArray)
				{
					const TSharedPtr<FJsonObject>& TCObj = TCVal->AsObject();
					if (!TCObj.IsValid()) continue;

					FAIToolCall ToolCall;
					ToolCall.Id = TCObj->GetStringField(TEXT("id"));

					const TSharedPtr<FJsonObject>* FuncObj;
					if (TCObj->TryGetObjectField(TEXT("function"), FuncObj))
					{
						ToolCall.Name = (*FuncObj)->GetStringField(TEXT("name"));
						FString ArgsStr = (*FuncObj)->GetStringField(TEXT("arguments"));
						ToolCall.Parameters = StringToJson(ArgsStr);
						if (!ToolCall.Parameters.IsValid())
						{
							ToolCall.Parameters = MakeShared<FJsonObject>();
						}
					}

					Result.ToolCalls.Add(MoveTemp(ToolCall));
				}
			}
		}
	}

	return Result;
}

// ── Responses API helpers ───────────────────────────────────────────────────

TArray<TSharedPtr<FJsonValue>> FOpenAIProvider::BuildResponsesInput(
	const TArray<FAIMessage>& Messages, FString& OutInstructions) const
{
	// Responses API uses:
	//   "instructions": "<system prompt>"         (top-level, not in input array)
	//   "input": [
	//     { "role": "user", "content": "..." },
	//     { "role": "assistant", "content": "..." },
	//     { "type": "function_call", "name": "...", "arguments": "...", "call_id": "..." },
	//     { "type": "function_call_output", "call_id": "...", "output": "..." },
	//   ]
	TArray<TSharedPtr<FJsonValue>> Result;
	OutInstructions.Empty();

	for (const FAIMessage& Msg : Messages)
	{
		switch (Msg.Role)
		{
		case EAIMessageRole::System:
			// System message → top-level "instructions" (concatenate if multiple)
			if (!OutInstructions.IsEmpty())
			{
				OutInstructions += TEXT("\n\n");
			}
			OutInstructions += Msg.Content;
			break;

		case EAIMessageRole::User:
			{
				TSharedPtr<FJsonObject> Item = MakeShared<FJsonObject>();
				Item->SetStringField(TEXT("role"), TEXT("user"));
				Item->SetStringField(TEXT("content"), Msg.Content);
				Result.Add(MakeShared<FJsonValueObject>(Item));
			}
			break;

		case EAIMessageRole::Assistant:
			{
				// Text content as an assistant message
				if (!Msg.Content.IsEmpty())
				{
					TSharedPtr<FJsonObject> Item = MakeShared<FJsonObject>();
					Item->SetStringField(TEXT("role"), TEXT("assistant"));
					Item->SetStringField(TEXT("content"), Msg.Content);
					Result.Add(MakeShared<FJsonValueObject>(Item));
				}

				// Tool calls → each becomes a separate "function_call" input item
				for (const FAIToolCall& TC : Msg.ToolCalls)
				{
					TSharedPtr<FJsonObject> Item = MakeShared<FJsonObject>();
					Item->SetStringField(TEXT("type"), TEXT("function_call"));
					Item->SetStringField(TEXT("call_id"), TC.Id);
					Item->SetStringField(TEXT("name"), TC.Name);
					Item->SetStringField(TEXT("arguments"), TC.Parameters.IsValid()
						? JsonToString(TC.Parameters.ToSharedRef()) : TEXT("{}"));
					Result.Add(MakeShared<FJsonValueObject>(Item));
				}
			}
			break;

		case EAIMessageRole::ToolResult:
			{
				// Tool result → "function_call_output"
				TSharedPtr<FJsonObject> Item = MakeShared<FJsonObject>();
				Item->SetStringField(TEXT("type"), TEXT("function_call_output"));
				Item->SetStringField(TEXT("call_id"), Msg.ToolCallId);
				Item->SetStringField(TEXT("output"), Msg.Content);
				Result.Add(MakeShared<FJsonValueObject>(Item));
			}
			break;
		}
	}

	return Result;
}

FAICompletionResponse FOpenAIProvider::ParseResponsesAPIResponse(const TSharedPtr<FJsonObject>& Json) const
{
	// Responses API response format:
	// {
	//   "id": "...",
	//   "output": [
	//     { "type": "message", "content": [ { "type": "output_text", "text": "..." } ] },
	//     { "type": "function_call", "name": "...", "arguments": "...", "call_id": "..." }
	//   ],
	//   "usage": { "input_tokens": N, "output_tokens": N }
	// }
	FAICompletionResponse Result;
	Result.bSuccess = true;

	// Parse usage
	const TSharedPtr<FJsonObject>* Usage;
	if (Json->TryGetObjectField(TEXT("usage"), Usage))
	{
		Result.InputTokens = (*Usage)->HasField(TEXT("input_tokens"))
			? (*Usage)->GetIntegerField(TEXT("input_tokens")) : 0;
		Result.OutputTokens = (*Usage)->HasField(TEXT("output_tokens"))
			? (*Usage)->GetIntegerField(TEXT("output_tokens")) : 0;
	}

	// Parse output array
	const TArray<TSharedPtr<FJsonValue>>* OutputArray;
	if (!Json->TryGetArrayField(TEXT("output"), OutputArray) || !OutputArray)
	{
		UE_LOG(LogTemp, Warning, TEXT("BlueprintAI: Responses API returned no 'output' array"));
		return Result;
	}

	FString CombinedText;

	for (const auto& OutputVal : *OutputArray)
	{
		const TSharedPtr<FJsonObject>& OutputObj = OutputVal->AsObject();
		if (!OutputObj.IsValid()) continue;

		FString OutputType = OutputObj->GetStringField(TEXT("type"));

		if (OutputType == TEXT("message"))
		{
			// Extract text from content array
			const TArray<TSharedPtr<FJsonValue>>* ContentArray;
			if (OutputObj->TryGetArrayField(TEXT("content"), ContentArray))
			{
				for (const auto& ContentVal : *ContentArray)
				{
					const TSharedPtr<FJsonObject>& ContentObj = ContentVal->AsObject();
					if (!ContentObj.IsValid()) continue;

					FString ContentType = ContentObj->GetStringField(TEXT("type"));
					if (ContentType == TEXT("output_text"))
					{
						FString Text = ContentObj->GetStringField(TEXT("text"));
						if (!Text.IsEmpty())
						{
							if (!CombinedText.IsEmpty()) CombinedText += TEXT("\n");
							CombinedText += Text;
						}
					}
				}
			}
		}
		else if (OutputType == TEXT("function_call"))
		{
			// Parse function call
			FAIToolCall ToolCall;
			ToolCall.Id = OutputObj->GetStringField(TEXT("call_id"));
			ToolCall.Name = OutputObj->GetStringField(TEXT("name"));

			FString ArgsStr = OutputObj->GetStringField(TEXT("arguments"));
			ToolCall.Parameters = StringToJson(ArgsStr);
			if (!ToolCall.Parameters.IsValid())
			{
				ToolCall.Parameters = MakeShared<FJsonObject>();
			}

			Result.ToolCalls.Add(MoveTemp(ToolCall));
		}
	}

	Result.TextContent = CombinedText;
	return Result;
}

// ── Chat Completions API helpers ────────────────────────────────────────────

TArray<TSharedPtr<FJsonValue>> FOpenAIProvider::BuildMessages(const TArray<FAIMessage>& Messages) const
{
	TArray<TSharedPtr<FJsonValue>> Result;

	for (const FAIMessage& Msg : Messages)
	{
		TSharedPtr<FJsonObject> MsgObj = MakeShared<FJsonObject>();

		switch (Msg.Role)
		{
		case EAIMessageRole::System:
			MsgObj->SetStringField(TEXT("role"), TEXT("system"));
			MsgObj->SetStringField(TEXT("content"), Msg.Content);
			break;

		case EAIMessageRole::User:
			MsgObj->SetStringField(TEXT("role"), TEXT("user"));
			MsgObj->SetStringField(TEXT("content"), Msg.Content);
			break;

		case EAIMessageRole::Assistant:
			{
				MsgObj->SetStringField(TEXT("role"), TEXT("assistant"));
				if (!Msg.Content.IsEmpty())
				{
					MsgObj->SetStringField(TEXT("content"), Msg.Content);
				}
				// Tool calls
				if (Msg.ToolCalls.Num() > 0)
				{
					TArray<TSharedPtr<FJsonValue>> TCArray;
					for (const FAIToolCall& TC : Msg.ToolCalls)
					{
						TSharedPtr<FJsonObject> TCObj = MakeShared<FJsonObject>();
						TCObj->SetStringField(TEXT("id"), TC.Id);
						TCObj->SetStringField(TEXT("type"), TEXT("function"));

						TSharedPtr<FJsonObject> FuncObj = MakeShared<FJsonObject>();
						FuncObj->SetStringField(TEXT("name"), TC.Name);
						FuncObj->SetStringField(TEXT("arguments"), JsonToString(TC.Parameters.ToSharedRef()));
						TCObj->SetObjectField(TEXT("function"), FuncObj);

						TCArray.Add(MakeShared<FJsonValueObject>(TCObj));
					}
					MsgObj->SetArrayField(TEXT("tool_calls"), TCArray);
				}
			}
			break;

		case EAIMessageRole::ToolResult:
			MsgObj->SetStringField(TEXT("role"), TEXT("tool"));
			MsgObj->SetStringField(TEXT("tool_call_id"), Msg.ToolCallId);
			MsgObj->SetStringField(TEXT("content"), Msg.Content);
			break;
		}

		Result.Add(MakeShared<FJsonValueObject>(MsgObj));
	}

	return Result;
}

FString FOpenAIProvider::GetModelsEndpoint() const
{
	// Derive models endpoint from the chat completions endpoint
	// e.g. https://api.openai.com/v1/chat/completions -> https://api.openai.com/v1/models
	FString Url = Endpoint;

	// Find /v1/ (or /v2/) and replace everything after with /models
	int32 VersionIdx = Url.Find(TEXT("/v"), ESearchCase::IgnoreCase, ESearchDir::FromEnd);
	if (VersionIdx != INDEX_NONE)
	{
		int32 NextSlash = Url.Find(TEXT("/"), ESearchCase::CaseSensitive, ESearchDir::FromStart, VersionIdx + 2);
		if (NextSlash != INDEX_NONE)
		{
			return Url.Left(NextSlash) + TEXT("/models");
		}
		return Url + TEXT("/models");
	}

	return TEXT("https://api.openai.com/v1/models");
}

TMap<FString, FString> FOpenAIProvider::GetAuthHeaders() const
{
	TMap<FString, FString> Headers;
	Headers.Add(TEXT("Authorization"), FString::Printf(TEXT("Bearer %s"), *ApiKey));
	return Headers;
}
