// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "AI/CustomProvider.h"
#include "AI/ClaudeProvider.h"
#include "AI/OpenAIProvider.h"

FCustomProvider::FCustomProvider(
	const FString& InEndpoint,
	const FString& InApiKey,
	const FString& InModel,
	EAIProviderType InFormat)
	: Endpoint(InEndpoint)
	, ApiKey(InApiKey)
	, Model(InModel)
	, Format(InFormat)
{
	// Normalize the endpoint so it always points to /chat/completions.
	// Users may enter:
	//   http://host:port/v1/chat/completions  (correct)
	//   http://host:port/v1/models            (models URL)
	//   http://host:port/v1                   (base)
	//   http://host:port                      (bare host)
	FString ChatEndpoint = Endpoint;

	if (ChatEndpoint.EndsWith(TEXT("/v1/models")))
	{
		ChatEndpoint = ChatEndpoint.LeftChop(7) + TEXT("/chat/completions"); // strip "/models", add "/chat/completions"
	}
	else if (ChatEndpoint.EndsWith(TEXT("/v1")) || ChatEndpoint.EndsWith(TEXT("/v1/")))
	{
		ChatEndpoint.RemoveFromEnd(TEXT("/"));
		ChatEndpoint += TEXT("/chat/completions");
	}
	else if (!ChatEndpoint.Contains(TEXT("/chat/completions")))
	{
		// Bare host or unknown path — assume OpenAI-compatible
		ChatEndpoint.RemoveFromEnd(TEXT("/"));
		ChatEndpoint += TEXT("/v1/chat/completions");
	}

	UE_LOG(LogTemp, Log, TEXT("BlueprintAI: Custom endpoint normalized: %s -> %s"), *Endpoint, *ChatEndpoint);

	// Create an inner provider that handles the actual formatting.
	// We override the endpoint at the HTTP level.
	if (Format == EAIProviderType::Claude)
	{
		InnerProvider = MakeShared<FClaudeProvider>(ApiKey, Model, TEXT("2023-06-01"));
	}
	else
	{
		// Default to OpenAI format
		InnerProvider = MakeShared<FOpenAIProvider>(ApiKey, Model, ChatEndpoint);
	}
}

void FCustomProvider::SendRequest(
	const FAICompletionRequest& Request,
	FOnAIResponse OnResponse,
	FOnAIError OnError)
{
	if (InnerProvider.IsValid())
	{
		// For OpenAI format, the endpoint is already set in the constructor.
		// For Claude format, we need to override the endpoint via direct HTTP call.
		if (Format == EAIProviderType::Claude)
		{
			// Build Claude-format body and send to custom endpoint
			// For simplicity, delegate to the inner provider
			// TODO: Override endpoint for Claude format custom providers
			InnerProvider->SendRequest(Request, OnResponse, OnError);
		}
		else
		{
			InnerProvider->SendRequest(Request, OnResponse, OnError);
		}
	}
	else
	{
		OnError.ExecuteIfBound(TEXT("Custom provider not initialized"));
	}
}

TSharedPtr<FJsonObject> FCustomProvider::FormatToolDefinition(
	const FString& Name,
	const FString& Description,
	const TSharedPtr<FJsonObject>& ParamSchema) const
{
	if (InnerProvider.IsValid())
	{
		return InnerProvider->FormatToolDefinition(Name, Description, ParamSchema);
	}
	return nullptr;
}

FAICompletionResponse FCustomProvider::ParseResponse(const FString& ResponseBody)
{
	// Not used directly — inner provider handles this
	FAICompletionResponse R;
	R.ErrorMessage = TEXT("ParseResponse called on FCustomProvider directly");
	return R;
}

FString FCustomProvider::GetModelsEndpoint() const
{
	if (Endpoint.IsEmpty()) return FString();

	FString Url = Endpoint;

	// If it already ends with /models, use it directly
	if (Url.EndsWith(TEXT("/models")))
	{
		return Url;
	}

	// Try to find /v1/ (or /v2/, etc.) and replace everything after with /models
	// e.g. http://host/v1/chat/completions -> http://host/v1/models
	int32 VersionIdx = Url.Find(TEXT("/v"), ESearchCase::IgnoreCase, ESearchDir::FromEnd);
	if (VersionIdx != INDEX_NONE)
	{
		// Find the slash AFTER /v1
		int32 NextSlash = Url.Find(TEXT("/"), ESearchCase::CaseSensitive, ESearchDir::FromStart, VersionIdx + 2);
		if (NextSlash != INDEX_NONE)
		{
			return Url.Left(NextSlash) + TEXT("/models");
		}
		// URL ends at /v1 — just append /models
		return Url + TEXT("/models");
	}

	// Fallback: strip last path segment, append /models
	int32 LastSlash;
	if (Url.FindLastChar(TEXT('/'), LastSlash) && LastSlash > 8) // > 8 to skip http://
	{
		return Url.Left(LastSlash) + TEXT("/models");
	}

	return FString();
}

TMap<FString, FString> FCustomProvider::GetAuthHeaders() const
{
	TMap<FString, FString> Headers;
	if (!ApiKey.IsEmpty())
	{
		Headers.Add(TEXT("Authorization"), FString::Printf(TEXT("Bearer %s"), *ApiKey));
	}
	return Headers;
}
