// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "AI/ConversationManager.h"
#include "Actions/ActionRegistry.h"
#include "Settings/BlueprintAISettings.h"
#include "Logging/BlueprintAILogger.h"
#include "Serialization/JsonSerializer.h"
#include "Serialization/JsonWriter.h"
#include "Misc/ScopedSlowTask.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "Engine/Blueprint.h"
#include "Memory/BlueprintAIMemory.h"

FConversationManager::FConversationManager(TSharedPtr<FActionRegistry> InRegistry)
	: ActionRegistry(InRegistry)
{
}

void FConversationManager::SetProvider(TSharedPtr<IAIProvider> InProvider)
{
	Provider = InProvider;
}

void FConversationManager::SendUserMessage(const FString& UserMessage)
{
	if (bIsBusy)
	{
		OnChatMessage.Broadcast(TEXT("System"), TEXT("Please wait — the AI is still processing."));
		return;
	}

	if (!Provider.IsValid())
	{
		OnChatMessage.Broadcast(TEXT("System"), TEXT("No AI provider configured. Please set up your API key in Project Settings → Plugins → Blueprint AI Assistant."));
		return;
	}

	// Display user message in UI
	OnChatMessage.Broadcast(TEXT("User"), UserMessage);
	FBlueprintAILogger::Get().Log(TEXT("USER"), UserMessage);

	// Add to history
	ConversationHistory.Add(FAIMessage::MakeUser(UserMessage));

	// Begin the agent loop
	CurrentIteration = 0;
	TextToolCallCounter = 0;
	ConsecutiveDuplicateBatches = 0;
	NudgeCount = 0;
	bHadDeferredWiring = false;
	WiringNudgeCount = 0;
	RecentErrorCounts.Empty();
	ConsecutiveFailedIterations = 0;
	LastBatchSignature.Empty();
	ConsecutiveIdenticalBatches = 0;
	MaxIterations = UBlueprintAISettings::Get()->MaxToolIterations;

	// Invalidate observation caches — the user may have manually edited Blueprints between messages
	InvalidateStaleCacheEntries();

	FBlueprintAILogger::Get().Logf(TEXT("AGENT"), TEXT("Starting agent loop (max %d iterations)"), MaxIterations);

	bIsBusy = true;
	OnBusyStateChanged.Broadcast(true);

	SendToProvider();
}

void FConversationManager::ClearConversation()
{
	ConversationHistory.Empty();
	ToolCallCache.Empty();
	ActionRegistry->ClearSessionState();
	TextToolCallCounter = 0;
	TotalInputTokensUsed = 0;
	TotalOutputTokensUsed = 0;
	LastRequestInputTokens = 0;
	LastRequestPayloadChars = 0;
	bNeedsSummarization = false;
	NudgeCount = 0;
	bHadDeferredWiring = false;
	WiringNudgeCount = 0;
	RecentErrorCounts.Empty();
	ConsecutiveFailedIterations = 0;
	OnContextUsageUpdated.Broadcast(0, 0, ModelContextWindow);
	OnChatMessage.Broadcast(TEXT("System"), TEXT("Conversation cleared."));
}

void FConversationManager::SetModelContextWindow(int32 ContextTokens)
{
	ModelContextWindow = ContextTokens;
	FBlueprintAILogger::Get().Logf(TEXT("CONTEXT"), TEXT("Model context window set to %d tokens"), ContextTokens);
	OnContextUsageUpdated.Broadcast(TotalInputTokensUsed, TotalOutputTokensUsed, ModelContextWindow);
}

int32 FConversationManager::GetEffectiveMaxPayloadChars() const
{
	if (ModelContextWindow > 0)
	{
		// Rough estimate: ~3 chars per token for English text
		int32 ModelBasedChars = ModelContextWindow * 3;
		FBlueprintAILogger::Get().Logf(TEXT("CONTEXT"), TEXT("Using model context: %d tokens → %d chars (overriding user settings)"),
			ModelContextWindow, ModelBasedChars);
		return ModelBasedChars;
	}
	return UBlueprintAISettings::Get()->MaxTotalPayloadChars;
}

FString FConversationManager::MakeToolCacheKey(const FString& Name, const TSharedPtr<FJsonObject>& Params)
{
	FString ParamsStr;
	if (Params.IsValid())
	{
		TSharedRef<TJsonWriter<TCHAR, TCondensedJsonPrintPolicy<TCHAR>>> Writer =
			TJsonWriterFactory<TCHAR, TCondensedJsonPrintPolicy<TCHAR>>::Create(&ParamsStr);
		FJsonSerializer::Serialize(Params.ToSharedRef(), Writer);
	}
	return Name + TEXT("|") + ParamsStr;
}

void FConversationManager::StopGeneration()
{
	if (!bIsBusy) return;

	if (Provider.IsValid())
	{
		Provider->CancelRequest();
	}

	// Set bIsBusy = false AFTER CancelRequest so that any stale callbacks
	// that sneak through are caught by the guard check in HandleResponse/HandleError.
	bIsBusy = false;
	OnBusyStateChanged.Broadcast(false);
	OnChatMessage.Broadcast(TEXT("System"), TEXT("Generation stopped by user."));
	FBlueprintAILogger::Get().Log(TEXT("SYSTEM"), TEXT("Generation stopped by user."));
}

// ─── Project Snapshot ────────────────────────────────────────────────────────

FString FConversationManager::BuildProjectSnapshot() const
{
	FString Snapshot;

	FAssetRegistryModule& AssetReg = FModuleManager::LoadModuleChecked<FAssetRegistryModule>("AssetRegistry");
	IAssetRegistry& Registry = AssetReg.Get();

	// Gather all assets under /Game
	TArray<FAssetData> AllAssets;
	Registry.GetAssetsByPath(FName(TEXT("/Game")), AllAssets, /*bRecursive=*/true);

	// Count by type and collect Blueprint names specifically
	TMap<FString, int32> TypeCounts;
	TArray<FString> BlueprintNames;
	for (const FAssetData& Asset : AllAssets)
	{
		FString TypeName = Asset.AssetClassPath.GetAssetName().ToString();
		TypeCounts.FindOrAdd(TypeName)++;

		// Collect Blueprint names (these are what the AI will work with)
		if (TypeName == TEXT("Blueprint") || TypeName == TEXT("WidgetBlueprint"))
		{
			BlueprintNames.Add(Asset.AssetName.ToString());
		}
	}

	Snapshot += TEXT("## Project Overview\n");
	Snapshot += FString::Printf(TEXT("Total assets: %d\n"), AllAssets.Num());

	// Asset type summary (just counts)
	TArray<FString> SortedTypes;
	TypeCounts.GetKeys(SortedTypes);
	SortedTypes.Sort();
	for (const FString& TypeName : SortedTypes)
	{
		Snapshot += FString::Printf(TEXT("  %s: %d\n"), *TypeName, TypeCounts[TypeName]);
	}

	// List all Blueprints by name (the AI needs to know these)
	if (BlueprintNames.Num() > 0)
	{
		BlueprintNames.Sort();
		Snapshot += TEXT("\nBlueprints in project:\n");
		for (const FString& Name : BlueprintNames)
		{
			Snapshot += FString::Printf(TEXT("  - %s\n"), *Name);
		}
	}

	Snapshot += TEXT("\n");
	return Snapshot;
}

// ─── System Prompt ───────────────────────────────────────────────────────────

FString FConversationManager::BuildWorkingState() const
{
	FString State;

	const auto& CreatedBPs = ActionRegistry->GetCreatedBlueprints();
	if (CreatedBPs.Num() > 0)
	{
		State += TEXT("=== SESSION STATE (created this conversation) ===\n");
		State += TEXT("Created Blueprints:\n");
		for (const auto& BP : CreatedBPs)
		{
			State += FString::Printf(TEXT("  - %s (%s)\n"), *BP.Key, *BP.Value);
		}
	}

	FString NodeInfo = ActionRegistry->GetTrackedNodesInfo();
	if (!NodeInfo.IsEmpty())
	{
		if (State.IsEmpty()) State += TEXT("=== SESSION STATE ===\n");
		State += TEXT("Tracked Nodes:\n");
		State += NodeInfo;
	}

	if (!State.IsEmpty())
	{
		State += TEXT("IMPORTANT: Do NOT re-create blueprints listed above. They already exist.\n");
		State += TEXT("Use ONLY the node IDs listed above for connect_nodes.\n\n");
	}

	return State;
}

FString FConversationManager::BuildSystemPrompt() const
{
	FString P;

	P += TEXT("You are a UE5 Blueprint assistant in the Editor. Call tools via <tool_call> XML tags.\n");
	P += TEXT("Format: <tool_call>{\"name\":\"tool\",\"parameters\":{...}}</tool_call>\n");
	P += TEXT("Multiple calls allowed per response. You MUST use <tool_call> tags to act — never just describe.\n");
	P += TEXT("CRITICAL: NEVER respond with only text describing what you plan to do. EVERY response MUST contain at least one <tool_call>. ");
	P += TEXT("NEVER ask the user for confirmation or permission — just execute immediately. ");
	P += TEXT("If the user asks you to do something, DO IT in that same response with tool calls. Do NOT say 'I will...' or 'Let me...' without including <tool_call> tags.\n\n");

	P += TEXT("=== WORKFLOW (MUST FOLLOW — STRICTLY ENFORCED) ===\n");
	P += TEXT("Step 1: Create ALL nodes (add_node) in one response. Do NOT include connect_nodes or set_pin_value in the same response.\n");
	P += TEXT("         If you mix add_node with connect_nodes or set_pin_value, the connect/set_pin calls are AUTOMATICALLY DEFERRED and will NOT execute.\n");
	P += TEXT("Step 2: Read the NODE ID REFERENCE returned from Step 1. NEVER predict or assume node IDs — use EXACTLY the IDs returned (N1, N2, etc.).\n");
	P += TEXT("Step 3: In your NEXT response, issue ALL connect_nodes and set_pin_value calls using the actual returned IDs. This is CRITICAL — nodes will remain unconnected if you skip this step.\n");
	P += TEXT("Step 4: compile_blueprint to check for errors.\n");
	P += TEXT("IMPORTANT: After creating nodes, you MUST follow up with connect_nodes + set_pin_value in the VERY NEXT response. Never skip Step 3!\n\n");

	P += TEXT("=== PIN NAMES ===\n");
	P += TEXT("Exec pins: output='then', input='execute'. Branch: output 'then' (true) and 'else' (false), input 'Condition'.\n");
	P += TEXT("Variable getter: output pin name = variable name (e.g. Get PlayerRef → 'PlayerRef', NOT 'ReturnValue').\n");
	P += TEXT("Variable setter: input pin name = variable name. Has 'execute' input and 'then' output.\n");
	P += TEXT("PURE nodes (math, getters, IsValid, GetDistanceTo, GetActorLocation, etc.) have NO exec pins — don't try to connect exec flow through them.\n");
	P += TEXT("SpawnActor: uses 'SpawnTransform' input (connect a MakeTransform), and 'Class' pin (set via set_pin_value).\n");
	P += TEXT("Sequence: 'then_0','then_1'. IsValid: bool output 'ReturnValue', exec output 'then'.\n");
	P += TEXT("ForLoop: 'FirstIndex','LastIndex' int inputs, 'LoopBody' and 'Completed' exec outputs, 'Index' int output.\n");
	P += TEXT("ForEachLoop: 'Array' input, 'LoopBody' and 'Completed' exec outputs, 'ArrayElement' and 'ArrayIndex' outputs.\n");
	P += TEXT("WhileLoop: 'Condition' bool input, 'LoopBody' and 'Completed' exec outputs.\n");
	P += TEXT("DoOnce: exec in 'execute', exec out 'Completed'. Fires only once. Has 'Reset' exec input.\n");
	P += TEXT("FlipFlop: exec out 'A' and 'B' (alternates), bool output 'IsA'.\n");
	P += TEXT("Gate: exec in 'Enter', 'Open', 'Close', 'Toggle'. Exec out 'Exit'.\n");
	P += TEXT("Events: BeginPlay='ReceiveBeginPlay', Tick='ReceiveTick', Overlap='ReceiveActorBeginOverlap'.\n");
	P += TEXT("add_function returns entry node with node_id and pins. ALWAYS connect entry's 'then' output → your first node's 'execute' input.\n\n");

	P += TEXT("=== COLLISION ===\n");
	P += TEXT("Use set_component_property to configure collision on any PrimitiveComponent (StaticMesh, Sphere, Box, Capsule, etc.):\n");
	P += TEXT("  CollisionProfileName: 'NoCollision', 'OverlapAll', 'BlockAll', 'Projectile', 'OverlapAllDynamic', etc.\n");
	P += TEXT("  CollisionEnabled: 'NoCollision', 'QueryOnly', 'PhysicsOnly', 'QueryAndPhysics'\n");
	P += TEXT("  bSimulatePhysics: 'true'/'false'\n");
	P += TEXT("  bGenerateOverlapEvents: 'true'/'false'\n");
	P += TEXT("  CollisionResponseToAllChannels: 'Block', 'Overlap', 'Ignore'\n");
	P += TEXT("  bNotifyRigidBodyCollision: 'true'/'false' (needed for Hit events)\n");
	P += TEXT("To make bullets pass through the turret, set CollisionProfileName='NoCollision' or 'OverlapAllDynamic' on the bullet mesh.\n\n");

	P += TEXT("=== FUNCTION NAMES ===\n");
	P += TEXT("UE5 internal function names often have K2_ prefix. Use these EXACT names for CallFunction:\n");
	P += TEXT("  GetActorLocation → class='Actor', name='K2_GetActorLocation'\n");
	P += TEXT("  GetActorRotation → class='Actor', name='K2_GetActorRotation'\n");
	P += TEXT("  SetActorLocation → class='Actor', name='K2_SetActorLocation'\n");
	P += TEXT("  SetActorRotation → class='Actor', name='K2_SetActorRotation'\n");
	P += TEXT("  DestroyActor → class='Actor', name='K2_DestroyActor'\n");
	P += TEXT("OR use dedicated node_types: Delay, PrintString, SetTimer, GetPlayerController, SpawnActor, Self, Branch, ForLoop, ForEachLoop, WhileLoop, DoOnce, FlipFlop, Gate.\n");
	P += TEXT("Space nodes ~300px apart on X.\n");
	P += TEXT("FindLookAtRotation: input pins are 'Start' and 'Target' (NOT 'From'/'To').\n\n");

	P += TEXT("=== CONTEXT EFFICIENCY ===\n");
	P += TEXT("Observation results from previous iterations are automatically compacted. If you need that data again, re-query.\n");
	P += TEXT("Prefer TARGETED queries over broad ones to keep context compact:\n");
	P += TEXT("  - search_blueprint_nodes(query='Jump') instead of get_blueprint_info(include_nodes=true) when looking for specific nodes.\n");
	P += TEXT("  - get_component_properties(component_name='CharacterMovement', name_filter='Jump') instead of dumping all properties.\n");
	P += TEXT("  - get_blueprint_info WITHOUT include_nodes=true unless you truly need the full node list.\n\n");

	P += TEXT("RULES:\n");
	P += TEXT("- ACT IMMEDIATELY. Never say 'I will create...' or 'Let me add...' without <tool_call> tags in the SAME response. Every response = tool calls.\n");
	P += TEXT("- NEVER ask for confirmation. The user already told you what to do — just do it.\n");
	P += TEXT("- If the user doesn't specify a Blueprint, call get_open_blueprint first to see what they have open.\n");
	P += TEXT("- NEVER re-call a tool with same parameters. If data was truncated, work with what you have.\n");
	P += TEXT("- After getting blueprint info, immediately start making changes.\n");
	P += TEXT("- add_component and add_variable are idempotent — OK to call even if they already exist.\n");
	P += TEXT("- Use save_memory to remember user preferences, project conventions, or important facts across sessions.\n");
	P += TEXT("- Use remove_memory to delete outdated memories.\n");
	P += TEXT("- If connect_nodes fails with 'incompatible pin types', call get_blueprint_info with include_nodes=true to check actual pin names and types.\n");
	P += TEXT("- If a tool keeps failing, try a DIFFERENT approach (use get_class_functions, check pin names, etc.) instead of retrying the same call.\n\n");

	// AI Memory section (persistent across sessions)
	P += FBlueprintAIMemory::Get().BuildMemoryPromptSection();

	// Compact tool reference
	P += ActionRegistry->GenerateToolDocumentation();

	// Working state (created BPs, tracked nodes) — never truncated
	P += BuildWorkingState();

	// Project snapshot
	P += BuildProjectSnapshot();

	return P;
}

// ─── Text-Based Tool Call Parsing ────────────────────────────────────────────

/** Attempt to parse JSON with basic error recovery (trailing chars, extra braces) */
static TSharedPtr<FJsonObject> TryParseJsonWithRepair(const FString& InJsonStr)
{
	FString JsonStr = InJsonStr.TrimStartAndEnd();

	// Attempt 1: Parse as-is
	{
		TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(JsonStr);
		TSharedPtr<FJsonObject> Result;
		if (FJsonSerializer::Deserialize(Reader, Result) && Result.IsValid())
		{
			return Result;
		}
	}

	// Attempt 2: Fix unbalanced braces — count { and } and trim trailing extras
	{
		int32 OpenCount = 0;
		int32 CloseCount = 0;
		bool bInString = false;
		bool bEscape = false;
		int32 BalancedEnd = -1;

		for (int32 i = 0; i < JsonStr.Len(); i++)
		{
			TCHAR C = JsonStr[i];
			if (bEscape) { bEscape = false; continue; }
			if (C == TEXT('\\') && bInString) { bEscape = true; continue; }
			if (C == TEXT('"')) { bInString = !bInString; continue; }
			if (bInString) continue;

			if (C == TEXT('{')) OpenCount++;
			else if (C == TEXT('}'))
			{
				CloseCount++;
				if (CloseCount == OpenCount && BalancedEnd < 0)
				{
					BalancedEnd = i;
				}
			}
		}

		if (BalancedEnd > 0 && BalancedEnd < JsonStr.Len() - 1)
		{
			// Trim everything after the balanced closing brace
			FString Fixed = JsonStr.Left(BalancedEnd + 1);
			TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(Fixed);
			TSharedPtr<FJsonObject> Result;
			if (FJsonSerializer::Deserialize(Reader, Result) && Result.IsValid())
			{
				UE_LOG(LogTemp, Log, TEXT("BlueprintAI: Repaired JSON (trimmed %d trailing chars)"), JsonStr.Len() - Fixed.Len());
				return Result;
			}
		}
	}

	// Attempt 3: Remove trailing non-JSON characters
	{
		int32 LastBrace = JsonStr.Find(TEXT("}"), ESearchCase::CaseSensitive, ESearchDir::FromEnd);
		if (LastBrace != INDEX_NONE)
		{
			FString Trimmed = JsonStr.Left(LastBrace + 1);
			TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(Trimmed);
			TSharedPtr<FJsonObject> Result;
			if (FJsonSerializer::Deserialize(Reader, Result) && Result.IsValid())
			{
				return Result;
			}
		}
	}

	return nullptr;
}

/** Extract a tool call from a parsed JSON object and add it to the array */
static bool ExtractToolCall(const TSharedPtr<FJsonObject>& JsonObj, TArray<FAIToolCall>& OutToolCalls, int32& Counter)
{
	if (!JsonObj.IsValid()) return false;

	FString ToolName = JsonObj->GetStringField(TEXT("name"));
	if (ToolName.IsEmpty()) return false;

	FAIToolCall Call;
	Call.Id = FString::Printf(TEXT("textcall_%d"), Counter++);
	Call.Name = ToolName;

	const TSharedPtr<FJsonObject>* ParamsObj;
	if (JsonObj->TryGetObjectField(TEXT("parameters"), ParamsObj))
	{
		Call.Parameters = *ParamsObj;
	}
	else
	{
		Call.Parameters = MakeShared<FJsonObject>();
	}

	OutToolCalls.Add(MoveTemp(Call));
	return true;
}

TArray<FAIToolCall> FConversationManager::ParseTextBasedToolCalls(const FString& TextContent, FString& OutCleanText) const
{
	TArray<FAIToolCall> ToolCalls;
	OutCleanText = TextContent;

	// Look for <tool_call>JSON</tool_call> patterns
	const FString OpenTag = TEXT("<tool_call>");
	const FString CloseTag = TEXT("</tool_call>");

	int32 SearchStart = 0;
	while (true)
	{
		int32 TagStart = OutCleanText.Find(OpenTag, ESearchCase::IgnoreCase, ESearchDir::FromStart, SearchStart);
		if (TagStart == INDEX_NONE) break;

		int32 JsonStart = TagStart + OpenTag.Len();
		int32 TagEnd = OutCleanText.Find(CloseTag, ESearchCase::IgnoreCase, ESearchDir::FromStart, JsonStart);
		if (TagEnd == INDEX_NONE) break;

		FString JsonStr = OutCleanText.Mid(JsonStart, TagEnd - JsonStart).TrimStartAndEnd();

		TSharedPtr<FJsonObject> JsonObj = TryParseJsonWithRepair(JsonStr);
		if (JsonObj.IsValid())
		{
			ExtractToolCall(JsonObj, ToolCalls, TextToolCallCounter);
		}
		else
		{
			UE_LOG(LogTemp, Warning, TEXT("BlueprintAI: Failed to parse text tool call JSON: %s"), *JsonStr.Left(200));
		}

		// Remove the tool call from the display text
		FString FullTag = OutCleanText.Mid(TagStart, (TagEnd + CloseTag.Len()) - TagStart);
		OutCleanText = OutCleanText.Replace(*FullTag, TEXT(""));
	}

	// ── Fallback: <function=name><parameter=key>value</parameter> format ──
	// Some free models (Gemma, Llama) use this XML-style format inside <tool_call> or standalone.
	{
		const FString FuncOpenPrefix = TEXT("<function=");
		int32 FuncSearch = 0;
		while (true)
		{
			int32 FuncStart = OutCleanText.Find(FuncOpenPrefix, ESearchCase::IgnoreCase, ESearchDir::FromStart, FuncSearch);
			if (FuncStart == INDEX_NONE) break;

			// Extract function name: <function=TOOL_NAME>
			int32 NameStart = FuncStart + FuncOpenPrefix.Len();
			int32 NameEnd = OutCleanText.Find(TEXT(">"), ESearchCase::CaseSensitive, ESearchDir::FromStart, NameStart);
			if (NameEnd == INDEX_NONE) break;

			FString FuncName = OutCleanText.Mid(NameStart, NameEnd - NameStart).TrimStartAndEnd();

			// Find the closing </function> tag
			int32 FuncClose = OutCleanText.Find(TEXT("</function>"), ESearchCase::IgnoreCase, ESearchDir::FromStart, NameEnd);
			if (FuncClose == INDEX_NONE)
			{
				// Try alternate: just take until end of text or next tag
				FuncClose = OutCleanText.Len();
			}

			// Extract all <parameter=KEY>VALUE</parameter> pairs between NameEnd and FuncClose
			TSharedPtr<FJsonObject> Params = MakeShared<FJsonObject>();
			FString ParamRegion = OutCleanText.Mid(NameEnd + 1, FuncClose - (NameEnd + 1));

			const FString ParamPrefix = TEXT("<parameter=");
			const FString ParamClose = TEXT("</parameter>");
			int32 ParamSearch = 0;
			while (true)
			{
				int32 PStart = ParamRegion.Find(ParamPrefix, ESearchCase::IgnoreCase, ESearchDir::FromStart, ParamSearch);
				if (PStart == INDEX_NONE) break;

				int32 PKStart = PStart + ParamPrefix.Len();
				int32 PKEnd = ParamRegion.Find(TEXT(">"), ESearchCase::CaseSensitive, ESearchDir::FromStart, PKStart);
				if (PKEnd == INDEX_NONE) break;

				FString ParamKey = ParamRegion.Mid(PKStart, PKEnd - PKStart).TrimStartAndEnd();

				int32 PVStart = PKEnd + 1;
				int32 PVEnd = ParamRegion.Find(ParamClose, ESearchCase::IgnoreCase, ESearchDir::FromStart, PVStart);
				if (PVEnd == INDEX_NONE) break;

				FString ParamVal = ParamRegion.Mid(PVStart, PVEnd - PVStart).TrimStartAndEnd();
				Params->SetStringField(ParamKey, ParamVal);

				ParamSearch = PVEnd + ParamClose.Len();
			}

			if (!FuncName.IsEmpty())
			{
				FAIToolCall Call;
				Call.Id = FString::Printf(TEXT("textcall_%d"), TextToolCallCounter++);
				Call.Name = FuncName;
				Call.Parameters = Params;
				ToolCalls.Add(MoveTemp(Call));

				UE_LOG(LogTemp, Log, TEXT("BlueprintAI: Parsed XML-style tool call: %s with %d params"), *FuncName, Params->Values.Num());
			}

			// Remove the full <function=...>...</function> block from display text
			int32 RemoveEnd = FuncClose;
			if (OutCleanText.Mid(FuncClose, 11) == TEXT("</function>"))
			{
				RemoveEnd = FuncClose + 11;
			}
			FString FullBlock = OutCleanText.Mid(FuncStart, RemoveEnd - FuncStart);
			OutCleanText = OutCleanText.Replace(*FullBlock, TEXT(""));
			// Don't advance FuncSearch — text shifted
		}
	}

	// Also try ```json / ```tool_call code blocks as secondary fallback
	const FString CodeBlockPatterns[] = {
		TEXT("```tool_call\n"),
		TEXT("```json\n{\"name\""),
	};

	for (const FString& Pattern : CodeBlockPatterns)
	{
		SearchStart = 0;
		while (true)
		{
			int32 BlockStart = OutCleanText.Find(Pattern, ESearchCase::IgnoreCase, ESearchDir::FromStart, SearchStart);
			if (BlockStart == INDEX_NONE) break;

			int32 JsonStartIdx;
			if (Pattern.Contains(TEXT("{\"name\"")))
			{
				JsonStartIdx = BlockStart + 8; // After ```json\n
			}
			else
			{
				JsonStartIdx = BlockStart + Pattern.Len();
			}

			int32 BlockEnd = OutCleanText.Find(TEXT("```"), ESearchCase::CaseSensitive, ESearchDir::FromStart, JsonStartIdx);
			if (BlockEnd == INDEX_NONE) break;

			FString JsonStr = OutCleanText.Mid(JsonStartIdx, BlockEnd - JsonStartIdx).TrimStartAndEnd();

			TSharedPtr<FJsonObject> JsonObj = TryParseJsonWithRepair(JsonStr);
			if (JsonObj.IsValid())
			{
				ExtractToolCall(JsonObj, ToolCalls, TextToolCallCounter);
			}

			FString FullBlock = OutCleanText.Mid(BlockStart, (BlockEnd + 3) - BlockStart);
			OutCleanText = OutCleanText.Replace(*FullBlock, TEXT(""));
		}
	}

	OutCleanText = OutCleanText.TrimStartAndEnd();
	return ToolCalls;
}

// ─── Agent Loop ──────────────────────────────────────────────────────────────

void FConversationManager::SendToProvider()
{
	if (!Provider.IsValid())
	{
		HandleError(TEXT("No AI provider available"));
		return;
	}

	if (CurrentIteration >= MaxIterations)
	{
		OnChatMessage.Broadcast(TEXT("System"),
			FString::Printf(TEXT("Reached maximum tool iterations (%d). Stopping."), MaxIterations));
		bIsBusy = false;
		OnBusyStateChanged.Broadcast(false);
		return;
	}

	// ── Mid-loop summarization: if context is getting large, summarize before sending ──
	if (bNeedsSummarization)
	{
		// SummarizeConversation() is async — its callback will call SendToProvider() again
		SummarizeConversation();
		return;
	}

	CurrentIteration++;

	const UBlueprintAISettings* Settings = UBlueprintAISettings::Get();

	// Build request
	FAICompletionRequest Request;
	Request.Temperature = Settings->Temperature;
	Request.MaxTokens = Settings->MaxTokens;

	// System message (always first)
	TArray<FAIMessage> FullMessages;
	FullMessages.Add(FAIMessage::MakeSystem(BuildSystemPrompt()));

	// Sanitize the conversation history for providers that don't support
	// the "tool" role (e.g. CopilotAPI Bridge, local model proxies).
	// Rules:
	//   1. Convert ToolResult messages to User role
	//   2. Strip tool_calls from assistant messages
	//   3. Never send empty content
	//   4. Merge consecutive same-role messages (bridges require strict alternation)
	//   5. Truncate oversized messages to prevent payload bloat
	//   6. Compact old observation tool results to short summaries (ephemeral results)

	const int32 MaxMessageChars = Settings->MaxMessageChars;
	const int32 MaxToolResultChars = Settings->MaxToolResultChars;

	// Determine which user message is the "latest" — observation results in it
	// should remain full so the AI can process them. Older ones get compacted.
	int32 LastUserMsgIndex = -1;
	for (int32 i = ConversationHistory.Num() - 1; i >= 0; --i)
	{
		if (ConversationHistory[i].Role == EAIMessageRole::User &&
			!ConversationHistory[i].Content.StartsWith(TEXT("Tool execution results:")))
		{
			// This is the last actual user text message (not a tool result block)
			LastUserMsgIndex = i;
			break;
		}
	}
	// Find the last tool-result block index (the most recent observation data the AI hasn't processed yet)
	int32 LastToolResultBlockIndex = -1;
	for (int32 i = ConversationHistory.Num() - 1; i >= 0; --i)
	{
		if (ConversationHistory[i].Role == EAIMessageRole::User &&
			ConversationHistory[i].Content.StartsWith(TEXT("Tool execution results:")))
		{
			LastToolResultBlockIndex = i;
			break;
		}
	}

	for (int32 MsgIdx = 0; MsgIdx < ConversationHistory.Num(); ++MsgIdx)
	{
		const FAIMessage& Msg = ConversationHistory[MsgIdx];
		FAIMessage Sanitized;

		// ── Ephemeral observation compaction ──
		// Old tool-result blocks (not the most recent one) get their observation data
		// compacted to short summaries. The AI already processed the full data in a
		// previous iteration — keeping it wastes context budget.
		bool bIsOldToolResultBlock = (Msg.Role == EAIMessageRole::User &&
			Msg.Content.StartsWith(TEXT("Tool execution results:")) &&
			MsgIdx != LastToolResultBlockIndex);

		if (bIsOldToolResultBlock)
		{
			FString Compacted = CompactOldToolResults(Msg.Content);
			Sanitized = FAIMessage::MakeUser(Compacted);
		}
		else if (Msg.Role == EAIMessageRole::ToolResult)
		{
			FString ResultContent = Msg.Content;
			// Apply tool result cap first (usually stricter than general message cap)
			if (ResultContent.Len() > MaxToolResultChars)
			{
				FBlueprintAILogger::Get().Logf(TEXT("TRUNCATE"), TEXT("Tool result '%s' truncated: %d -> %d chars"),
					*Msg.ToolName, ResultContent.Len(), MaxToolResultChars);
				ResultContent = ResultContent.Left(MaxToolResultChars)
					+ FString::Printf(TEXT("\n...[truncated, %d chars total]"), ResultContent.Len());
			}
			Sanitized = FAIMessage::MakeUser(
				FString::Printf(TEXT("[Tool Result: %s]\n%s"), *Msg.ToolName, *ResultContent));
		}
		else if (Msg.Role == EAIMessageRole::Assistant)
		{
			FString Content = Msg.Content;
			if (Content.IsEmpty()) Content = TEXT("(Calling tools...)");
			if (Content.Len() > MaxMessageChars)
			{
				Content = Content.Left(MaxMessageChars)
					+ TEXT("\n...[truncated]");
			}
			Sanitized = FAIMessage::MakeAssistant(Content);
		}
		else
		{
			Sanitized = Msg;
			if (Sanitized.Content.IsEmpty())
			{
				continue; // Skip empty messages entirely
			}
			if (Sanitized.Content.Len() > MaxMessageChars)
			{
				// Preserve NODE ID REFERENCE block — critical for AI to connect nodes correctly
				const FString NodeRefMarker = TEXT("=== NODE ID REFERENCE ===");
				int32 NodeRefPos = Sanitized.Content.Find(NodeRefMarker);
				if (NodeRefPos != INDEX_NONE)
				{
					FString BeforeRef = Sanitized.Content.Left(NodeRefPos);
					FString RefAndAfter = Sanitized.Content.Mid(NodeRefPos);
					int32 AllowedBefore = FMath::Max(500, MaxMessageChars - RefAndAfter.Len());
					if (BeforeRef.Len() > AllowedBefore)
					{
						BeforeRef = BeforeRef.Left(AllowedBefore) + TEXT("\n...[truncated, node refs preserved]\n");
					}
					Sanitized.Content = BeforeRef + RefAndAfter;
				}
				else
				{
					Sanitized.Content = Sanitized.Content.Left(MaxMessageChars)
						+ TEXT("\n...[truncated]");
				}
			}
		}

		// Merge consecutive same-role messages to ensure strict alternation
		if (FullMessages.Num() > 0 && FullMessages.Last().Role == Sanitized.Role)
		{
			FullMessages.Last().Content += TEXT("\n\n") + Sanitized.Content;
		}
		else
		{
			FullMessages.Add(Sanitized);
		}
	}

	// Post-merge: cap any merged messages that grew too large (e.g. 3 tool results merged into one user msg)
	const int32 MaxMergedChars = MaxMessageChars + 1000;
	for (int32 i = 1; i < FullMessages.Num(); i++) // skip system prompt at [0]
	{
		if (FullMessages[i].Content.Len() > MaxMergedChars)
		{
			// Preserve NODE ID REFERENCE in merged messages too
			const FString NodeRefMarker = TEXT("=== NODE ID REFERENCE ===");
			int32 NodeRefPos = FullMessages[i].Content.Find(NodeRefMarker);
			if (NodeRefPos != INDEX_NONE)
			{
				FString BeforeRef = FullMessages[i].Content.Left(NodeRefPos);
				FString RefAndAfter = FullMessages[i].Content.Mid(NodeRefPos);
				int32 AllowedBefore = FMath::Max(500, MaxMergedChars - RefAndAfter.Len());
				if (BeforeRef.Len() > AllowedBefore)
				{
					BeforeRef = BeforeRef.Left(AllowedBefore) + TEXT("\n...[merged truncated, node refs preserved]\n");
				}
				FullMessages[i].Content = BeforeRef + RefAndAfter;
			}
			else
			{
				FullMessages[i].Content = FullMessages[i].Content.Left(MaxMergedChars)
					+ TEXT("\n...[truncated]");
			}
		}
	}

	// ── Total payload budget: drop oldest non-system messages if over limit ──
	const int32 MaxTotalChars = GetEffectiveMaxPayloadChars();
	if (MaxTotalChars > 0)
	{
		auto CalcTotal = [&FullMessages]() -> int32
		{
			int32 Total = 0;
			for (const FAIMessage& M : FullMessages) Total += M.Content.Len();
			return Total;
		};

		// Drop messages from index 1 (right after system prompt) until under budget.
		// Always keep system prompt [0] and the last 2 messages (current user + latest context).
		while (CalcTotal() > MaxTotalChars && FullMessages.Num() > 3)
		{
			UE_LOG(LogTemp, Log, TEXT("BlueprintAI: Payload %d chars > %d limit, dropping oldest message (%d chars)"),
				CalcTotal(), MaxTotalChars, FullMessages[1].Content.Len());
			FullMessages.RemoveAt(1);

			// After removal, ensure alternation is still valid (no consecutive same-role).
			// If [0]=system and [1] is now also system... that can't happen. But if we
			// removed a user msg and now have assistant-assistant, merge them.
			if (FullMessages.Num() > 2 && FullMessages[1].Role == FullMessages[0].Role)
			{
				// This shouldn't happen since [0] is always system, but safety check
				FullMessages.RemoveAt(1);
			}
		}
	}

	Request.Messages = FullMessages;

	// ── Hard safety cap: never exceed ~100KB regardless of user settings ──
	// OpenAI drops connections on very large payloads (87KB caused the failure in testing).
	// This is a non-configurable safety net.
	{
		static constexpr int32 HardMaxChars = 60000; // ~60K chars ≈ ~80KB JSON ≈ safe for any provider
		auto CalcHardTotal = [&FullMessages]() -> int32
		{
			int32 Total = 0;
			for (const FAIMessage& M : FullMessages) Total += M.Content.Len();
			return Total;
		};

		while (CalcHardTotal() > HardMaxChars && FullMessages.Num() > 3)
		{
			FBlueprintAILogger::Get().Logf(TEXT("WARN"), TEXT("Hard safety cap: payload %d > %d, dropping message [1] (%d chars)"),
				CalcHardTotal(), HardMaxChars, FullMessages[1].Content.Len());
			FullMessages.RemoveAt(1);
		}

		Request.Messages = FullMessages;
	}

	// DON'T send the tools array — we use text-based tool calling via
	// <tool_call> tags in the system prompt. Sending the tools parameter
	// confuses bridges/proxies that don't support structured tool calling.
	// (The tools array is left empty.)

	// Send async
	// Debug: log the message roles being sent
	UE_LOG(LogTemp, Log, TEXT("BlueprintAI: Sending %d messages (iteration %d):"), Request.Messages.Num(), CurrentIteration);
	int32 TotalChars = 0;
	for (int32 i = 0; i < Request.Messages.Num(); i++)
	{
		const FAIMessage& M = Request.Messages[i];
		FString RoleName;
		switch (M.Role)
		{
			case EAIMessageRole::System: RoleName = TEXT("system"); break;
			case EAIMessageRole::User: RoleName = TEXT("user"); break;
			case EAIMessageRole::Assistant: RoleName = TEXT("assistant"); break;
			case EAIMessageRole::ToolResult: RoleName = TEXT("TOOL(!)"); break;
		}
		TotalChars += M.Content.Len();
		UE_LOG(LogTemp, Log, TEXT("  [%d] %s (%d chars)%s"),
			i, *RoleName, M.Content.Len(),
			M.ToolCalls.Num() > 0 ? *FString::Printf(TEXT(" +%d tool_calls"), M.ToolCalls.Num()) : TEXT(""));
	}
	UE_LOG(LogTemp, Log, TEXT("BlueprintAI: Total payload ~%d chars, tools=%d"), TotalChars, Request.Tools.Num());
	LastRequestPayloadChars = TotalChars;

	// Log the full outgoing payload for debugging
	FBlueprintAILogger::Get().Logf(TEXT("HTTP-OUT"), TEXT("Sending %d messages (iteration %d), ~%d chars total"),
		Request.Messages.Num(), CurrentIteration, TotalChars);
	for (int32 j = 0; j < Request.Messages.Num(); j++)
	{
		const FAIMessage& Msg = Request.Messages[j];
		FString RoleStr;
		switch (Msg.Role)
		{
			case EAIMessageRole::System: RoleStr = TEXT("system"); break;
			case EAIMessageRole::User: RoleStr = TEXT("user"); break;
			case EAIMessageRole::Assistant: RoleStr = TEXT("assistant"); break;
			case EAIMessageRole::ToolResult: RoleStr = TEXT("tool"); break;
		}
		FBlueprintAILogger::Get().Logf(TEXT("HTTP-OUT"), TEXT("  [%d] %s (%d chars): %s"),
			j, *RoleStr, Msg.Content.Len(), *Msg.Content.Left(500));
	}

	FOnAIResponse ResponseDelegate;
	ResponseDelegate.BindSP(this, &FConversationManager::HandleResponse);

	FOnAIError ErrorDelegate;
	ErrorDelegate.BindSP(this, &FConversationManager::HandleError);

	Provider->SendRequest(Request, ResponseDelegate, ErrorDelegate);
}

void FConversationManager::HandleResponse(const FAICompletionResponse& Response)
{
	// Guard: if we're no longer busy, this is a stale response from a
	// cancelled or timed-out request — ignore it completely.
	if (!bIsBusy)
	{
		UE_LOG(LogTemp, Warning, TEXT("BlueprintAI: Ignoring stale response (bIsBusy=false)"));
		FBlueprintAILogger::Get().Log(TEXT("WARN"), TEXT("Ignoring stale response (bIsBusy=false)"));
		return;
	}

	// ── Track context usage ──
	if (Response.InputTokens > 0 || Response.OutputTokens > 0)
	{
		TotalInputTokensUsed += Response.InputTokens;
		TotalOutputTokensUsed += Response.OutputTokens;
		LastRequestInputTokens = Response.InputTokens;
		FBlueprintAILogger::Get().Logf(TEXT("CONTEXT"), TEXT("Tokens this request: in=%d out=%d | Cumulative: in=%d out=%d | Context window: %d"),
			Response.InputTokens, Response.OutputTokens, TotalInputTokensUsed, TotalOutputTokensUsed, ModelContextWindow);
		OnContextUsageUpdated.Broadcast(TotalInputTokensUsed, TotalOutputTokensUsed, ModelContextWindow);

		// Check if we're approaching context limits
		if (ModelContextWindow > 0 && LastRequestInputTokens > (ModelContextWindow * 3 / 4))
		{
			bNeedsSummarization = true;
			FBlueprintAILogger::Get().Logf(TEXT("CONTEXT"), TEXT("Input tokens (%d) > 75%% of context (%d) — summarization needed"),
				LastRequestInputTokens, ModelContextWindow);
		}
	}
	else if (!Response.TextContent.IsEmpty())
	{
		// Provider returned 0 tokens (common with bridges/proxies) — estimate from char counts
		int32 EstInputTokens = LastRequestPayloadChars / 4;
		int32 EstOutputTokens = Response.TextContent.Len() / 4;
		TotalInputTokensUsed += EstInputTokens;
		TotalOutputTokensUsed += EstOutputTokens;
		LastRequestInputTokens = EstInputTokens;
		FBlueprintAILogger::Get().Logf(TEXT("CONTEXT"), TEXT("Tokens estimated (proxy returned 0): in~%d out~%d | Cumulative: in~%d out~%d"),
			EstInputTokens, EstOutputTokens, TotalInputTokensUsed, TotalOutputTokensUsed);
		OnContextUsageUpdated.Broadcast(TotalInputTokensUsed, TotalOutputTokensUsed, ModelContextWindow);

		if (ModelContextWindow > 0 && EstInputTokens > (ModelContextWindow * 3 / 4))
		{
			bNeedsSummarization = true;
			FBlueprintAILogger::Get().Logf(TEXT("CONTEXT"), TEXT("Estimated input tokens (%d) > 75%% of context (%d) — summarization needed"),
				EstInputTokens, ModelContextWindow);
		}
	}

	// Char-based summarization fallback
	if (!bNeedsSummarization && ConversationHistory.Num() > 10)
	{
		int32 TotalHistoryChars = 0;
		for (const FAIMessage& HMsg : ConversationHistory)
		{
			TotalHistoryChars += HMsg.Content.Len();
		}
		int32 EffectiveMax = GetEffectiveMaxPayloadChars();
		if (EffectiveMax > 0 && TotalHistoryChars > (EffectiveMax * 3 / 4))
		{
			bNeedsSummarization = true;
			FBlueprintAILogger::Get().Logf(TEXT("CONTEXT"), TEXT("History chars (%d) > 75%% of payload limit (%d) — char-based summarization triggered"),
				TotalHistoryChars, EffectiveMax);
		}
	}

	// Log the full raw AI response
	FBlueprintAILogger::Get().Logf(TEXT("AI-RAW"), TEXT("Response received (%d chars text, %d structured tool_calls)"),
		Response.TextContent.Len(), Response.ToolCalls.Num());
	if (!Response.TextContent.IsEmpty())
	{
		FBlueprintAILogger::Get().Log(TEXT("AI-TEXT"), Response.TextContent);
	}
	if (!Response.RawJson.IsEmpty())
	{
		FBlueprintAILogger::Get().Log(TEXT("AI-JSON"), Response.RawJson.Left(5000));
	}

	// First check for structured tool calls (standard OpenAI/Claude format)
	if (Response.ToolCalls.Num() > 0)
	{
		FBlueprintAILogger::Get().Logf(TEXT("TOOL"), TEXT("Processing %d structured tool calls"), Response.ToolCalls.Num());
		for (const FAIToolCall& TC : Response.ToolCalls)
		{
			FBlueprintAILogger::Get().Logf(TEXT("TOOL"), TEXT("  Structured call: %s (id=%s)"), *TC.Name, *TC.Id);
		}
		// Display text content if any
		if (!Response.TextContent.IsEmpty())
		{
			OnChatMessage.Broadcast(TEXT("AI"), Response.TextContent);
		}

		// Add assistant message to history (with tool calls)
		ConversationHistory.Add(FAIMessage::MakeAssistant(Response.TextContent, Response.ToolCalls));

		// Execute tools
		ExecuteToolCalls(Response.ToolCalls);

		// Continue the loop — send tool results back to AI
		SendToProvider();
		return;
	}

	// No structured tool calls — try text-based fallback parsing
	if (!Response.TextContent.IsEmpty())
	{
		FString CleanText;
		TArray<FAIToolCall> TextToolCalls = ParseTextBasedToolCalls(Response.TextContent, CleanText);

		if (TextToolCalls.Num() > 0)
		{
			FBlueprintAILogger::Get().Logf(TEXT("TOOL"), TEXT("Parsed %d text-based tool calls from response"), TextToolCalls.Num());
			for (const FAIToolCall& TC : TextToolCalls)
			{
				FBlueprintAILogger::Get().Logf(TEXT("TOOL"), TEXT("  Text call: %s (id=%s)"), *TC.Name, *TC.Id);
			}
			// Display the clean text (without tool call tags)
			if (!CleanText.IsEmpty())
			{
				OnChatMessage.Broadcast(TEXT("AI"), CleanText);
			}

			// Add the assistant text as a plain message (NO tool_calls array).
			// This avoids needing "tool" role messages which bridges don't support.
			ConversationHistory.Add(FAIMessage::MakeAssistant(CleanText));

			// Execute tools and collect results into a single user-role message
			// so the bridge only sees user/assistant/system roles.
			ExecuteTextToolCalls(TextToolCalls);

			// Continue the loop
			SendToProvider();
			return;
		}

		// No tool calls at all — just text response
		OnChatMessage.Broadcast(TEXT("AI"), Response.TextContent);
		ConversationHistory.Add(FAIMessage::MakeAssistant(Response.TextContent));

		// ── Wiring nudge: AI has deferred wiring but hasn't issued connect/set_pin calls ──
		if (bHadDeferredWiring && WiringNudgeCount < 3)
		{
			WiringNudgeCount++;
			FBlueprintAILogger::Get().Logf(TEXT("WIRING-NUDGE"), TEXT("AI responded without wiring calls despite deferral (nudge #%d)"), WiringNudgeCount);
			OnChatMessage.Broadcast(TEXT("System"), TEXT("[System] Deferred wiring pending — nudging AI to connect nodes."));

			FString NudgeMsg;
			NudgeMsg += TEXT("You previously created nodes but the connect_nodes and set_pin_value calls were DEFERRED.\n");
			NudgeMsg += TEXT("The nodes are currently UNCONNECTED. You MUST issue connect_nodes and set_pin_value calls NOW.\n\n");

			FString NodeInfo = ActionRegistry->GetTrackedNodesInfo();
			if (!NodeInfo.IsEmpty())
			{
				NudgeMsg += TEXT("=== NODE ID REFERENCE ===\n");
				NudgeMsg += NodeInfo;
				NudgeMsg += TEXT("\n");
			}

			NudgeMsg += TEXT("Use these IDs with connect_nodes and set_pin_value. Do NOT re-create nodes. Do NOT call get_blueprint_info. ACT NOW.");

			ConversationHistory.Add(FAIMessage::MakeUser(NudgeMsg));
			SendToProvider();
			return;
		}

		// Narration nudge: detect AI describing intent without acting
		// Only nudge for actual intent-planning phrases, NOT for end-of-turn questions.
		// Also skip if the AI already executed tools in this agent turn — it's done working.
		if (NudgeCount < 5 && CurrentIteration <= 2)
		{
			FString LowerText = Response.TextContent.ToLower();

			// Only detect genuine planning-without-acting phrases
			bool bLooksLikeNarration =
				(LowerText.Contains(TEXT("let me")) || LowerText.Contains(TEXT("i'll")) ||
				 LowerText.Contains(TEXT("i will")) || LowerText.Contains(TEXT("i need to")) ||
				 LowerText.Contains(TEXT("i should")) || LowerText.Contains(TEXT("let's")) ||
				 LowerText.Contains(TEXT("i'm going to")) ||
				 LowerText.Contains(TEXT("first, i")) || LowerText.Contains(TEXT("i'll start")));

			// Don't nudge if the AI is asking a question (end-of-turn)
			bool bIsAskingQuestion =
				(LowerText.Contains(TEXT("would you like")) || LowerText.Contains(TEXT("do you want")) ||
				 LowerText.Contains(TEXT("shall i")) || LowerText.Contains(TEXT("what would you")) ||
				 LowerText.Contains(TEXT("what do you")) || LowerText.Contains(TEXT("?")));

			if (bLooksLikeNarration && !bIsAskingQuestion)
			{
				NudgeCount++;
				FBlueprintAILogger::Get().Logf(TEXT("NUDGE"), TEXT("AI narrated without acting (nudge #%d) — prompting tool use"), NudgeCount);
				OnChatMessage.Broadcast(TEXT("System"), TEXT("[System] AI described intent without calling tools — nudging to act."));

				ConversationHistory.Add(FAIMessage::MakeUser(
					TEXT("STOP DESCRIBING. You just responded with text only and NO tool calls. ")
					TEXT("This is NOT allowed. You MUST use <tool_call> tags to take action RIGHT NOW. ")
					TEXT("Do NOT ask for permission or confirmation. Do NOT describe what you plan to do. ")
					TEXT("Include <tool_call> tags in your very next response or the conversation will end.")));

				SendToProvider();
				return;
			}
		}

		FBlueprintAILogger::Get().Log(TEXT("AI"), TEXT("End of turn (no tool calls)."));
	}

	bIsBusy = false;
	OnBusyStateChanged.Broadcast(false);
}

// ─── Summarization ───────────────────────────────────────────────────────────

void FConversationManager::SummarizeConversation()
{
	if (ConversationHistory.Num() < 4)
	{
		// Not enough history to summarize
		bNeedsSummarization = false;
		return;
	}

	FBlueprintAILogger::Get().Log(TEXT("SUMMARIZE"), TEXT("Summarizing conversation to free context space..."));
	OnChatMessage.Broadcast(TEXT("System"), TEXT("[System] Summarizing conversation to free context space..."));

	// Build the conversation text for the summarizer
	FString ConversationText;
	for (const FAIMessage& Msg : ConversationHistory)
	{
		FString RoleName;
		switch (Msg.Role)
		{
		case EAIMessageRole::User: RoleName = TEXT("User"); break;
		case EAIMessageRole::Assistant: RoleName = TEXT("Assistant"); break;
		case EAIMessageRole::System: RoleName = TEXT("System"); break;
		case EAIMessageRole::ToolResult: RoleName = TEXT("ToolResult"); break;
		}
		ConversationText += FString::Printf(TEXT("[%s]: %s\n"), *RoleName, *Msg.Content.Left(500));
	}

	// Save the last user message (current task)
	FAIMessage LastUserMessage;
	for (int32 i = ConversationHistory.Num() - 1; i >= 0; --i)
	{
		if (ConversationHistory[i].Role == EAIMessageRole::User &&
			!ConversationHistory[i].Content.StartsWith(TEXT("Tool execution results:")))
		{
			LastUserMessage = ConversationHistory[i];
			break;
		}
	}

	// Build summarization request
	FAICompletionRequest SumRequest;
	SumRequest.Temperature = 0.2f;
	SumRequest.MaxTokens = 1024;

	TArray<FAIMessage> SumMessages;
	SumMessages.Add(FAIMessage::MakeSystem(
		TEXT("Summarize the following UE5 Blueprint AI conversation into a concise summary. Include:\n")
		TEXT("- What blueprints were created (names and paths)\n")
		TEXT("- What components were added\n")
		TEXT("- What nodes were connected\n")
		TEXT("- What remains to be done\n")
		TEXT("- Any user preferences expressed\n")
		TEXT("Keep it under 500 words. Be factual and specific.")
	));
	SumMessages.Add(FAIMessage::MakeUser(ConversationText.Left(15000)));

	SumRequest.Messages = SumMessages;

	// Send synchronous-style summarization (using the same async path but with a simple callback)
	TWeakPtr<FConversationManager> WeakSelf = AsShared();

	FOnAIResponse SumResponseDelegate;
	SumResponseDelegate.BindLambda([WeakSelf, LastUserMessage](const FAICompletionResponse& SumResponse)
	{
		TSharedPtr<FConversationManager> Self = WeakSelf.Pin();
		if (!Self.IsValid()) return;

		if (SumResponse.bSuccess && !SumResponse.TextContent.IsEmpty())
		{
			// Replace conversation history with summary + last user message
			Self->ConversationHistory.Empty();
			Self->ToolCallCache.Empty(); // Clear cache — old entries reference pre-summary state
			Self->ConversationHistory.Add(FAIMessage::MakeUser(
				TEXT("Previous conversation summary:\n") + SumResponse.TextContent
			));
			if (!LastUserMessage.Content.IsEmpty())
			{
				Self->ConversationHistory.Add(LastUserMessage);
			}

			FBlueprintAILogger::Get().Log(TEXT("SUMMARIZE"), TEXT("Conversation summarized successfully — resuming agent loop."));
			Self->OnChatMessage.Broadcast(TEXT("System"), TEXT("[System] Conversation summarized to free context space."));
		}
		else
		{
			FBlueprintAILogger::Get().Log(TEXT("SUMMARIZE"), TEXT("Summarization failed — keeping original history."));
		}

		Self->bNeedsSummarization = false;

		// Resume the agent loop — SendToProvider will now continue with compacted history
		if (Self->bIsBusy)
		{
			Self->SendToProvider();
		}
	});

	FOnAIError SumErrorDelegate;
	SumErrorDelegate.BindLambda([WeakSelf](const FString& Error)
	{
		TSharedPtr<FConversationManager> Self = WeakSelf.Pin();
		if (!Self.IsValid()) return;

		FBlueprintAILogger::Get().Logf(TEXT("SUMMARIZE"), TEXT("Summarization failed: %s — resuming without summarization."), *Error);
		Self->bNeedsSummarization = false;

		// Resume the agent loop even if summarization failed
		if (Self->bIsBusy)
		{
			Self->SendToProvider();
		}
	});

	Provider->SendRequest(SumRequest, SumResponseDelegate, SumErrorDelegate);
}

void FConversationManager::HandleError(const FString& Error)
{
	// Guard: if we're no longer busy, this is a stale error (e.g. from a
	// cancelled request whose completion fired after CancelRequest). Ignore.
	if (!bIsBusy)
	{
		UE_LOG(LogTemp, Warning, TEXT("BlueprintAI: Ignoring stale error (bIsBusy=false): %s"), *Error);
		return;
	}

	// Remove any poisoned messages from the history that could cause
	// subsequent requests to also fail (e.g. tool role messages that
	// the bridge doesn't support, or assistant messages with tool_calls).
	// Walk backwards and remove tool results and any assistant messages with
	// attached tool calls that were part of the current failed iteration.
	while (ConversationHistory.Num() > 0)
	{
		const FAIMessage& Last = ConversationHistory.Last();
		if (Last.Role == EAIMessageRole::ToolResult)
		{
			ConversationHistory.Pop();
		}
		else if (Last.Role == EAIMessageRole::Assistant && Last.ToolCalls.Num() > 0)
		{
			ConversationHistory.Pop();
		}
		else if (Last.Role == EAIMessageRole::User && Last.Content.StartsWith(TEXT("Tool execution results:")))
		{
			// Remove aggregated text-tool-call results too
			ConversationHistory.Pop();
		}
		else if (Last.Role == EAIMessageRole::Assistant && Last.Content.IsEmpty())
		{
			// Remove empty assistant messages from text tool calls
			ConversationHistory.Pop();
		}
		else
		{
			break;
		}
	}

	OnChatMessage.Broadcast(TEXT("Error"), Error);
	FBlueprintAILogger::Get().Log(TEXT("ERROR"), Error);

	bIsBusy = false;
	OnBusyStateChanged.Broadcast(false);
}

FString FConversationManager::ComputeBatchSignature(const TArray<FAIToolCall>& ToolCalls)
{
	// Build a sorted list of "name|params_json" strings so order doesn't matter
	TArray<FString> Entries;
	Entries.Reserve(ToolCalls.Num());
	for (const FAIToolCall& Call : ToolCalls)
	{
		FString ParamsStr;
		if (Call.Parameters.IsValid())
		{
			TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&ParamsStr);
			FJsonSerializer::Serialize(Call.Parameters.ToSharedRef(), Writer);
		}
		Entries.Add(FString::Printf(TEXT("%s|%s"), *Call.Name, *ParamsStr));
	}
	Entries.Sort();
	return FString::Join(Entries, TEXT(";;"));
}

void FConversationManager::CheckSuccessLoop(const TArray<FAIToolCall>& ToolCalls)
{
	FString Signature = ComputeBatchSignature(ToolCalls);

	if (!LastBatchSignature.IsEmpty() && Signature == LastBatchSignature)
	{
		ConsecutiveIdenticalBatches++;
		FBlueprintAILogger::Get().Logf(TEXT("SUCCESS-LOOP"), TEXT("Identical batch repeated %d time(s)"), ConsecutiveIdenticalBatches);
	}
	else
	{
		ConsecutiveIdenticalBatches = 0;
	}
	LastBatchSignature = Signature;

	if (ConsecutiveIdenticalBatches >= 3)
	{
		FString Guidance = TEXT("CRITICAL: You have repeated the EXACT SAME set of tool calls ")
			+ FString::Printf(TEXT("%d times in a row. "), ConsecutiveIdenticalBatches + 1)
			+ TEXT("All calls succeeded but you are making ZERO progress. ")
			+ TEXT("STOP repeating these calls immediately. Either:\n")
			+ TEXT("1. Finish your work and respond with a final summary message (no more tool calls), OR\n")
			+ TEXT("2. Try a completely DIFFERENT approach to solve the remaining problem.\n")
			+ TEXT("Do NOT call the same tools with the same parameters again.");

		FBlueprintAILogger::Get().Log(TEXT("SUCCESS-LOOP"), Guidance);
		ConversationHistory.Add(FAIMessage::MakeUser(Guidance));
	}
}

// Actions that modify Blueprint state — after any of these succeeds,
// compile_blueprint and get_blueprint_info caches become stale.
static const TSet<FString> StateChangingActions = {
	TEXT("add_node"), TEXT("connect_nodes"), TEXT("delete_node"),
	TEXT("set_pin_value"), TEXT("add_variable"), TEXT("add_function"),
	TEXT("set_variable_defaults"), TEXT("rename_variable"), TEXT("create_blueprint"),
	TEXT("add_component"), TEXT("set_component_property"), TEXT("remove_component"),
	TEXT("disconnect_nodes")
};

// Actions that should NEVER be deduped (they observe or change mutable state).
static const TSet<FString> NeverDedupActions = {
	TEXT("compile_blueprint"),
	TEXT("get_open_blueprint"),
	TEXT("get_selected_nodes")
};

void FConversationManager::InvalidateStaleCacheEntries()
{
	// Remove cached observation AND state-changing tool entries
	// because the Blueprint state has changed since they were cached.
	// State-changing entries must also be invalidated so re-adding a node with
	// identical params after deleting one is not blocked as a "duplicate".
	// All observation tools whose cached results may be stale after any state change
	static const TArray<FString> ObservationCachePrefixes = {
		TEXT("compile_blueprint|"),
		TEXT("get_blueprint_info|"),
		TEXT("get_project_structure|"),
		TEXT("get_class_functions|"),
		TEXT("list_assets|"),
		TEXT("search_blueprint_nodes|"),
		TEXT("get_component_properties|")
	};

	TArray<FString> KeysToRemove;
	for (const auto& Pair : ToolCallCache)
	{
		// Invalidate observation caches
		bool bIsObservation = false;
		for (const FString& Prefix : ObservationCachePrefixes)
		{
			if (Pair.Key.StartsWith(Prefix))
			{
				bIsObservation = true;
				break;
			}
		}
		if (bIsObservation)
		{
			KeysToRemove.Add(Pair.Key);
			continue;
		}

		// Invalidate state-changing tool caches so identical re-calls are allowed
		for (const FString& Action : StateChangingActions)
		{
			if (Pair.Key.StartsWith(Action + TEXT("|")))
			{
				KeysToRemove.Add(Pair.Key);
				break;
			}
		}
	}
	for (const FString& Key : KeysToRemove)
	{
		ToolCallCache.Remove(Key);
	}
	if (KeysToRemove.Num() > 0)
	{
		FBlueprintAILogger::Get().Logf(TEXT("TOOL-DEDUP"), TEXT("Invalidated %d stale cache entries after state change"),
			KeysToRemove.Num());
	}
}

FString FConversationManager::CompactOldToolResults(const FString& AggregatedToolResults)
{
	// This function takes an old "Tool execution results:\n\n[tool_name] SUCCESS\nResult: {...}\n\n..."
	// block and replaces verbose observation tool JSON with short summaries.
	// State-changing tool results (add_node, connect_nodes, etc.) are kept as-is since they're already compact.

	// Observation tools whose full JSON should be replaced with summaries
	static const TSet<FString> VerboseObservationTools = {
		TEXT("get_blueprint_info"),
		TEXT("get_class_functions"),
		TEXT("get_project_structure"),
		TEXT("list_assets"),
		TEXT("get_open_blueprint"),
		TEXT("get_selected_nodes"),
		TEXT("search_blueprint_nodes"),
		TEXT("get_component_properties")
	};

	FString Result;
	Result.Reserve(AggregatedToolResults.Len());

	// Parse the aggregated results line-by-line, looking for observation tool blocks
	TArray<FString> Lines;
	AggregatedToolResults.ParseIntoArray(Lines, TEXT("\n"), false);

	bool bInObservationBlock = false;
	FString CurrentToolName;
	int32 SkippedChars = 0;

	for (int32 i = 0; i < Lines.Num(); ++i)
	{
		const FString& Line = Lines[i];

		// Detect start of a tool result: "[tool_name] SUCCESS" or "[tool_name] FAILED" or "[tool_name] OK:"
		if (Line.StartsWith(TEXT("[")))
		{
			int32 CloseBracket = Line.Find(TEXT("]"), ESearchCase::CaseSensitive, ESearchDir::FromStart, 1);
			if (CloseBracket != INDEX_NONE)
			{
				FString ToolName = Line.Mid(1, CloseBracket - 1);

				if (VerboseObservationTools.Contains(ToolName))
				{
					// Check if this is a SUCCESS block with a "Result: {" following
					FString AfterBracket = Line.Mid(CloseBracket + 1).TrimStart();
					bool bHasResult = (AfterBracket.StartsWith(TEXT("SUCCESS")) || AfterBracket.StartsWith(TEXT("FAILED")));

					if (bHasResult)
					{
						// Extract the message part for a compact summary
						FString StatusWord = AfterBracket.StartsWith(TEXT("SUCCESS")) ? TEXT("OK") : TEXT("FAILED");
						Result += FString::Printf(TEXT("[%s] %s (data from previous iteration — re-query if needed)\n"), *ToolName, *StatusWord);
						bInObservationBlock = true;
						CurrentToolName = ToolName;
						continue;
					}

					// DUPLICATE lines — keep as-is (already compact)
					if (AfterBracket.Contains(TEXT("DUPLICATE")))
					{
						Result += Line + TEXT("\n");
						continue;
					}
				}

				// Not an observation tool — end any active skip block
				if (bInObservationBlock)
				{
					bInObservationBlock = false;
					CurrentToolName.Empty();
				}
			}
		}

		// If we're inside an observation block, skip the "Result: { ... }" JSON lines
		if (bInObservationBlock)
		{
			SkippedChars += Line.Len() + 1;
			// Detect end of block: an empty line after JSON, or start of next tool result
			// We'll simply skip until we hit a line starting with "[" or "Continue working"
			// or "=== NODE ID" or end of content
			if (i + 1 < Lines.Num())
			{
				const FString& NextLine = Lines[i + 1];
				if (NextLine.StartsWith(TEXT("[")) ||
					NextLine.StartsWith(TEXT("Continue working")) ||
					NextLine.StartsWith(TEXT("=== NODE ID")) ||
					NextLine.StartsWith(TEXT("STOP:")) ||
					NextLine.StartsWith(TEXT("WARNING:")) ||
					NextLine.StartsWith(TEXT("CRITICAL:")) ||
					NextLine.StartsWith(TEXT("DEFERRED:")))
				{
					bInObservationBlock = false;
					CurrentToolName.Empty();
				}
			}
			else
			{
				// Last line — end the block
				bInObservationBlock = false;
				CurrentToolName.Empty();
			}
			continue;
		}

		Result += Line + TEXT("\n");
	}

	if (SkippedChars > 0)
	{
		FBlueprintAILogger::Get().Logf(TEXT("COMPACT"), TEXT("Compacted old tool results: removed ~%d chars of stale observation data"),
			SkippedChars);
	}

	return Result;
}

void FConversationManager::ExecuteToolCalls(const TArray<FAIToolCall>& ToolCalls)
{
	int32 BatchFailCount = 0;
	int32 BatchSuccessCount = 0;

	for (const FAIToolCall& Call : ToolCalls)
	{
		OnToolExecution.Broadcast(Call.Name, TEXT("Executing..."));
		FBlueprintAILogger::Get().Logf(TEXT("TOOL-EXEC"), TEXT("Executing structured tool: %s (id=%s)"), *Call.Name, *Call.Id);
		if (Call.Parameters.IsValid())
		{
			FString ParamsStr;
			TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&ParamsStr);
			FJsonSerializer::Serialize(Call.Parameters.ToSharedRef(), Writer);
			FBlueprintAILogger::Get().Logf(TEXT("TOOL-EXEC"), TEXT("  Params: %s"), *ParamsStr.Left(2000));
		}

		// Duplicate detection for structured calls (skip for never-dedup actions)
		FString CacheKey = MakeToolCacheKey(Call.Name, Call.Parameters);
		if (!NeverDedupActions.Contains(Call.Name) && ToolCallCache.Contains(CacheKey))
		{
			FBlueprintAILogger::Get().Logf(TEXT("TOOL-DEDUP"), TEXT("Duplicate structured call blocked: %s"), *Call.Name);
			FString DedupResult = TEXT("{\"success\":true,\"message\":\"DUPLICATE - You already have this data. Use the previous result. Do NOT re-request.\"}");
			ConversationHistory.Add(FAIMessage::MakeToolResult(Call.Id, Call.Name, DedupResult));
			OnToolExecution.Broadcast(Call.Name, TEXT("OK (cached)"));
			continue;
		}

		TSharedPtr<IBlueprintAction> Action = ActionRegistry->FindAction(Call.Name);
		if (!Action.IsValid())
		{
			FString ErrorResult = FString::Printf(TEXT("{\"success\":false,\"message\":\"Unknown tool '%s'\"}"), *Call.Name);
			ConversationHistory.Add(FAIMessage::MakeToolResult(Call.Id, Call.Name, ErrorResult));
			OnToolExecution.Broadcast(Call.Name, TEXT("Unknown tool"));
			FBlueprintAILogger::Get().Logf(TEXT("TOOL-EXEC"), TEXT("  FAILED: Unknown tool '%s'"), *Call.Name);
			continue;
		}

		// Execute on game thread
		FBlueprintActionResult Result = Action->Execute(Call.Parameters);

		FString ResultJson = Result.ToJsonString();

		// Only cache successful results — failed calls should not block retries
		if (Result.bSuccess)
		{
			ToolCallCache.Add(CacheKey, ResultJson);
			BatchSuccessCount++;
		}
		else
		{
			BatchFailCount++;
			RecentErrorCounts.FindOrAdd(Call.Name)++;
		}

		// If a state-changing action succeeded, invalidate stale observation caches
		if (Result.bSuccess && StateChangingActions.Contains(Call.Name))
		{
			InvalidateStaleCacheEntries();
		}

		ConversationHistory.Add(FAIMessage::MakeToolResult(Call.Id, Call.Name, ResultJson));

		FBlueprintAILogger::Get().Logf(TEXT("TOOL-RESULT"), TEXT("  %s -> %s: %s"),
			*Call.Name, Result.bSuccess ? TEXT("OK") : TEXT("FAIL"), *ResultJson.Left(2000));

		OnToolExecution.Broadcast(Call.Name,
			FString::Printf(TEXT("%s: %s"),
				Result.bSuccess ? TEXT("OK") : TEXT("FAIL"),
				*Result.Message));
	}

	// Error loop detection for structured tool calls
	if (BatchSuccessCount == 0 && BatchFailCount > 0)
	{
		ConsecutiveFailedIterations++;
	}
	else if (BatchSuccessCount > 0)
	{
		ConsecutiveFailedIterations = 0;
	}

	// Inject guidance if error thresholds are hit
	bool bNeedsErrorGuidance = false;
	FString ErrorGuidance;
	for (const auto& ErrPair : RecentErrorCounts)
	{
		if (ErrPair.Value >= 5)
		{
			ErrorGuidance += FString::Printf(TEXT("Tool '%s' has FAILED %d times. Try a DIFFERENT approach: use get_blueprint_info with include_nodes=true to check actual pin names/types, or get_class_functions to find correct functions. Do NOT retry the same failing call.\n"),
				*ErrPair.Key, ErrPair.Value);
			bNeedsErrorGuidance = true;
		}
	}
	if (ConsecutiveFailedIterations >= 3)
	{
		ErrorGuidance += TEXT("CRITICAL: Multiple consecutive iterations have ALL failed. STOP and use get_blueprint_info with include_nodes=true to inspect the current blueprint state before trying anything else.\n");
		bNeedsErrorGuidance = true;
	}
	if (bNeedsErrorGuidance)
	{
		FBlueprintAILogger::Get().Log(TEXT("ERROR-LOOP"), ErrorGuidance);
		ConversationHistory.Add(FAIMessage::MakeUser(ErrorGuidance));
	}

	// Success loop detection: catch identical batches repeating even when all succeed
	CheckSuccessLoop(ToolCalls);
}

void FConversationManager::ExecuteTextToolCalls(const TArray<FAIToolCall>& ToolCalls)
{
	const int32 MaxToolResultChars = UBlueprintAISettings::Get()->MaxToolResultChars;
	int32 DuplicateCount = 0;
	int32 BatchFailCount = 0;
	int32 BatchSuccessCount = 0;
	bool bHadAddNode = false;
	FString AggregatedResults;
	AggregatedResults += TEXT("Tool execution results:\n\n");

	// ── Observation tools: keep full JSON results ──
	static const TSet<FString> ObservationTools = {
		TEXT("get_blueprint_info"), TEXT("list_assets"), TEXT("get_class_functions"),
		TEXT("get_project_structure"), TEXT("get_open_blueprint"), TEXT("get_selected_nodes"),
		TEXT("search_blueprint_nodes"), TEXT("get_component_properties")
	};

	// ── Workflow enforcement: detect mixed add_node + connect_nodes/set_pin_value ──
	TArray<FAIToolCall> DeferredWiringCalls;
	bool bHasAddNode = false;
	bool bHasConnect = false;
	bool bHasSetPin = false;
	for (const FAIToolCall& Call : ToolCalls)
	{
		if (Call.Name == TEXT("add_node")) bHasAddNode = true;
		if (Call.Name == TEXT("connect_nodes")) bHasConnect = true;
		if (Call.Name == TEXT("set_pin_value")) bHasSetPin = true;
	}
	bool bDeferWiring = bHasAddNode && (bHasConnect || bHasSetPin);
	if (bDeferWiring)
	{
		FBlueprintAILogger::Get().Log(TEXT("WORKFLOW"), TEXT("Mixed add_node + connect_nodes/set_pin_value detected — deferring wiring calls to next iteration"));
	}

	for (const FAIToolCall& Call : ToolCalls)
	{
		// Defer connect_nodes and set_pin_value when mixed with add_node
		if (bDeferWiring && (Call.Name == TEXT("connect_nodes") || Call.Name == TEXT("set_pin_value")))
		{
			DeferredWiringCalls.Add(Call);
			continue;
		}

		OnToolExecution.Broadcast(Call.Name, TEXT("Executing..."));
		FBlueprintAILogger::Get().Logf(TEXT("TOOL-EXEC"), TEXT("Executing text-based tool: %s (id=%s)"), *Call.Name, *Call.Id);
		if (Call.Parameters.IsValid())
		{
			FString ParamsStr;
			TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&ParamsStr);
			FJsonSerializer::Serialize(Call.Parameters.ToSharedRef(), Writer);
			FBlueprintAILogger::Get().Logf(TEXT("TOOL-EXEC"), TEXT("  Params: %s"), *ParamsStr.Left(2000));
		}

		// ── Duplicate detection ──
		FString CacheKey = MakeToolCacheKey(Call.Name, Call.Parameters);
		if (!NeverDedupActions.Contains(Call.Name) && ToolCallCache.Contains(CacheKey))
		{
			DuplicateCount++;
			FBlueprintAILogger::Get().Logf(TEXT("TOOL-DEDUP"), TEXT("Duplicate call blocked: %s"), *Call.Name);
			AggregatedResults += FString::Printf(
				TEXT("[%s] DUPLICATE — You already have this data from a previous call. Use the result above. Do NOT re-request.\n\n"),
				*Call.Name);
			OnToolExecution.Broadcast(Call.Name, TEXT("OK (cached)"));
			continue;
		}

		TSharedPtr<IBlueprintAction> Action = ActionRegistry->FindAction(Call.Name);
		if (!Action.IsValid())
		{
			AggregatedResults += FString::Printf(
				TEXT("[%s] ERROR: Unknown tool '%s'\n\n"), *Call.Name, *Call.Name);
			OnToolExecution.Broadcast(Call.Name, TEXT("Unknown tool"));
			FBlueprintAILogger::Get().Logf(TEXT("TOOL-EXEC"), TEXT("  FAILED: Unknown tool '%s'"), *Call.Name);
			continue;
		}

		FBlueprintActionResult Result = Action->Execute(Call.Parameters);
		FString ResultJson = Result.ToJsonString();

		// Track success/failure for error loop detection
		if (Result.bSuccess)
		{
			BatchSuccessCount++;
		}
		else
		{
			BatchFailCount++;
			RecentErrorCounts.FindOrAdd(Call.Name)++;
		}

		// Track if we had add_node calls
		if (Call.Name == TEXT("add_node") && Result.bSuccess) bHadAddNode = true;

		// Apply MaxToolResultChars cap to EACH individual result BEFORE aggregating
		if (ResultJson.Len() > MaxToolResultChars)
		{
			FBlueprintAILogger::Get().Logf(TEXT("TRUNCATE"), TEXT("Text tool result '%s' truncated: %d -> %d chars"),
				*Call.Name, ResultJson.Len(), MaxToolResultChars);
			ResultJson = ResultJson.Left(MaxToolResultChars)
				+ FString::Printf(TEXT("\n...[truncated, was %d chars]"), ResultJson.Len());
		}

		// Only cache successful results — failed calls should not block retries
		if (Result.bSuccess)
		{
			ToolCallCache.Add(CacheKey, ResultJson);
		}

		// If a state-changing action succeeded, invalidate stale observation caches
		if (Result.bSuccess && StateChangingActions.Contains(Call.Name))
		{
			InvalidateStaleCacheEntries();
		}

		// ── Compact vs full result format ──
		if (Result.bSuccess && !ObservationTools.Contains(Call.Name))
		{
			// Compact one-liner for state-changing operations
			AggregatedResults += FString::Printf(TEXT("[%s] OK: %s\n"), *Call.Name, *Result.Message);
		}
		else
		{
			// Full JSON for observation tools and failures
			AggregatedResults += FString::Printf(
				TEXT("[%s] %s\nResult: %s\n\n"),
				*Call.Name,
				Result.bSuccess ? TEXT("SUCCESS") : TEXT("FAILED"),
				*ResultJson);
		}

		FBlueprintAILogger::Get().Logf(TEXT("TOOL-RESULT"), TEXT("  %s -> %s: %s"),
			*Call.Name, Result.bSuccess ? TEXT("OK") : TEXT("FAIL"), *ResultJson.Left(2000));

		OnToolExecution.Broadcast(Call.Name,
			FString::Printf(TEXT("%s: %s"),
				Result.bSuccess ? TEXT("OK") : TEXT("FAIL"),
				*Result.Message));
	}

	// ── Append node ID summary if add_node was called ──
	if (bHadAddNode)
	{
		FString NodeInfo = ActionRegistry->GetTrackedNodesInfo();
		if (!NodeInfo.IsEmpty())
		{
			AggregatedResults += TEXT("\n=== NODE ID REFERENCE ===\n");
			AggregatedResults += NodeInfo;
			AggregatedResults += TEXT("Use ONLY these IDs for connect_nodes.\n");
		}
	}

	// ── Report deferred wiring calls ──
	if (DeferredWiringCalls.Num() > 0)
	{
		bHadDeferredWiring = true;

		// Count deferred by type for logging
		int32 DeferredConnects = 0, DeferredPins = 0;
		for (const FAIToolCall& D : DeferredWiringCalls)
		{
			if (D.Name == TEXT("connect_nodes")) DeferredConnects++;
			else if (D.Name == TEXT("set_pin_value")) DeferredPins++;
		}

		FBlueprintAILogger::Get().Logf(TEXT("WORKFLOW"), TEXT("Deferred %d connect_nodes + %d set_pin_value calls"),
			DeferredConnects, DeferredPins);

		// Brief note in aggregated results (may be truncated, that's OK)
		AggregatedResults += FString::Printf(
			TEXT("\nDEFERRED: %d wiring calls (connect_nodes + set_pin_value) were skipped — you mixed them with add_node.\n"),
			DeferredWiringCalls.Num());
	}
	else
	{
		// No deferral this iteration; only clear the flag if the AI actually
		// issued wiring calls (meaning it followed through successfully)
		if (bHadDeferredWiring)
		{
			bool bHasWiringThisBatch = false;
			for (const FAIToolCall& Call : ToolCalls)
			{
				if (Call.Name == TEXT("connect_nodes") || Call.Name == TEXT("set_pin_value"))
				{
					bHasWiringThisBatch = true;
					break;
				}
			}
			if (bHasWiringThisBatch)
			{
				bHadDeferredWiring = false;
				WiringNudgeCount = 0;
				FBlueprintAILogger::Get().Log(TEXT("WORKFLOW"), TEXT("AI followed through with deferred wiring calls — flag cleared"));
			}
			else if (WiringNudgeCount < 3)
			{
				// AI issued tool calls but none were wiring — it's distracted.
				// Append a reminder to the aggregated results.
				WiringNudgeCount++;
				FBlueprintAILogger::Get().Logf(TEXT("WIRING-NUDGE"),
					TEXT("AI called tools but skipped wiring (nudge #%d in batch)"), WiringNudgeCount);
				AggregatedResults += TEXT("\n⚠ REMINDER: You have UNCONNECTED nodes from a previous step. ");
				AggregatedResults += TEXT("Issue connect_nodes and set_pin_value calls NOW before doing anything else.\n");
			}
		}
	}

	// Track consecutive all-duplicate batches for loop breaking
	if (DuplicateCount > 0 && DuplicateCount == ToolCalls.Num())
	{
		ConsecutiveDuplicateBatches++;
		FBlueprintAILogger::Get().Logf(TEXT("TOOL-DEDUP"), TEXT("All %d calls were duplicates (consecutive batch #%d)"),
			DuplicateCount, ConsecutiveDuplicateBatches);
	}
	else
	{
		ConsecutiveDuplicateBatches = 0;
	}

	// Error loop detection
	if (BatchSuccessCount == 0 && BatchFailCount > 0)
	{
		ConsecutiveFailedIterations++;
	}
	else if (BatchSuccessCount > 0)
	{
		ConsecutiveFailedIterations = 0;
	}

	// Per-tool repeated failure guidance
	for (const auto& ErrPair : RecentErrorCounts)
	{
		if (ErrPair.Value >= 5)
		{
			AggregatedResults += FString::Printf(
				TEXT("\nWARNING: Tool '%s' has FAILED %d times. Try a DIFFERENT approach: use get_blueprint_info with include_nodes=true to check actual pin names/types, or get_class_functions to find correct functions. Do NOT retry the same failing call.\n"),
				*ErrPair.Key, ErrPair.Value);
		}
	}

	if (ConsecutiveFailedIterations >= 3)
	{
		AggregatedResults += TEXT("\nCRITICAL: Multiple consecutive iterations have ALL failed. STOP and use get_blueprint_info with include_nodes=true to inspect the current blueprint state before trying anything else.\n");
	}

	if (ConsecutiveDuplicateBatches >= 2)
	{
		AggregatedResults += TEXT("STOP: You have called the same tools multiple times. The data you have is all that is available. "
			"Work with the existing results NOW — start deleting bad nodes, adding correct ones, and connecting them. "
			"Do NOT call get_blueprint_info again.\n\n");
	}
	else
	{
		AggregatedResults += TEXT("\nContinue working based on these results. If you need to call more tools, use <tool_call> tags.");
	}

	// Add the main aggregated results as a user message
	ConversationHistory.Add(FAIMessage::MakeUser(AggregatedResults));

	// ── SEPARATE short message for deferred wiring — guaranteed not to be truncated ──
	// The main aggregated results can be 5-10K+ chars and gets truncated by SendToProvider,
	// which may chop off the NODE ID REFERENCE and deferral notice. This short follow-up
	// message contains ONLY the critical info the AI needs to issue wiring calls.
	if (DeferredWiringCalls.Num() > 0)
	{
		FString WiringMsg;
		WiringMsg += TEXT("=== ACTION REQUIRED: WIRING DEFERRED ===\n");
		WiringMsg += FString::Printf(
			TEXT("%d connect_nodes and set_pin_value calls were SKIPPED because you mixed them with add_node.\n"),
			DeferredWiringCalls.Num());
		WiringMsg += TEXT("In your NEXT response, you MUST issue connect_nodes and set_pin_value calls using the node IDs below.\n\n");

		// Include the NODE ID REFERENCE so the AI has it even if the main message was truncated
		FString NodeInfo = ActionRegistry->GetTrackedNodesInfo();
		if (!NodeInfo.IsEmpty())
		{
			WiringMsg += TEXT("=== NODE ID REFERENCE ===\n");
			WiringMsg += NodeInfo;
			WiringMsg += TEXT("\n");
		}

		WiringMsg += TEXT("Do NOT call add_node again. Do NOT call get_blueprint_info. ");
		WiringMsg += TEXT("Use the IDs above with connect_nodes and set_pin_value NOW.");

		ConversationHistory.Add(FAIMessage::MakeUser(WiringMsg));

		FBlueprintAILogger::Get().Logf(TEXT("WORKFLOW"), TEXT("Added separate wiring deferral message (%d chars)"), WiringMsg.Len());
	}

	// ── Post-batch cache invalidation ──
	// If add_node calls ran in this batch, observation caches (get_blueprint_info, etc.)
	// that were populated DURING this same batch are now stale. Invalidate them so the AI
	// can re-fetch fresh data if needed in the next iteration.
	if (bHadAddNode)
	{
		InvalidateStaleCacheEntries();
	}

	// Success loop detection: catch identical batches repeating even when all succeed
	CheckSuccessLoop(ToolCalls);
}
