// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "AI/ClaudeProvider.h"
#include "Serialization/JsonSerializer.h"
#include "Serialization/JsonWriter.h"

static const FString ClaudeEndpoint = TEXT("https://api.anthropic.com/v1/messages");

FClaudeProvider::FClaudeProvider(const FString& InApiKey, const FString& InModel, const FString& InApiVersion)
	: ApiKey(InApiKey)
	, Model(InModel)
	, ApiVersion(InApiVersion.IsEmpty() ? TEXT("2023-06-01") : InApiVersion)
{
}

void FClaudeProvider::SendRequest(
	const FAICompletionRequest& Request,
	FOnAIResponse OnResponse,
	FOnAIError OnError)
{
	TMap<FString, FString> Headers;
	Headers.Add(TEXT("x-api-key"), ApiKey);
	Headers.Add(TEXT("anthropic-version"), ApiVersion);

	TSharedRef<FJsonObject> Body = MakeShared<FJsonObject>();
	Body->SetStringField(TEXT("model"), Request.Model.IsEmpty() ? Model : Request.Model);
	Body->SetNumberField(TEXT("max_tokens"), Request.MaxTokens);
	Body->SetNumberField(TEXT("temperature"), Request.Temperature);

	// Extract system message
	for (const FAIMessage& Msg : Request.Messages)
	{
		if (Msg.Role == EAIMessageRole::System)
		{
			Body->SetStringField(TEXT("system"), Msg.Content);
			break;
		}
	}

	// Build messages (non-system)
	Body->SetArrayField(TEXT("messages"), BuildMessages(Request.Messages));

	// Tools
	if (Request.Tools.Num() > 0)
	{
		TArray<TSharedPtr<FJsonValue>> ToolsArray;
		for (const auto& Tool : Request.Tools)
		{
			ToolsArray.Add(MakeShared<FJsonValueObject>(Tool));
		}
		Body->SetArrayField(TEXT("tools"), ToolsArray);
	}

	SendHttpRequest(ClaudeEndpoint, Headers, Body, OnResponse, OnError);
}

TSharedPtr<FJsonObject> FClaudeProvider::FormatToolDefinition(
	const FString& Name,
	const FString& Description,
	const TSharedPtr<FJsonObject>& ParamSchema) const
{
	// Claude format:
	// { "name": "...", "description": "...", "input_schema": { JSON Schema } }
	TSharedPtr<FJsonObject> Tool = MakeShared<FJsonObject>();
	Tool->SetStringField(TEXT("name"), Name);
	Tool->SetStringField(TEXT("description"), Description);
	if (ParamSchema.IsValid())
	{
		Tool->SetObjectField(TEXT("input_schema"), ParamSchema);
	}
	return Tool;
}

FAICompletionResponse FClaudeProvider::ParseResponse(const FString& ResponseBody)
{
	FAICompletionResponse Result;

	TSharedPtr<FJsonObject> Json = StringToJson(ResponseBody);
	if (!Json.IsValid())
	{
		Result.ErrorMessage = TEXT("Failed to parse Claude response JSON");
		return Result;
	}

	// Check for API error
	if (Json->HasField(TEXT("error")))
	{
		const TSharedPtr<FJsonObject>* ErrorObj;
		if (Json->TryGetObjectField(TEXT("error"), ErrorObj))
		{
			Result.ErrorMessage = (*ErrorObj)->GetStringField(TEXT("message"));
		}
		else
		{
			Result.ErrorMessage = TEXT("Unknown Claude API error");
		}
		return Result;
	}

	Result.bSuccess = true;

	// Parse usage
	const TSharedPtr<FJsonObject>* Usage;
	if (Json->TryGetObjectField(TEXT("usage"), Usage))
	{
		Result.InputTokens = (*Usage)->GetIntegerField(TEXT("input_tokens"));
		Result.OutputTokens = (*Usage)->GetIntegerField(TEXT("output_tokens"));
	}

	// Parse content blocks
	const TArray<TSharedPtr<FJsonValue>>* ContentArray;
	if (Json->TryGetArrayField(TEXT("content"), ContentArray))
	{
		for (const auto& ContentVal : *ContentArray)
		{
			const TSharedPtr<FJsonObject>& Block = ContentVal->AsObject();
			if (!Block.IsValid()) continue;

			FString Type = Block->GetStringField(TEXT("type"));

			if (Type == TEXT("text"))
			{
				Result.TextContent += Block->GetStringField(TEXT("text"));
			}
			else if (Type == TEXT("tool_use"))
			{
				FAIToolCall ToolCall;
				ToolCall.Id = Block->GetStringField(TEXT("id"));
				ToolCall.Name = Block->GetStringField(TEXT("name"));

				const TSharedPtr<FJsonObject>* InputObj;
				if (Block->TryGetObjectField(TEXT("input"), InputObj))
				{
					ToolCall.Parameters = *InputObj;
				}
				else
				{
					ToolCall.Parameters = MakeShared<FJsonObject>();
				}

				Result.ToolCalls.Add(MoveTemp(ToolCall));
			}
		}
	}

	return Result;
}

TArray<TSharedPtr<FJsonValue>> FClaudeProvider::BuildMessages(const TArray<FAIMessage>& Messages) const
{
	TArray<TSharedPtr<FJsonValue>> Result;

	for (const FAIMessage& Msg : Messages)
	{
		if (Msg.Role == EAIMessageRole::System)
		{
			continue; // System is set at top level in Claude API
		}

		TSharedPtr<FJsonObject> MsgObj = MakeShared<FJsonObject>();

		if (Msg.Role == EAIMessageRole::User)
		{
			MsgObj->SetStringField(TEXT("role"), TEXT("user"));
			MsgObj->SetStringField(TEXT("content"), Msg.Content);
		}
		else if (Msg.Role == EAIMessageRole::Assistant)
		{
			MsgObj->SetStringField(TEXT("role"), TEXT("assistant"));

			// Assistant messages can contain text + tool_use blocks
			TArray<TSharedPtr<FJsonValue>> ContentBlocks;

			if (!Msg.Content.IsEmpty())
			{
				TSharedPtr<FJsonObject> TextBlock = MakeShared<FJsonObject>();
				TextBlock->SetStringField(TEXT("type"), TEXT("text"));
				TextBlock->SetStringField(TEXT("text"), Msg.Content);
				ContentBlocks.Add(MakeShared<FJsonValueObject>(TextBlock));
			}

			for (const FAIToolCall& TC : Msg.ToolCalls)
			{
				TSharedPtr<FJsonObject> ToolBlock = MakeShared<FJsonObject>();
				ToolBlock->SetStringField(TEXT("type"), TEXT("tool_use"));
				ToolBlock->SetStringField(TEXT("id"), TC.Id);
				ToolBlock->SetStringField(TEXT("name"), TC.Name);
				ToolBlock->SetObjectField(TEXT("input"), TC.Parameters.IsValid() ? TC.Parameters : MakeShared<FJsonObject>());
				ContentBlocks.Add(MakeShared<FJsonValueObject>(ToolBlock));
			}

			MsgObj->SetArrayField(TEXT("content"), ContentBlocks);
		}
		else if (Msg.Role == EAIMessageRole::ToolResult)
		{
			MsgObj->SetStringField(TEXT("role"), TEXT("user"));

			// Claude expects tool results as user messages with tool_result content blocks
			TArray<TSharedPtr<FJsonValue>> ContentBlocks;
			TSharedPtr<FJsonObject> ToolResultBlock = MakeShared<FJsonObject>();
			ToolResultBlock->SetStringField(TEXT("type"), TEXT("tool_result"));
			ToolResultBlock->SetStringField(TEXT("tool_use_id"), Msg.ToolCallId);
			ToolResultBlock->SetStringField(TEXT("content"), Msg.Content);
			ContentBlocks.Add(MakeShared<FJsonValueObject>(ToolResultBlock));

			MsgObj->SetArrayField(TEXT("content"), ContentBlocks);
		}

		Result.Add(MakeShared<FJsonValueObject>(MsgObj));
	}

	return Result;
}

FString FClaudeProvider::GetModelsEndpoint() const
{
	return TEXT("https://api.anthropic.com/v1/models");
}

TMap<FString, FString> FClaudeProvider::GetAuthHeaders() const
{
	TMap<FString, FString> Headers;
	Headers.Add(TEXT("x-api-key"), ApiKey);
	Headers.Add(TEXT("anthropic-version"), ApiVersion);
	return Headers;
}

TArray<FModelInfo> FClaudeProvider::ParseModelListResponse(const FString& ResponseBody) const
{
	// Claude returns { "data": [ { "id": "...", "type": "model" }, ... ] }
	return FAIProviderBase::ParseModelListResponse(ResponseBody);
}
