// Copyright (c) 2026 CelestiaDominance. All Rights Reserved.

#include "AI/AIProviderBase.h"
#include "Logging/BlueprintAILogger.h"
#include "Settings/BlueprintAISettings.h"
#include "HttpModule.h"
#include "Interfaces/IHttpResponse.h"
#include "Serialization/JsonSerializer.h"
#include "Serialization/JsonWriter.h"
#include "Serialization/JsonReader.h"

FAIProviderBase::~FAIProviderBase()
{
	CancelRequest();
}

void FAIProviderBase::CancelRequest()
{	if (ActiveRequest.IsValid())
	{
		// Unbind the completion delegate BEFORE cancelling to prevent the
		// cancellation from firing stale HandleError/HandleResponse callbacks.
		// Without this, CancelRequest() queues a Failed completion event
		// that fires on the next HTTP tick and corrupts the busy state.
		ActiveRequest->OnProcessRequestComplete().Unbind();
		ActiveRequest->CancelRequest();
		ActiveRequest.Reset();
	}
}

void FAIProviderBase::FetchAvailableModels(FOnModelsReceived OnSuccess, FOnAIError OnError)
{
	FString ModelsUrl = GetModelsEndpoint();
	if (ModelsUrl.IsEmpty())
	{
		OnError.ExecuteIfBound(TEXT("This provider does not support listing models"));
		return;
	}

	TMap<FString, FString> Headers = GetAuthHeaders();

	TWeakPtr<IAIProvider> WeakSelf = AsShared();
	SendHttpGetRequest(ModelsUrl, Headers,
		[WeakSelf, OnSuccess, OnError](const FString& ResponseBody)
		{
			TSharedPtr<IAIProvider> Self = WeakSelf.Pin();
			if (!Self.IsValid()) return;

			FAIProviderBase* Provider = static_cast<FAIProviderBase*>(Self.Get());
			TArray<FModelInfo> Models = Provider->ParseModelListResponse(ResponseBody);
			OnSuccess.ExecuteIfBound(Models);
		},
		OnError
	);
}

void FAIProviderBase::SendHttpGetRequest(
	const FString& Url,
	const TMap<FString, FString>& Headers,
	TFunction<void(const FString&)> OnSuccess,
	FOnAIError OnError)
{
	TSharedRef<IHttpRequest> Request = FHttpModule::Get().CreateRequest();
	Request->SetURL(Url);
	Request->SetVerb(TEXT("GET"));

	for (const auto& Header : Headers)
	{
		Request->SetHeader(Header.Key, Header.Value);
	}

	Request->OnProcessRequestComplete().BindLambda(
		[OnSuccess, OnError](FHttpRequestPtr Req, FHttpResponsePtr Resp, bool bConnectedSuccessfully)
		{
			if (!bConnectedSuccessfully || !Resp.IsValid())
			{
				OnError.ExecuteIfBound(TEXT("HTTP connection failed fetching models"));
				return;
			}

			const int32 Code = Resp->GetResponseCode();
			const FString ResponseBody = Resp->GetContentAsString();

			if (Code < 200 || Code >= 300)
			{
				// For model listing, a 404 just means the endpoint doesn't
				// exist — return empty rather than showing a scary error.
				if (Code == 404)
				{
					UE_LOG(LogTemp, Log, TEXT("BlueprintAI: Models endpoint returned 404 — not available"));
					OnSuccess(TEXT("{\"data\": []}"));
					return;
				}
				FString ErrorMsg = FString::Printf(TEXT("HTTP %d: %s"), Code, *ResponseBody.Left(500));
				OnError.ExecuteIfBound(ErrorMsg);
				return;
			}

			OnSuccess(ResponseBody);
		});

	Request->ProcessRequest();
}

TArray<FModelInfo> FAIProviderBase::ParseModelListResponse(const FString& ResponseBody) const
{
	// Default parser: handles OpenAI-style { "data": [ { "id": "..." }, ... ] }
	// Also extracts pricing if present (OpenRouter format)
	TArray<FModelInfo> Models;

	TSharedPtr<FJsonObject> Json = StringToJson(ResponseBody);
	if (!Json.IsValid()) return Models;

	int32 SkippedCount = 0;

	const TArray<TSharedPtr<FJsonValue>>* DataArray = nullptr;
	if (Json->TryGetArrayField(TEXT("data"), DataArray) && DataArray)
	{
		for (const auto& Item : *DataArray)
		{
			const TSharedPtr<FJsonObject>* Obj = nullptr;
			if (!Item->TryGetObject(Obj) || !Obj || !(*Obj)->HasField(TEXT("id")))
			{
				continue;
			}

			FString ModelId = (*Obj)->GetStringField(TEXT("id"));

			// Filter out obviously non-chat/non-text models
			{
				FString IdLower = ModelId.ToLower();
				bool bSkip = false;

				// Embedding models
				if (IdLower.Contains(TEXT("embed"))) bSkip = true;
				// Text-to-speech
				else if (IdLower.Contains(TEXT("tts"))) bSkip = true;
				// Image generation
				else if (IdLower.Contains(TEXT("dall-e"))) bSkip = true;
				// Audio transcription/translation
				else if (IdLower.Contains(TEXT("whisper"))) bSkip = true;
				// Content moderation
				else if (IdLower.Contains(TEXT("moderation"))) bSkip = true;
				// Realtime/audio-only models
				else if (IdLower.Contains(TEXT("realtime"))) bSkip = true;
				// Legacy completion-only models
				else if (IdLower.Contains(TEXT("davinci"))) bSkip = true;
				else if (IdLower.Contains(TEXT("babbage"))) bSkip = true;
				else if (IdLower.Contains(TEXT("curie"))) bSkip = true;
				else if (IdLower.Contains(TEXT("ada")) && !IdLower.Contains(TEXT("gpt"))) bSkip = true;

				if (bSkip)
				{
					SkippedCount++;
					continue;
				}
			}

			FModelInfo Info;
			Info.Id = ModelId;
			Info.DisplayName = (*Obj)->HasField(TEXT("name")) ? (*Obj)->GetStringField(TEXT("name")) : Info.Id;

			// Context length
			if ((*Obj)->HasField(TEXT("context_length")))
			{
				Info.ContextLength = (int32)(*Obj)->GetNumberField(TEXT("context_length"));
			}

			// Pricing (OpenRouter format: { "pricing": { "prompt": "0.000003", "completion": "0.000015" } })
			// Prices are per-token as strings. Convert to per-1M-tokens.
			const TSharedPtr<FJsonObject>* PricingObj = nullptr;
			if ((*Obj)->TryGetObjectField(TEXT("pricing"), PricingObj) && PricingObj)
			{
				FString PromptPrice, CompletionPrice;
				if ((*PricingObj)->TryGetStringField(TEXT("prompt"), PromptPrice) &&
					(*PricingObj)->TryGetStringField(TEXT("completion"), CompletionPrice))
				{
					double InputPerToken = FCString::Atod(*PromptPrice);
					double OutputPerToken = FCString::Atod(*CompletionPrice);

					Info.InputPricePer1M = InputPerToken * 1000000.0;
					Info.OutputPricePer1M = OutputPerToken * 1000000.0;

					// Free if both are zero
					Info.bIsFree = (InputPerToken == 0.0 && OutputPerToken == 0.0);
				}
			}

			Models.Add(MoveTemp(Info));
		}
	}

	// Sort: free models first, then by id
	Models.Sort([](const FModelInfo& A, const FModelInfo& B)
	{
		if (A.bIsFree != B.bIsFree) return A.bIsFree;
		return A.Id < B.Id;
	});

	if (SkippedCount > 0)
	{
		UE_LOG(LogTemp, Log, TEXT("BlueprintAI: Found %d models total, showing %d text/chat models (filtered %d non-text models)"),
			Models.Num() + SkippedCount, Models.Num(), SkippedCount);
	}

	return Models;
}

void FAIProviderBase::SendHttpRequest(
	const FString& Url,
	const TMap<FString, FString>& Headers,
	const TSharedRef<FJsonObject>& Body,
	FOnAIResponse OnResponse,
	FOnAIError OnError)
{
	CancelRequest(); // Cancel any previous request

	TSharedRef<IHttpRequest> Request = FHttpModule::Get().CreateRequest();
	Request->SetURL(Url);
	Request->SetVerb(TEXT("POST"));
	Request->SetHeader(TEXT("Content-Type"), TEXT("application/json"));

	// Use configurable timeout — bridges/proxies need more time than direct API
	const float TimeoutSec = static_cast<float>(UBlueprintAISettings::Get()->HttpRequestTimeoutSeconds);
	Request->SetTimeout(TimeoutSec);
	Request->SetActivityTimeout(TimeoutSec); // GPT-5 sends 0 bytes while reasoning

	for (const auto& Header : Headers)
	{
		Request->SetHeader(Header.Key, Header.Value);
	}

	FString PayloadStr = JsonToString(Body);
	UE_LOG(LogTemp, Log, TEXT("BlueprintAI: Sending HTTP POST to %s (%d bytes)"), *Url, PayloadStr.Len());

	// Log the full outgoing HTTP request
	FBlueprintAILogger::Get().Logf(TEXT("HTTP-REQ"), TEXT("POST %s (%d bytes)"), *Url, PayloadStr.Len());
	FBlueprintAILogger::Get().Log(TEXT("HTTP-REQ-BODY"), PayloadStr.Left(10000));
	if (PayloadStr.Len() > 10000)
	{
		FBlueprintAILogger::Get().Logf(TEXT("HTTP-REQ-BODY"), TEXT("... [truncated, %d total bytes]"), PayloadStr.Len());
	}

	Request->SetContentAsString(PayloadStr);

	// Prevent capturing 'this' unsafely — copy fields we need
	TWeakPtr<IAIProvider> WeakSelf = AsShared();

	Request->OnProcessRequestComplete().BindLambda(
		[WeakSelf, OnResponse, OnError, Url, PayloadStr, TimeoutSec](FHttpRequestPtr Req, FHttpResponsePtr Resp, bool bConnectedSuccessfully)
		{
			TSharedPtr<IAIProvider> Self = WeakSelf.Pin();
			if (!Self.IsValid()) return;

			FAIProviderBase* Provider = static_cast<FAIProviderBase*>(Self.Get());

			if (!bConnectedSuccessfully || !Resp.IsValid())
			{
				FString Detail = TEXT("HTTP connection failed");
				if (Req.IsValid())
				{
					Detail += FString::Printf(TEXT(" to %s"), *Req->GetURL());
					Detail += FString::Printf(TEXT(" | Status: %s"), EHttpRequestStatus::ToString(Req->GetStatus()));
				}
				if (Resp.IsValid())
				{
					Detail += FString::Printf(TEXT(" | HTTP %d"), Resp->GetResponseCode());
				}
				FBlueprintAILogger::Get().Log(TEXT("HTTP-ERR"), Detail);

				// Automatic retry on connection failure (once)
				if (!Provider->bIsRetrying)
				{
					Provider->bIsRetrying = true;
					FBlueprintAILogger::Get().Log(TEXT("HTTP-RETRY"), TEXT("Connection failed — retrying once..."));
					UE_LOG(LogTemp, Warning, TEXT("BlueprintAI: Connection failed, retrying once..."));

					TSharedRef<IHttpRequest> RetryReq = FHttpModule::Get().CreateRequest();
					RetryReq->SetURL(Url);
					RetryReq->SetVerb(TEXT("POST"));
					RetryReq->SetHeader(TEXT("Content-Type"), TEXT("application/json"));
					RetryReq->SetTimeout(TimeoutSec);
					RetryReq->SetActivityTimeout(TimeoutSec);
					if (Req.IsValid())
					{
						// Copy auth headers from original request
						for (const FString& Header : Req->GetAllHeaders())
						{
							int32 ColonIdx;
							if (Header.FindChar(TEXT(':'), ColonIdx))
							{
								FString Key = Header.Left(ColonIdx).TrimStartAndEnd();
								FString Val = Header.Mid(ColonIdx + 1).TrimStartAndEnd();
								if (!Key.Equals(TEXT("Content-Type"), ESearchCase::IgnoreCase))
								{
									RetryReq->SetHeader(Key, Val);
								}
							}
						}
					}
					RetryReq->SetContentAsString(PayloadStr);

					RetryReq->OnProcessRequestComplete().BindLambda(
						[WeakSelf, OnResponse, OnError](FHttpRequestPtr Req2, FHttpResponsePtr Resp2, bool bOk2)
						{
							TSharedPtr<IAIProvider> Self2 = WeakSelf.Pin();
							if (!Self2.IsValid()) return;
							FAIProviderBase* Prov2 = static_cast<FAIProviderBase*>(Self2.Get());
							Prov2->bIsRetrying = false;

							if (!bOk2 || !Resp2.IsValid())
							{
								FString Err2 = TEXT("HTTP retry also failed");
								if (Req2.IsValid()) Err2 += FString::Printf(TEXT(" to %s"), *Req2->GetURL());
								FBlueprintAILogger::Get().Log(TEXT("HTTP-ERR"), Err2);
								OnError.ExecuteIfBound(Err2);
								return;
							}
							const int32 Code2 = Resp2->GetResponseCode();
							const FString Body2 = Resp2->GetContentAsString();
							FBlueprintAILogger::Get().Logf(TEXT("HTTP-RETRY"), TEXT("Retry succeeded: HTTP %d (%d bytes)"), Code2, Body2.Len());
							FBlueprintAILogger::Get().Log(TEXT("HTTP-RESP-BODY"), Body2.Left(10000));

							if (Code2 < 200 || Code2 >= 300)
							{
								OnError.ExecuteIfBound(FString::Printf(TEXT("HTTP %d: %s"), Code2, *Body2));
								return;
							}
							FAICompletionResponse Parsed = Prov2->ParseResponse(Body2);
							Parsed.RawJson = Body2;
							if (Parsed.bSuccess) OnResponse.ExecuteIfBound(Parsed);
							else OnError.ExecuteIfBound(Parsed.ErrorMessage);
						});

					Provider->ActiveRequest = RetryReq;
					RetryReq->ProcessRequest();
					return;
				}

				Provider->bIsRetrying = false;
				OnError.ExecuteIfBound(Detail);
				return;
			}

			Provider->bIsRetrying = false;

			const int32 Code = Resp->GetResponseCode();
			const FString ResponseBody = Resp->GetContentAsString();

			FBlueprintAILogger::Get().Logf(TEXT("HTTP-RESP"), TEXT("HTTP %d (%d bytes)"), Code, ResponseBody.Len());
			FBlueprintAILogger::Get().Log(TEXT("HTTP-RESP-BODY"), ResponseBody.Left(10000));
			if (ResponseBody.Len() > 10000)
			{
				FBlueprintAILogger::Get().Logf(TEXT("HTTP-RESP-BODY"), TEXT("... [truncated, %d total bytes]"), ResponseBody.Len());
			}

			if (Code < 200 || Code >= 300)
			{
				FString ErrorMsg = FString::Printf(TEXT("HTTP %d: %s"), Code, *ResponseBody);
				OnError.ExecuteIfBound(ErrorMsg);
				return;
			}

			FAICompletionResponse ParsedResponse = Provider->ParseResponse(ResponseBody);
			ParsedResponse.RawJson = ResponseBody;

			if (ParsedResponse.bSuccess)
			{
				OnResponse.ExecuteIfBound(ParsedResponse);
			}
			else
			{
				OnError.ExecuteIfBound(ParsedResponse.ErrorMessage);
			}
		});

	ActiveRequest = Request;
	Request->ProcessRequest();
}

// ─── JSON Helpers ───────────────────────────────────────────────────────────

FString FAIProviderBase::JsonToString(const TSharedRef<FJsonObject>& Obj)
{
	FString Output;
	TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&Output);
	FJsonSerializer::Serialize(Obj, Writer);
	return Output;
}

TSharedPtr<FJsonObject> FAIProviderBase::StringToJson(const FString& Str)
{
	TSharedPtr<FJsonObject> Obj;
	TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(Str);
	FJsonSerializer::Deserialize(Reader, Obj);
	return Obj;
}

TSharedPtr<FJsonObject> FAIProviderBase::MakeStringProp(const FString& Description)
{
	TSharedPtr<FJsonObject> Prop = MakeShared<FJsonObject>();
	Prop->SetStringField(TEXT("type"), TEXT("string"));
	Prop->SetStringField(TEXT("description"), Description);
	return Prop;
}

TSharedPtr<FJsonObject> FAIProviderBase::MakeIntProp(const FString& Description)
{
	TSharedPtr<FJsonObject> Prop = MakeShared<FJsonObject>();
	Prop->SetStringField(TEXT("type"), TEXT("integer"));
	Prop->SetStringField(TEXT("description"), Description);
	return Prop;
}

TSharedPtr<FJsonObject> FAIProviderBase::MakeBoolProp(const FString& Description)
{
	TSharedPtr<FJsonObject> Prop = MakeShared<FJsonObject>();
	Prop->SetStringField(TEXT("type"), TEXT("boolean"));
	Prop->SetStringField(TEXT("description"), Description);
	return Prop;
}

TSharedPtr<FJsonObject> FAIProviderBase::MakeArrayProp(const FString& Description, TSharedPtr<FJsonObject> ItemsSchema)
{
	TSharedPtr<FJsonObject> Prop = MakeShared<FJsonObject>();
	Prop->SetStringField(TEXT("type"), TEXT("array"));
	Prop->SetStringField(TEXT("description"), Description);
	if (ItemsSchema.IsValid())
	{
		Prop->SetObjectField(TEXT("items"), ItemsSchema);
	}
	return Prop;
}
