<?php

declare(strict_types=1);

require_once dirname(__DIR__) . '/includes/bootstrap.php';

const DEFAULT_SITE_NAME = 'Untitled Imageboard';
const DEFAULT_LEGAL_EMAIL = 'owner@example.com';

$db = appDb();
$route = routePath();
$routeParts = $route === '' ? [] : array_map('rawurldecode', explode('/', $route));
$method = strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET'));

function intInRange($value, int $min, int $max, int $fallback): int
{
    $v = (int)$value;
    if ($v < $min || $v > $max) {
        return $fallback;
    }
    return $v;
}

function queryPage(): int
{
    return max(1, (int)($_GET['page'] ?? 1));
}

function pageOffset(int $page, int $perPage): int
{
    return ($page - 1) * $perPage;
}

function totalPages(int $total, int $perPage): int
{
    return max(1, (int)ceil($total / max(1, $perPage)));
}

function bytesToMb(int $bytes): int
{
    return max(1, (int)ceil($bytes / (1024 * 1024)));
}

function mediaThumbUrl(int $mediaId): string
{
    return '/thumb/' . max(1, $mediaId);
}

function mediaTypeLabel(array $mediaRow): string
{
    $kind = strtolower(trim((string)($mediaRow['media_kind'] ?? '')));
    $mime = strtolower(trim((string)($mediaRow['mime_type'] ?? '')));

    if ($kind === 'video' || str_starts_with($mime, 'video/')) {
        return 'VIDEO';
    }

    if ($mime === 'image/gif') {
        return 'GIF';
    }

    return 'IMAGE';
}

function mediaTypeBadgeClass(array $mediaRow): string
{
    $label = mediaTypeLabel($mediaRow);
    if ($label === 'VIDEO') {
        return 'media-kind-video';
    }
    if ($label === 'GIF') {
        return 'media-kind-gif';
    }
    return 'media-kind-image';
}

function authorLinkFromRow(array $row): string
{
    $userId = (int)($row['user_id'] ?? 0);
    if ($userId <= 0) {
        return '@Anonymous';
    }

    $handle = publicHandleFromRow($row);
    return '<a href="' . e(userProfileUrl($userId)) . '">@' . e($handle) . '</a>';
}

function renderCommentBodyWithMentions(string $body): string
{
    $safe = e($body);
    $highlighted = preg_replace('/(^|\s)@(\w{3,30})/u', '$1<span class="reply-mention">@$2</span>', $safe);
    $linked = preg_replace_callback('/&gt;&gt;(OP|\d+)/i', static function (array $m): string {
        $raw = strtoupper((string)$m[1]) === 'OP' ? 'OP' : (string)((int)$m[1]);
        $target = $raw === 'OP' ? 'post-op' : ('comment-' . $raw);
        return '<a class="reply-mention reply-ref" href="#' . e($target) . '">&gt;&gt;' . e($raw) . '</a>';
    }, (string)$highlighted);
    return nl2br((string)$linked);
}

function renderPagination(string $basePath, array $query, int $page, int $pages): string
{
    if ($pages <= 1) {
        return '';
    }

    $html = '<nav class="pager" aria-label="Pagination">';

    if ($page > 1) {
        $prevQuery = $query;
        $prevQuery['page'] = $page - 1;
        $html .= '<a href="' . e($basePath . '?' . http_build_query($prevQuery)) . '">← Previous</a>';
    }

    $html .= '<span>Page ' . $page . ' of ' . $pages . '</span>';

    if ($page < $pages) {
        $nextQuery = $query;
        $nextQuery['page'] = $page + 1;
        $html .= '<a href="' . e($basePath . '?' . http_build_query($nextQuery)) . '">Next →</a>';
    }

    $html .= '</nav>';
    return $html;
}

function canonicalUrl(string $fallbackPath = '/'): string
{
    $uri = (string)($_SERVER['REQUEST_URI'] ?? $fallbackPath);
    $path = (string)parse_url($uri, PHP_URL_PATH);
    $query = (string)parse_url($uri, PHP_URL_QUERY);

    parse_str($query, $queryArr);
    if (isset($queryArr['page']) && (int)$queryArr['page'] <= 1) {
        unset($queryArr['page']);
    }

    $final = $path !== '' ? $path : $fallbackPath;
    if (!empty($queryArr)) {
        $final .= '?' . http_build_query($queryArr);
    }

    return absoluteUrl($final);
}

function sanitizeFaviconHref(string $value): string
{
    $v = trim($value);
    $isValid = $v !== ''
        && str_starts_with($v, '/')
        && !str_contains($v, '"')
        && !str_contains($v, "'");

    return $isValid ? $v : '';
}

function configuredFaviconHref(): string
{
    $uploaded = trim(settingGet('site_favicon_storage', ''));
    if ($uploaded !== '') {
        return '/site-favicon';
    }

    return sanitizeFaviconHref(settingGet('site_favicon_url', ''));
}

function withVersionQuery(string $path, string $version): string
{
    if ($path === '') {
        return '';
    }

    $sep = str_contains($path, '?') ? '&' : '?';
    return $path . $sep . 'v=' . rawurlencode($version);
}

function cssAssetHref(): string
{
    $path = '/assets/css/style.css';
    $filePath = PUBLIC_PATH . $path;
    $version = is_file($filePath)
        ? (string)filemtime($filePath)
        : settingGet('site_assets_version', '1');

    return withVersionQuery($path, $version);
}

function faviconAssetHref(): string
{
    $base = configuredFaviconHref();
    if ($base === '') {
        return '';
    }

    $stored = trim(settingGet('site_favicon_storage', ''));
    if ($stored !== '') {
        return withVersionQuery($base, $stored);
    }

    $version = settingGet('site_assets_version', '1');
    $filePath = PUBLIC_PATH . $base;
    if (is_file($filePath)) {
        $version = (string)filemtime($filePath);
    }

    return withVersionQuery($base, $version);
}

function setSiteCookie(string $name, string $value, int $maxAgeSeconds): void
{
    setcookie($name, $value, [
        'expires' => time() + max(60, $maxAgeSeconds),
        'path' => '/',
        'secure' => isHttpsRequest(),
        'httponly' => true,
        'samesite' => 'Lax',
    ]);
    $_COOKIE[$name] = $value;
}

function cookieConsentState(): string
{
    $state = trim((string)($_COOKIE['cookie_consent'] ?? ''));
    return in_array($state, ['accepted', 'essential'], true) ? $state : '';
}

function hasAcceptedPostDisclaimer(): bool
{
    return trim((string)($_COOKIE['post_disclaimer_accepted'] ?? '')) === '1';
}

function registrationEnabled(): bool
{
    return settingGet('registration_enabled', '1') === '1';
}

function autoApproveNewPostsEnabled(): bool
{
    return settingGet('auto_approve_new_posts', '1') === '1';
}

function safeLocalPath(string $candidate, string $fallback = '/'): string
{
    $candidate = trim($candidate);
    if ($candidate === '') {
        return $fallback;
    }

    if (str_starts_with($candidate, 'http://') || str_starts_with($candidate, 'https://')) {
        $path = (string)parse_url($candidate, PHP_URL_PATH);
        $query = (string)parse_url($candidate, PHP_URL_QUERY);
        $candidate = $path !== '' ? $path : '/';
        if ($query !== '') {
            $candidate .= '?' . $query;
        }
    }

    if (!str_starts_with($candidate, '/')) {
        return $fallback;
    }

    if (str_contains($candidate, "\r") || str_contains($candidate, "\n")) {
        return $fallback;
    }

    return $candidate;
}

function renderLayout(string $title, string $content, array $options = []): void
{
    $siteName = settingGet('site_name', DEFAULT_SITE_NAME);
    $tagline = settingGet('site_tagline', 'Celestial low-glow community board');
    $flashSuccess = flashGet('success');
    $flashError = flashGet('error');
    $user = currentUser();
    $showAdminLink = adminIpAllowed();

    $isAdminArea = (bool)($options['is_admin'] ?? false);
    $description = trim((string)($options['description'] ?? settingGet('site_meta_description', 'Community imageboard with secure media uploads and moderation.')));
    $canonical = trim((string)($options['canonical'] ?? canonicalUrl()));
    $noIndex = (bool)($options['noindex'] ?? false);
    $bodyClass = trim((string)($options['body_class'] ?? ''));
    $cursorThemeEnabled = settingGet('cursor_theme_enabled', '1') === '1';
    $scripts = $options['scripts'] ?? [];
    if (!is_array($scripts)) {
        $scripts = [];
    }
    $bodyClasses = [];
    if ($bodyClass !== '') {
        $bodyClasses[] = $bodyClass;
    }
    if ($cursorThemeEnabled) {
        $bodyClasses[] = 'cursor-theme-enabled';
    }
    $finalBodyClass = trim(implode(' ', $bodyClasses));
    $faviconHref = faviconAssetHref();
    $brandMode = (string)settingGet('brand_display_mode', 'name_only');
    if (!in_array($brandMode, ['favicon_and_name', 'name_only', 'favicon_only'], true)) {
        $brandMode = 'name_only';
    }

    if (!headers_sent()) {
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');
        header('Expires: 0');
    }

    echo '<!DOCTYPE html>';
    echo '<html lang="en"><head>';
    echo '<meta charset="utf-8">';
    echo '<meta name="viewport" content="width=device-width, initial-scale=1">';
    echo '<title>' . e($title) . ' • ' . e($siteName) . '</title>';
    echo '<meta name="description" content="' . e($description) . '">';
    echo '<meta property="og:site_name" content="' . e($siteName) . '">';
    echo '<meta property="og:title" content="' . e($title . ' • ' . $siteName) . '">';
    echo '<meta property="og:description" content="' . e($description) . '">';
    echo '<meta property="og:type" content="website">';
    if ($canonical !== '') {
        echo '<link rel="canonical" href="' . e($canonical) . '">';
        echo '<meta property="og:url" content="' . e($canonical) . '">';
    }
    if ($noIndex) {
        echo '<meta name="robots" content="noindex,follow">';
    }
    if ($faviconHref !== '') {
        echo '<link rel="icon" href="' . e($faviconHref) . '">';
        echo '<link rel="shortcut icon" href="' . e($faviconHref) . '">';
    }
    echo '<link rel="stylesheet" href="' . e(cssAssetHref()) . '">';
    echo '</head><body' . ($finalBodyClass !== '' ? ' class="' . e($finalBodyClass) . '"' : '') . '>';

    echo '<header class="site-header"><div class="wrap">';
    if (($brandMode === 'favicon_and_name' || $brandMode === 'favicon_only') && $faviconHref !== '') {
        echo '<a class="brand" href="/">';
        echo '<img class="brand-favicon" src="' . e($faviconHref) . '" alt="" aria-hidden="true">';
        if ($brandMode === 'favicon_and_name') {
            echo '<span class="brand-text">' . e($siteName) . '</span>';
        } else {
            echo '<span class="sr-only">' . e($siteName) . '</span>';
        }
        echo '</a>';
    } else {
        echo '<a class="brand" href="/">' . e($siteName) . '</a>';
    }
    echo '<p class="tagline">' . e($tagline) . '</p>';
    echo '<nav class="top-nav">';
    echo '<a href="/">Home</a>';
    echo '<a href="/boards">Boards</a>';
    if ($showAdminLink) {
        echo '<a href="/admin">Admin</a>';
    }
    if ($user) {
        echo '<a class="user-chip" href="' . e(userProfileUrl((int)$user['id'])) . '">@' . e((string)$user['username']) . '</a>';
        echo '<a href="/logout">Logout</a>';
    } else {
        echo '<a href="/login">Login</a>';
        if (registrationEnabled()) {
            echo '<a href="/register">Register</a>';
        }
        if (passwordResetEnabled()) {
            echo '<a href="/forgot-password">Forgot Password</a>';
        }
    }
    echo '</nav>';

    echo '<form class="search-form" method="GET" action="/search">';
    echo '<input type="search" name="q" placeholder="Search posts" value="' . e((string)($_GET['q'] ?? '')) . '">';
    echo '<button type="submit">Search</button>';
    echo '</form>';

    echo '</div></header>';

    echo '<main class="wrap">';
    if ($isAdminArea) {
        $pendingPostsCount = 0;
        if (isAdmin()) {
            $pendingPostsCount = (int)appDb()->query('SELECT COUNT(*) FROM posts WHERE status = "pending"')->fetchColumn();
        }

        echo '<div class="admin-tabs">';
        echo '<a href="/admin">Dashboard</a>';
        echo '<a href="/admin/categories">Categories</a>';
        echo '<a href="/admin/pending-posts">Pending Posts <span class="tab-badge">' . $pendingPostsCount . '</span></a>';
        echo '<a href="/admin/moderation">Moderation</a>';
        echo '<a href="/admin/users">Users</a>';
        echo '<a href="/admin/settings">Settings</a>';
        echo '</div>';
    }

    if ($flashSuccess !== '') {
        echo '<div class="alert alert-success">' . e($flashSuccess) . '</div>';
    }
    if ($flashError !== '') {
        echo '<div class="alert alert-error">' . e($flashError) . '</div>';
    }

    if (cookieConsentState() === '') {
        echo '<aside class="cookie-banner"><h2>Cookies & Privacy</h2>';
        echo '<p>We use essential cookies to keep logins and security features working. You can also allow additional cookies.</p>';
        echo '<div class="cookie-actions">';
        echo '<form method="POST" class="inline-form">';
        echo '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">';
        echo '<input type="hidden" name="website" value="">';
        echo '<input type="hidden" name="action" value="cookie_accept_all">';
        echo '<button type="submit">Accept cookies</button>';
        echo '</form>';
        echo '<form method="POST" class="inline-form">';
        echo '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">';
        echo '<input type="hidden" name="website" value="">';
        echo '<input type="hidden" name="action" value="cookie_essential_only">';
        echo '<button type="submit">Essential only</button>';
        echo '</form>';
        echo '<a href="/cookie-settings">Cookie settings</a>';
        echo '</div></aside>';
    }

    echo $content;
    echo '</main>';

    echo '<footer class="site-footer"><div class="wrap">';
    echo '<nav class="footer-links">';
    echo '<a href="/about">About</a>';
    echo '<a href="/terms">Terms of Service</a>';
    echo '<a href="/privacy">Privacy Policy</a>';
    echo '<a href="/legal-notice">Legal Notice</a>';
    echo '<a href="/dmca">DMCA</a>';
    echo '<a href="/legal">Legal</a>';
    echo '<a href="/cookie-settings">Cookie settings</a>';
    echo '<a href="/contact">Contact</a>';
    echo '<a href="/sitemap.xml">Sitemap</a>';
    echo '</nav>';
    echo '<p>© ' . date('Y') . ' ' . e($siteName) . '. All rights reserved.</p>';
    echo '</div></footer>';

    foreach ($scripts as $script) {
        $src = trim((string)$script);
        if ($src === '' || !str_starts_with($src, '/')) {
            continue;
        }
        $assetVersion = settingGet('site_assets_version', '1');
        $assetPath = PUBLIC_PATH . $src;
        if (is_file($assetPath)) {
            $assetVersion = (string)filemtime($assetPath);
        }
        echo '<script src="' . e(withVersionQuery($src, $assetVersion)) . '"></script>';
    }

    echo '</body></html>';
}

if ($route === 'robots.txt') {
    header('Content-Type: text/plain; charset=utf-8');
    $customRules = trim(settingGet('robots_custom_rules', ''));
    if ($customRules !== '') {
        echo $customRules . "\n";
        if (stripos($customRules, 'Sitemap:') === false) {
            echo 'Sitemap: ' . absoluteUrl('/sitemap.xml') . "\n";
        }
    } else {
        echo "User-agent: *\nAllow: /\nSitemap: " . absoluteUrl('/sitemap.xml') . "\n";
    }
    exit;
}

if ($route === 'sitemap.xml') {
    header('Content-Type: application/xml; charset=utf-8');

    $urls = [
        absoluteUrl('/'),
        absoluteUrl('/search'),
        absoluteUrl('/about'),
        absoluteUrl('/terms'),
        absoluteUrl('/privacy'),
        absoluteUrl('/legal-notice'),
        absoluteUrl('/dmca'),
        absoluteUrl('/legal'),
        absoluteUrl('/contact'),
        absoluteUrl('/cookie-settings'),
    ];

    $boardRows = $db->query('SELECT c.slug AS c_slug, s.slug AS s_slug FROM subcategories s JOIN categories c ON c.id = s.category_id ORDER BY s.id DESC')->fetchAll();
    foreach ($boardRows as $row) {
        $urls[] = absoluteUrl('/board/' . $row['c_slug'] . '/' . $row['s_slug']);
    }

    $postRows = $db->query('SELECT id FROM posts WHERE status = "active" ORDER BY id DESC LIMIT 500')->fetchAll();
    foreach ($postRows as $row) {
        $urls[] = absoluteUrl('/post/' . (int)$row['id']);
    }

    echo "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
    echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
    foreach ($urls as $url) {
        echo '<url><loc>' . e($url) . '</loc></url>';
    }
    echo '</urlset>';
    exit;
}

if ($route === 'site-favicon' || $route === 'favicon.ico') {
    $storedName = trim(settingGet('site_favicon_storage', ''));
    if ($storedName !== '') {
        $fullPath = DATA_PATH . '/uploads/favicons/' . basename($storedName);
        if (is_file($fullPath)) {
            $allowedMime = [
                'image/x-icon',
                'image/vnd.microsoft.icon',
                'image/png',
                'image/webp',
            ];
            $mime = strtolower(trim(settingGet('site_favicon_mime', 'image/x-icon')));
            if (!in_array($mime, $allowedMime, true)) {
                $mime = 'image/x-icon';
            }

            header('Content-Type: ' . $mime);
            header('Content-Length: ' . (string)filesize($fullPath));
            header('Cache-Control: no-cache, must-revalidate, max-age=0');
            header('Pragma: no-cache');
            header('Expires: 0');
            readfile($fullPath);
            exit;
        }
    }

    $fallback = sanitizeFaviconHref(settingGet('site_favicon_url', ''));
    if ($fallback !== '' && $fallback !== '/site-favicon' && $fallback !== '/favicon.ico') {
        redirectTo($fallback);
    }

    render404();
}

if ($method === 'POST') {
    ensureCsrfForPost();

    if (!checkHoneypotField()) {
        flashSet('error', 'Request rejected.');
        redirectTo((string)($_SERVER['HTTP_REFERER'] ?? '/'));
    }

    $action = trim((string)($_POST['action'] ?? ''));

    if ($action === 'cookie_accept_all' || $action === 'cookie_essential_only') {
        $state = $action === 'cookie_accept_all' ? 'accepted' : 'essential';
        setSiteCookie('cookie_consent', $state, 31536000);
        $target = safeLocalPath((string)($_SERVER['HTTP_REFERER'] ?? '/'), '/');
        redirectTo($target);
    }

    if ($action === 'cookie_settings_save') {
        $choice = (string)($_POST['cookie_choice'] ?? 'essential');
        $state = $choice === 'accepted' ? 'accepted' : 'essential';
        setSiteCookie('cookie_consent', $state, 31536000);
        flashSet('success', 'Cookie settings updated.');
        redirectTo('/cookie-settings');
    }

    if ($action === 'accept_post_disclaimer') {
        $target = safeLocalPath((string)($_POST['redirect_path'] ?? '/'), '/');
        setSiteCookie('post_disclaimer_accepted', '1', 31536000);
        redirectTo($target);
    }

    if ($action === 'decline_post_disclaimer') {
        redirectTo('/');
    }

    if ($action === 'register') {
        if (!registrationEnabled()) {
            flashSet('error', 'Registration is currently disabled by admin.');
            redirectTo('/login');
        }

        if (!consumeRateLimit(actorRateKey('register'), 5, 3600)) {
            flashSet('error', 'Too many registration attempts. Try again later.');
            redirectTo('/register');
        }

        [$ok, $message] = doRegister((string)($_POST['username'] ?? ''), (string)($_POST['email'] ?? ''), (string)($_POST['password'] ?? ''));
        flashSet($ok ? 'success' : 'error', $message);
        redirectTo($ok ? '/login' : '/register');
    }

    if ($action === 'login') {
        if (!consumeRateLimit(actorRateKey('login'), 30, 600)) {
            flashSet('error', 'Too many login attempts. Please wait a bit.');
            redirectTo('/login');
        }

        $ok = doLogin((string)($_POST['username'] ?? ''), (string)($_POST['password'] ?? ''), false);
        $reason = loginBlockReason();
        flashSet($ok ? 'success' : 'error', $ok ? 'Welcome back.' : ($reason !== '' ? $reason : 'Invalid credentials or too many attempts.'));
        redirectTo($ok ? '/' : '/login');
    }

    if ($action === 'admin_login') {
        requireAdminIpOr404();

        if (!consumeRateLimit(actorRateKey('admin_login'), 30, 600)) {
            flashSet('error', 'Too many admin login attempts. Please wait.');
            redirectTo('/admin');
        }

        $ok = doLogin((string)($_POST['username'] ?? ''), (string)($_POST['password'] ?? ''), true);
        $reason = loginBlockReason();
        flashSet($ok ? 'success' : 'error', $ok ? 'Admin login successful.' : ($reason !== '' ? $reason : 'Invalid admin credentials.'));
        redirectTo('/admin');
    }

    if ($action === 'forgot_password') {
        if (!passwordResetEnabled()) {
            flashSet('error', 'Password reset is not enabled by admin.');
            redirectTo('/forgot-password');
        }

        if (!consumeRateLimit(actorRateKey('forgot_password'), 6, 3600)) {
            flashSet('error', 'Too many reset requests. Try later.');
            redirectTo('/forgot-password');
        }

        $email = trim((string)($_POST['email'] ?? ''));
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            flashSet('error', 'Enter a valid email.');
            redirectTo('/forgot-password');
        }

        sendPasswordResetLink($email);
        flashSet('success', 'If the email exists, a reset link has been sent.');
        redirectTo('/forgot-password');
    }

    if ($action === 'reset_password') {
        if (!passwordResetEnabled()) {
            flashSet('error', 'Password reset is not enabled by admin.');
            redirectTo('/login');
        }

        $token = trim((string)($_POST['token'] ?? ''));
        $password = (string)($_POST['password'] ?? '');
        $confirm = (string)($_POST['password_confirm'] ?? '');

        if ($password !== $confirm) {
            flashSet('error', 'Passwords do not match.');
            redirectTo('/reset-password?token=' . rawurlencode($token));
        }

        if (!resetPasswordFromToken($token, $password)) {
            flashSet('error', 'Invalid or expired token, or weak password (min 10 chars).');
            redirectTo('/reset-password?token=' . rawurlencode($token));
        }

        flashSet('success', 'Password updated. You can now login.');
        redirectTo('/login');
    }

    if ($action === 'create_post') {
        requireAuth();

        $subcategoryId = (int)($_POST['subcategory_id'] ?? 0);
        $title = trim((string)($_POST['title'] ?? ''));
        $body = trim((string)($_POST['body'] ?? ''));

        if ($subcategoryId <= 0 || strlen($title) < 3) {
            flashSet('error', 'Title and board are required.');
            redirectTo((string)($_SERVER['HTTP_REFERER'] ?? '/'));
        }

        $boardStmt = $db->prepare('SELECT sc.id, sc.slug AS sub_slug, c.slug AS cat_slug
            FROM subcategories sc
            JOIN categories c ON c.id = sc.category_id
            WHERE sc.id = ?
            LIMIT 1');
        $boardStmt->execute([$subcategoryId]);
        $boardRow = $boardStmt->fetch();
        if (!$boardRow) {
            flashSet('error', 'Invalid board.');
            redirectTo('/');
        }

        $rule = boardRuleForSubcategory($subcategoryId);
        $userId = (int)currentUser()['id'];

        if (captchaRequiredForBoardRule($rule, $userId)) {
            $captchaAnswer = trim((string)($_POST['captcha_answer'] ?? ''));
            if (!verifyCaptchaForBoard($subcategoryId, $userId, $captchaAnswer)) {
                flashSet('error', 'CAPTCHA failed. Please solve the challenge again.');
                redirectTo((string)($_SERVER['HTTP_REFERER'] ?? '/'));
            }
        }

        if (!consumeRateLimit(actorRateKey('post-global', $userId), intInRange(settingGet('anti_spam_posts_per_10m', '8'), 1, 100, 8), 600)) {
            flashSet('error', 'Posting too fast. Slow down.');
            redirectTo((string)($_SERVER['HTTP_REFERER'] ?? '/'));
        }

        if (!consumeRateLimit(actorRateKey('post-board-' . $subcategoryId, $userId), $rule['max_posts_per_hour'], 3600)) {
            flashSet('error', 'Board posting limit reached for this hour.');
            redirectTo((string)($_SERVER['HTTP_REFERER'] ?? '/'));
        }

        $lastStmt = $db->prepare('SELECT created_at FROM posts WHERE user_id = ? AND subcategory_id = ? ORDER BY id DESC LIMIT 1');
        $lastStmt->execute([$userId, $subcategoryId]);
        $lastCreatedAt = $lastStmt->fetchColumn();
        if ($lastCreatedAt !== false) {
            $age = time() - strtotime((string)$lastCreatedAt);
            if ($age < $rule['cooldown_seconds']) {
                $remaining = $rule['cooldown_seconds'] - $age;
                flashSet('error', 'Please wait ' . $remaining . 's before posting again in this board.');
                redirectTo((string)($_SERVER['HTTP_REFERER'] ?? '/'));
            }
        }

        $uploadedCount = countUploadedFiles('media');
        if ($uploadedCount > $rule['max_files']) {
            flashSet('error', 'Max files for this board: ' . $rule['max_files']);
            redirectTo((string)($_SERVER['HTTP_REFERER'] ?? '/'));
        }

        $postStatus = autoApproveNewPostsEnabled() ? 'active' : 'pending';
        $now = dbNow();
        $stmt = $db->prepare('INSERT INTO posts (subcategory_id, user_id, title, body, status, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?)');
        $stmt->execute([$subcategoryId, $userId, mb_substr($title, 0, 150), mb_substr($body, 0, 8000), $postStatus, $now, $now]);
        $postId = (int)$db->lastInsertId();

        [$savedMedia, $uploadErrors] = storeMediaFromRequest($postId, $userId, 'media', [
            'max_files' => $rule['max_files'],
            'max_file_bytes' => $rule['max_file_bytes'],
            'allow_video' => $rule['allow_video'],
        ]);

        if ($body === '' && empty($savedMedia)) {
            $db->prepare('DELETE FROM posts WHERE id = ?')->execute([$postId]);
            flashSet('error', 'Post needs text or at least one valid image/video.');
            redirectTo((string)($_SERVER['HTTP_REFERER'] ?? '/'));
        }

        if (!empty($uploadErrors)) {
            flashSet('error', implode(' | ', $uploadErrors));
        }

        if ($postStatus === 'pending') {
            flashSet('success', 'Post submitted successfully and is pending admin approval.');
        } elseif (empty($uploadErrors)) {
            flashSet('success', 'Post created.');
        }

        if ($postStatus === 'active') {
            triggerSitemapPing('post_created');
            redirectTo('/post/' . $postId);
        }

        $pendingRedirect = '/board/' . (string)$boardRow['cat_slug'] . '/' . (string)$boardRow['sub_slug'];
        if ($postStatus === 'pending') {
            $pendingRedirect .= '?post_pending=1';
        }
        redirectTo($pendingRedirect);
    }

    if ($action === 'add_comment') {
        requireAuth();

        $postId = (int)($_POST['post_id'] ?? 0);
        $body = trim((string)($_POST['body'] ?? ''));

        if ($postId <= 0) {
            flashSet('error', 'Invalid comment target.');
            redirectTo((string)($_SERVER['HTTP_REFERER'] ?? '/'));
        }

        $postStmt = $db->prepare('SELECT p.id, p.subcategory_id FROM posts p WHERE p.id = ? AND p.status = "active" LIMIT 1');
        $postStmt->execute([$postId]);
        $postRow = $postStmt->fetch();
        if (!$postRow) {
            flashSet('error', 'Post not found.');
            redirectTo('/');
        }

        $rule = boardRuleForSubcategory((int)$postRow['subcategory_id']);
        $userId = (int)currentUser()['id'];

        if (captchaRequiredForBoardRule($rule, $userId)) {
            $captchaAnswer = trim((string)($_POST['captcha_answer'] ?? ''));
            if (!verifyCaptchaForBoard((int)$postRow['subcategory_id'], $userId, $captchaAnswer)) {
                flashSet('error', 'CAPTCHA failed. Please solve the challenge again.');
                redirectTo('/post/' . $postId);
            }
        }

        if (!consumeRateLimit(actorRateKey('comment-global', $userId), intInRange(settingGet('anti_spam_comments_per_10m', '20'), 1, 200, 20), 600)) {
            flashSet('error', 'Commenting too fast. Slow down.');
            redirectTo('/post/' . $postId);
        }

        if (!consumeRateLimit(actorRateKey('comment-board-' . (int)$postRow['subcategory_id'], $userId), $rule['max_comments_per_hour'], 3600)) {
            flashSet('error', 'Board comment limit reached for this hour.');
            redirectTo('/post/' . $postId);
        }

        $now = dbNow();
        $stmt = $db->prepare('INSERT INTO comments (post_id, user_id, body, status, created_at) VALUES (?, ?, ?, "active", ?)');
        $stmt->execute([$postId, $userId, mb_substr($body, 0, 3000), $now]);
        $commentId = (int)$db->lastInsertId();

        [$savedMedia, $uploadErrors] = storeMediaFromRequest($postId, $userId, 'comment_media', [
            'max_files' => $rule['max_files'],
            'max_file_bytes' => $rule['max_file_bytes'],
            'allow_video' => $rule['allow_video'],
            'comment_id' => $commentId,
        ]);

        if ($body === '' && empty($savedMedia)) {
            $db->prepare('DELETE FROM comments WHERE id = ?')->execute([$commentId]);
            flashSet('error', 'Comment needs text or at least one valid media file.');
            redirectTo('/post/' . $postId);
        }

        if (!empty($uploadErrors)) {
            flashSet('error', implode(' | ', $uploadErrors));
        } else {
            flashSet('success', 'Comment added.');
        }
        triggerSitemapPing('comment_added');
        redirectTo('/post/' . $postId);
    }

    if ($action === 'update_profile') {
        requireAuth();

        $userId = (int)currentUser()['id'];
        $displayName = trim((string)($_POST['display_name'] ?? ''));

        if ($displayName !== '' && !preg_match('/^\w{3,30}$/', $displayName)) {
            flashSet('error', 'Visible username must be 3-30 chars: letters, numbers, underscore.');
            redirectTo(userProfileUrl($userId));
        }

        if ($displayName !== '') {
            $dupe = $db->prepare('SELECT id FROM users WHERE lower(display_name) = lower(?) AND id <> ? LIMIT 1');
            $dupe->execute([$displayName, $userId]);
            if ($dupe->fetch()) {
                flashSet('error', 'That visible username is already used.');
                redirectTo(userProfileUrl($userId));
            }
        }

        $db->prepare('UPDATE users SET display_name = ? WHERE id = ?')->execute([$displayName, $userId]);
        flashSet('success', $displayName === '' ? 'Visible username cleared. You now appear as @Anonymous.' : 'Visible username updated.');
        redirectTo(userProfileUrl($userId));
    }

    if ($action === 'contact_send') {
        if (!consumeRateLimit(actorRateKey('contact'), intInRange(settingGet('anti_spam_contact_per_hour', '5'), 1, 100, 5), 3600)) {
            flashSet('error', 'Too many contact requests. Please try later.');
            redirectTo('/contact');
        }

        $name = trim((string)($_POST['name'] ?? ''));
        $email = trim((string)($_POST['email'] ?? ''));
        $message = trim((string)($_POST['message'] ?? ''));

        if ($name === '' || !filter_var($email, FILTER_VALIDATE_EMAIL) || $message === '') {
            flashSet('error', 'Please provide valid contact details.');
            redirectTo('/contact');
        }

        $stmt = $db->prepare('INSERT INTO contact_messages (name, email, message, ip_address, created_at) VALUES (?, ?, ?, ?, ?)');
        $stmt->execute([$name, $email, mb_substr($message, 0, 5000), clientIp(), dbNow()]);

        if (settingGet('contact_forward_enabled', '0') === '1' && smtpIsConfigured()) {
            $target = settingGet('legal_email', DEFAULT_LEGAL_EMAIL);
            $subject = '[Contact Form] ' . mb_substr($name, 0, 100);
            $body = "Name: {$name}\nEmail: {$email}\nIP: " . clientIp() . "\n\nMessage:\n{$message}";
            smtpSendTextMail($target, $subject, $body);
        }

        flashSet('success', 'Message sent.');
        redirectTo('/contact');
    }

    if (str_starts_with($action, 'admin_')) {
        requireAdminAuth();
        $adminId = (int)currentUser()['id'];

        if ($action === 'admin_create_category') {
            $name = trim((string)($_POST['name'] ?? ''));
            $description = trim((string)($_POST['description'] ?? ''));
            if ($name === '') {
                flashSet('error', 'Category name required.');
                redirectTo('/admin/categories');
            }

            $slug = uniqueSlug($db, 'categories', (string)($_POST['slug'] ?? $name));
            $stmt = $db->prepare('INSERT INTO categories (name, slug, description, sort_order, created_at) VALUES (?, ?, ?, ?, ?)');
            $stmt->execute([$name, $slug, $description, (int)($_POST['sort_order'] ?? 0), dbNow()]);
            logAudit($adminId, 'create_category', $name . ' [' . $slug . ']');
            flashSet('success', 'Category created.');
            triggerSitemapPing('category_created');
            redirectTo('/admin/categories');
        }

        if ($action === 'admin_create_subcategory') {
            $categoryId = (int)($_POST['category_id'] ?? 0);
            $name = trim((string)($_POST['name'] ?? ''));
            $description = trim((string)($_POST['description'] ?? ''));
            if ($categoryId <= 0 || $name === '') {
                flashSet('error', 'Subcategory name and parent category required.');
                redirectTo('/admin/categories');
            }

            $slug = uniqueSlug($db, 'subcategories', (string)($_POST['slug'] ?? $name), $categoryId);
            $stmt = $db->prepare('INSERT INTO subcategories (category_id, name, slug, description, sort_order, created_at) VALUES (?, ?, ?, ?, ?, ?)');
            $stmt->execute([$categoryId, $name, $slug, $description, (int)($_POST['sort_order'] ?? 0), dbNow()]);
            logAudit($adminId, 'create_subcategory', $name . ' [' . $slug . ']');
            flashSet('success', 'Subcategory created.');
            triggerSitemapPing('subcategory_created');
            redirectTo('/admin/categories');
        }

        if ($action === 'admin_rename_category') {
            $categoryId = (int)($_POST['category_id'] ?? 0);
            $name = trim((string)($_POST['name'] ?? ''));
            $description = trim((string)($_POST['description'] ?? ''));
            if ($categoryId <= 0 || $name === '') {
                flashSet('error', 'Valid category and name required.');
                redirectTo('/admin/categories');
            }

            $catStmt = $db->prepare('SELECT name, slug, description FROM categories WHERE id = ? LIMIT 1');
            $catStmt->execute([$categoryId]);
            $cat = $catStmt->fetch();
            if (!$cat) {
                flashSet('error', 'Category not found.');
                redirectTo('/admin/categories');
            }

            $baseSlug = slugify($name);
            $candidateSlug = $baseSlug;
            $counter = 1;
            while (true) {
                $slugCheckStmt = $db->prepare('SELECT 1 FROM categories WHERE slug = ? AND id <> ? LIMIT 1');
                $slugCheckStmt->execute([$candidateSlug, $categoryId]);
                if (!$slugCheckStmt->fetchColumn()) {
                    $newSlug = $candidateSlug;
                    break;
                }
                $counter++;
                $candidateSlug = $baseSlug . '-' . $counter;
            }

            $db->prepare('UPDATE categories SET name = ?, description = ?, slug = ? WHERE id = ?')->execute([$name, $description, $newSlug, $categoryId]);

            $slugChanged = $newSlug !== (string)$cat['slug'];
            logAudit($adminId, 'rename_category', 'id=' . $categoryId . ' slug_from=' . (string)$cat['slug'] . ' slug_to=' . $newSlug . ' name_from=' . (string)$cat['name'] . ' name_to=' . $name);
            flashSet('success', $slugChanged ? 'Category updated and slug auto-updated.' : 'Category updated.');
            triggerSitemapPing($slugChanged ? 'category_slug_updated' : 'category_updated');
            redirectTo('/admin/categories');
        }

        if ($action === 'admin_rename_subcategory') {
            $subId = (int)($_POST['subcategory_id'] ?? 0);
            $name = trim((string)($_POST['name'] ?? ''));
            $description = trim((string)($_POST['description'] ?? ''));
            if ($subId <= 0 || $name === '') {
                flashSet('error', 'Valid subcategory and name required.');
                redirectTo('/admin/categories');
            }

            $subStmt = $db->prepare('SELECT category_id, name, slug, description FROM subcategories WHERE id = ? LIMIT 1');
            $subStmt->execute([$subId]);
            $sub = $subStmt->fetch();
            if (!$sub) {
                flashSet('error', 'Subcategory not found.');
                redirectTo('/admin/categories');
            }

            $baseSlug = slugify($name);
            $candidateSlug = $baseSlug;
            $counter = 1;
            while (true) {
                $slugCheckStmt = $db->prepare('SELECT 1 FROM subcategories WHERE category_id = ? AND slug = ? AND id <> ? LIMIT 1');
                $slugCheckStmt->execute([(int)$sub['category_id'], $candidateSlug, $subId]);
                if (!$slugCheckStmt->fetchColumn()) {
                    $newSlug = $candidateSlug;
                    break;
                }
                $counter++;
                $candidateSlug = $baseSlug . '-' . $counter;
            }

            $db->prepare('UPDATE subcategories SET name = ?, description = ?, slug = ? WHERE id = ?')->execute([$name, $description, $newSlug, $subId]);

            $slugChanged = $newSlug !== (string)$sub['slug'];
            logAudit($adminId, 'rename_subcategory', 'id=' . $subId . ' category_id=' . (int)$sub['category_id'] . ' slug_from=' . (string)$sub['slug'] . ' slug_to=' . $newSlug . ' name_from=' . (string)$sub['name'] . ' name_to=' . $name);
            flashSet('success', $slugChanged ? 'Subcategory updated and slug auto-updated.' : 'Subcategory updated.');
            triggerSitemapPing($slugChanged ? 'subcategory_slug_updated' : 'subcategory_updated');
            redirectTo('/admin/categories');
        }

        if ($action === 'admin_resync_category_slugs') {
            $categoriesToSync = $db->query('SELECT id, name, slug FROM categories ORDER BY id ASC')->fetchAll();
            $updatedCount = 0;

            foreach ($categoriesToSync as $categoryToSync) {
                $categoryId = (int)($categoryToSync['id'] ?? 0);
                $name = trim((string)($categoryToSync['name'] ?? ''));
                $oldSlug = (string)($categoryToSync['slug'] ?? '');
                if ($categoryId <= 0 || $name === '') {
                    continue;
                }

                $baseSlug = slugify($name);
                $candidateSlug = $baseSlug;
                $counter = 1;
                while (true) {
                    $slugCheckStmt = $db->prepare('SELECT 1 FROM categories WHERE slug = ? AND id <> ? LIMIT 1');
                    $slugCheckStmt->execute([$candidateSlug, $categoryId]);
                    if (!$slugCheckStmt->fetchColumn()) {
                        break;
                    }
                    $counter++;
                    $candidateSlug = $baseSlug . '-' . $counter;
                }

                if ($candidateSlug === $oldSlug) {
                    continue;
                }

                $db->prepare('UPDATE categories SET slug = ? WHERE id = ?')->execute([$candidateSlug, $categoryId]);
                $updatedCount++;
            }

            $totalCount = count($categoriesToSync);
            logAudit($adminId, 'resync_category_slugs', 'updated=' . $updatedCount . ' total=' . $totalCount);
            flashSet('success', 'Category slug resync complete. Updated ' . $updatedCount . ' of ' . $totalCount . ' categories.');
            if ($updatedCount > 0) {
                triggerSitemapPing('category_slug_resync');
            }
            redirectTo('/admin/categories');
        }

        if ($action === 'admin_resync_subcategory_slugs') {
            $subsToSync = $db->query('SELECT id, category_id, name, slug FROM subcategories ORDER BY id ASC')->fetchAll();
            $updatedCount = 0;

            foreach ($subsToSync as $subToSync) {
                $subId = (int)($subToSync['id'] ?? 0);
                $categoryId = (int)($subToSync['category_id'] ?? 0);
                $name = trim((string)($subToSync['name'] ?? ''));
                $oldSlug = (string)($subToSync['slug'] ?? '');
                if ($subId <= 0 || $categoryId <= 0 || $name === '') {
                    continue;
                }

                $baseSlug = slugify($name);
                $candidateSlug = $baseSlug;
                $counter = 1;
                while (true) {
                    $slugCheckStmt = $db->prepare('SELECT 1 FROM subcategories WHERE category_id = ? AND slug = ? AND id <> ? LIMIT 1');
                    $slugCheckStmt->execute([$categoryId, $candidateSlug, $subId]);
                    if (!$slugCheckStmt->fetchColumn()) {
                        break;
                    }
                    $counter++;
                    $candidateSlug = $baseSlug . '-' . $counter;
                }

                if ($candidateSlug === $oldSlug) {
                    continue;
                }

                $db->prepare('UPDATE subcategories SET slug = ? WHERE id = ?')->execute([$candidateSlug, $subId]);
                $updatedCount++;
            }

            $totalCount = count($subsToSync);
            logAudit($adminId, 'resync_subcategory_slugs', 'updated=' . $updatedCount . ' total=' . $totalCount);
            flashSet('success', 'Subcategory slug resync complete. Updated ' . $updatedCount . ' of ' . $totalCount . ' subcategories.');
            if ($updatedCount > 0) {
                triggerSitemapPing('subcategory_slug_resync');
            }
            redirectTo('/admin/categories');
        }

        if ($action === 'admin_delete_category') {
            $categoryId = (int)($_POST['category_id'] ?? 0);
            if ($categoryId <= 0) {
                flashSet('error', 'Invalid category.');
                redirectTo('/admin/categories');
            }

            $catStmt = $db->prepare('SELECT name, slug FROM categories WHERE id = ? LIMIT 1');
            $catStmt->execute([$categoryId]);
            $cat = $catStmt->fetch();
            if (!$cat) {
                flashSet('error', 'Category not found.');
                redirectTo('/admin/categories');
            }

            $db->prepare('DELETE FROM categories WHERE id = ?')->execute([$categoryId]);
            logAudit($adminId, 'delete_category', (string)$cat['name'] . ' [' . (string)$cat['slug'] . ']');
            flashSet('success', 'Category deleted (with all child boards/content).');
            triggerSitemapPing('category_deleted');
            redirectTo('/admin/categories');
        }

        if ($action === 'admin_delete_subcategory') {
            $subId = (int)($_POST['subcategory_id'] ?? 0);
            if ($subId <= 0) {
                flashSet('error', 'Invalid subcategory.');
                redirectTo('/admin/categories');
            }

            $subStmt = $db->prepare('SELECT name, slug FROM subcategories WHERE id = ? LIMIT 1');
            $subStmt->execute([$subId]);
            $sub = $subStmt->fetch();
            if (!$sub) {
                flashSet('error', 'Subcategory not found.');
                redirectTo('/admin/categories');
            }

            try {
                $db->beginTransaction();

                $postCountStmt = $db->prepare('SELECT COUNT(*) FROM posts WHERE subcategory_id = ?');
                $postCountStmt->execute([$subId]);
                $postCount = (int)$postCountStmt->fetchColumn();

                $commentCountStmt = $db->prepare('SELECT COUNT(*) FROM comments WHERE post_id IN (SELECT id FROM posts WHERE subcategory_id = ?)');
                $commentCountStmt->execute([$subId]);
                $commentCount = (int)$commentCountStmt->fetchColumn();

                $mediaCountStmt = $db->prepare('SELECT COUNT(*) FROM media WHERE post_id IN (SELECT id FROM posts WHERE subcategory_id = ?)');
                $mediaCountStmt->execute([$subId]);
                $mediaCount = (int)$mediaCountStmt->fetchColumn();

                $db->prepare('DELETE FROM board_rules WHERE subcategory_id = ?')->execute([$subId]);
                $db->prepare('DELETE FROM comments WHERE post_id IN (SELECT id FROM posts WHERE subcategory_id = ?)')->execute([$subId]);
                $db->prepare('DELETE FROM media WHERE post_id IN (SELECT id FROM posts WHERE subcategory_id = ?)')->execute([$subId]);
                $db->prepare('DELETE FROM posts WHERE subcategory_id = ?')->execute([$subId]);

                $deleteSubStmt = $db->prepare('DELETE FROM subcategories WHERE id = ?');
                $deleteSubStmt->execute([$subId]);
                if ($deleteSubStmt->rowCount() < 1) {
                    $db->rollBack();
                    flashSet('error', 'Subcategory could not be deleted. Please refresh and try again.');
                    redirectTo('/admin/categories');
                }

                $db->commit();
            } catch (Throwable $exception) {
                if ($db->inTransaction()) {
                    $db->rollBack();
                }
                flashSet('error', 'Could not delete subcategory right now. Please try again.');
                redirectTo('/admin/categories');
            }

            logAudit($adminId, 'delete_subcategory', (string)$sub['name'] . ' [' . (string)$sub['slug'] . '] posts=' . $postCount . ' comments=' . $commentCount . ' media=' . $mediaCount);
            flashSet('success', 'Subcategory deleted with ' . $postCount . ' thread(s), ' . $commentCount . ' comment(s), and ' . $mediaCount . ' media file(s).');
            triggerSitemapPing('subcategory_deleted');
            redirectTo('/admin/categories');
        }

        if ($action === 'admin_save_board_rule') {
            $subId = (int)($_POST['subcategory_id'] ?? 0);
            if ($subId <= 0) {
                flashSet('error', 'Invalid board rule target.');
                redirectTo('/admin/categories');
            }

            $maxFiles = intInRange($_POST['max_files'] ?? 10, 1, 20, 10);
            $maxFileMb = intInRange($_POST['max_file_mb'] ?? 20, 1, 50, 20);
            $maxFileBytes = $maxFileMb * 1024 * 1024;
            $cooldown = intInRange($_POST['cooldown_seconds'] ?? 10, 0, 3600, 10);
            $maxPostsHour = intInRange($_POST['max_posts_per_hour'] ?? 200, 1, 500, 200);
            $maxCommentsHour = intInRange($_POST['max_comments_per_hour'] ?? 500, 1, 1000, 500);
            $allowVideo = isset($_POST['allow_video']) ? 1 : 0;
            $captchaMode = (string)($_POST['captcha_mode'] ?? 'low_trust');
            if (!in_array($captchaMode, ['off', 'low_trust', 'all'], true)) {
                $captchaMode = 'low_trust';
            }

            $stmt = $db->prepare('INSERT INTO board_rules (subcategory_id, max_files, max_file_bytes, cooldown_seconds, max_posts_per_hour, max_comments_per_hour, allow_video, captcha_mode, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?) ON CONFLICT(subcategory_id) DO UPDATE SET max_files = excluded.max_files, max_file_bytes = excluded.max_file_bytes, cooldown_seconds = excluded.cooldown_seconds, max_posts_per_hour = excluded.max_posts_per_hour, max_comments_per_hour = excluded.max_comments_per_hour, allow_video = excluded.allow_video, captcha_mode = excluded.captcha_mode, updated_at = excluded.updated_at');
            $stmt->execute([$subId, $maxFiles, $maxFileBytes, $cooldown, $maxPostsHour, $maxCommentsHour, $allowVideo, $captchaMode, dbNow()]);
            logAudit($adminId, 'save_board_rule', 'subcategory_id=' . $subId);
            flashSet('success', 'Board rule saved.');
            redirectTo('/admin/categories');
        }

        if ($action === 'admin_delete_post') {
            $postId = (int)($_POST['post_id'] ?? 0);
            if ($postId > 0) {
                $db->prepare('UPDATE posts SET status = "deleted", updated_at = ? WHERE id = ?')->execute([dbNow(), $postId]);
                logAudit($adminId, 'delete_post', 'post_id=' . $postId);
                triggerSitemapPing('post_deleted');
            }
            flashSet('success', 'Post removed.');
            redirectTo('/admin/moderation');
        }

        if ($action === 'admin_approve_post') {
            $postId = (int)($_POST['post_id'] ?? 0);
            if ($postId > 0) {
                $stmt = $db->prepare('UPDATE posts SET status = "active", updated_at = ? WHERE id = ? AND status = "pending"');
                $stmt->execute([dbNow(), $postId]);
                if ($stmt->rowCount() > 0) {
                    logAudit($adminId, 'approve_post', 'post_id=' . $postId);
                    triggerSitemapPing('post_approved');
                    flashSet('success', 'Pending post approved.');
                } else {
                    flashSet('error', 'Pending post not found or already processed.');
                }
            }
            redirectTo('/admin/pending-posts');
        }

        if ($action === 'admin_decline_post') {
            $postId = (int)($_POST['post_id'] ?? 0);
            if ($postId > 0) {
                $stmt = $db->prepare('UPDATE posts SET status = "deleted", updated_at = ? WHERE id = ? AND status = "pending"');
                $stmt->execute([dbNow(), $postId]);
                if ($stmt->rowCount() > 0) {
                    logAudit($adminId, 'decline_post', 'post_id=' . $postId);
                    flashSet('success', 'Pending post declined.');
                } else {
                    flashSet('error', 'Pending post not found or already processed.');
                }
            }
            redirectTo('/admin/pending-posts');
        }

        if ($action === 'admin_delete_comment') {
            $commentId = (int)($_POST['comment_id'] ?? 0);
            if ($commentId > 0) {
                $db->prepare('UPDATE comments SET status = "deleted" WHERE id = ?')->execute([$commentId]);
                logAudit($adminId, 'delete_comment', 'comment_id=' . $commentId);
                triggerSitemapPing('comment_deleted');
            }
            flashSet('success', 'Comment removed.');
            redirectTo('/admin/moderation');
        }

        if ($action === 'admin_delete_media') {
            $mediaId = (int)($_POST['media_id'] ?? 0);
            $stmt = $db->prepare('SELECT * FROM media WHERE id = ? LIMIT 1');
            $stmt->execute([$mediaId]);
            $media = $stmt->fetch();
            if ($media) {
                $path = mediaStoragePath($media);
                if (is_file($path)) {
                    @unlink($path);
                }
                $db->prepare('DELETE FROM media WHERE id = ?')->execute([$mediaId]);
                logAudit($adminId, 'delete_media', 'media_id=' . $mediaId);
                triggerSitemapPing('media_deleted');
            }
            flashSet('success', 'Media removed.');
            redirectTo('/admin/moderation');
        }

        if ($action === 'admin_ban_user' || $action === 'admin_unban_user') {
            $userId = (int)($_POST['user_id'] ?? 0);
            $target = $db->prepare('SELECT id, role FROM users WHERE id = ? LIMIT 1');
            $target->execute([$userId]);
            $row = $target->fetch();

            if ($row && $row['role'] !== 'admin') {
                $banValue = $action === 'admin_ban_user' ? 1 : 0;
                $db->prepare('UPDATE users SET is_banned = ? WHERE id = ?')->execute([$banValue, $userId]);
                logAudit($adminId, $action, 'user_id=' . $userId);
                flashSet('success', $banValue === 1 ? 'User banned.' : 'User unbanned.');
            } else {
                flashSet('error', 'Cannot modify another admin account here.');
            }
            redirectTo('/admin/users');
        }

        if ($action === 'admin_save_settings') {
            $warnings = [];
            $assetsVersion = (string)time();

            settingSet('site_name', trim((string)($_POST['site_name'] ?? DEFAULT_SITE_NAME)));
            settingSet('site_tagline', trim((string)($_POST['site_tagline'] ?? '')));
            settingSet('legal_email', trim((string)($_POST['legal_email'] ?? DEFAULT_LEGAL_EMAIL)));
            settingSet('site_meta_description', trim((string)($_POST['site_meta_description'] ?? '')));
            settingSet('cursor_theme_enabled', isset($_POST['cursor_theme_enabled']) ? '1' : '0');
            settingSet('site_favicon_url', sanitizeFaviconHref((string)($_POST['site_favicon_url'] ?? '')));
            $brandDisplayMode = (string)($_POST['brand_display_mode'] ?? 'name_only');
            if (!in_array($brandDisplayMode, ['favicon_and_name', 'name_only', 'favicon_only'], true)) {
                $brandDisplayMode = 'name_only';
            }
            settingSet('brand_display_mode', $brandDisplayMode);
            settingSet('site_assets_version', $assetsVersion);

            $faviconUpload = $_FILES['site_favicon_file'] ?? null;
            if (is_array($faviconUpload) && isset($faviconUpload['error']) && (int)$faviconUpload['error'] !== UPLOAD_ERR_NO_FILE) {
                $uploadErr = (int)$faviconUpload['error'];
                if ($uploadErr !== UPLOAD_ERR_OK) {
                    $warnings[] = 'Favicon upload failed. Please try again.';
                } else {
                    $uploadSize = (int)($faviconUpload['size'] ?? 0);
                    if ($uploadSize <= 0 || $uploadSize > 1048576) {
                        $warnings[] = 'Favicon must be between 1 byte and 1 MB.';
                    } else {
                        $tmpName = (string)($faviconUpload['tmp_name'] ?? '');
                        $finfo = new finfo(FILEINFO_MIME_TYPE);
                        $mime = strtolower((string)$finfo->file($tmpName));
                        $allowed = [
                            'image/x-icon' => 'ico',
                            'image/vnd.microsoft.icon' => 'ico',
                            'image/png' => 'png',
                            'image/webp' => 'webp',
                        ];

                        if (!isset($allowed[$mime])) {
                            $warnings[] = 'Unsupported favicon format. Use .ico, .png, or .webp.';
                        } else {
                            $targetDir = DATA_PATH . '/uploads/favicons';
                            if (!is_dir($targetDir)) {
                                mkdir($targetDir, 0755, true);
                            }

                            try {
                                $newStorage = bin2hex(random_bytes(16)) . '.' . $allowed[$mime];
                                $targetPath = $targetDir . '/' . $newStorage;
                                if (!move_uploaded_file($tmpName, $targetPath)) {
                                    $warnings[] = 'Could not store uploaded favicon.';
                                } else {
                                    $oldStorage = trim(settingGet('site_favicon_storage', ''));
                                    if ($oldStorage !== '') {
                                        $oldPath = $targetDir . '/' . basename($oldStorage);
                                        if (is_file($oldPath)) {
                                            @unlink($oldPath);
                                        }
                                    }

                                    settingSet('site_favicon_storage', $newStorage);
                                    settingSet('site_favicon_mime', $mime);
                                    settingSet('site_favicon_url', '/site-favicon');
                                }
                            } catch (Throwable $exception) {
                                $warnings[] = 'Favicon processing failed.';
                            }
                        }
                    }
                }
            }

            settingSet('about_page_content', trim((string)($_POST['about_page_content'] ?? '')));
            settingSet('dmca_page_content', trim((string)($_POST['dmca_page_content'] ?? '')));
            settingSet('legal_page_content', trim((string)($_POST['legal_page_content'] ?? '')));
            settingSet('terms_page_content', trim((string)($_POST['terms_page_content'] ?? '')));
            settingSet('privacy_page_content', trim((string)($_POST['privacy_page_content'] ?? '')));
            settingSet('legal_notice_page_content', trim((string)($_POST['legal_notice_page_content'] ?? '')));
            settingSet('post_disclaimer_enabled', isset($_POST['post_disclaimer_enabled']) ? '1' : '0');
            settingSet('post_disclaimer_title', mb_substr(trim((string)($_POST['post_disclaimer_title'] ?? 'Content Disclaimer')), 0, 120));
            settingSet('post_disclaimer_body', trim((string)($_POST['post_disclaimer_body'] ?? '')));

            settingSet('pagination_per_page', (string)intInRange($_POST['pagination_per_page'] ?? 20, 5, 100, 20));
            settingSet('board_default_max_files', (string)intInRange($_POST['board_default_max_files'] ?? 10, 1, 20, 10));
            $defaultFileMb = intInRange($_POST['board_default_max_file_mb'] ?? 20, 1, 50, 20);
            settingSet('board_default_max_file_mb', (string)$defaultFileMb);
            settingSet('board_default_max_file_bytes', (string)($defaultFileMb * 1024 * 1024));
            settingSet('board_default_cooldown_seconds', (string)intInRange($_POST['board_default_cooldown_seconds'] ?? 10, 0, 3600, 10));
            settingSet('board_default_max_posts_per_hour', (string)intInRange($_POST['board_default_max_posts_per_hour'] ?? 200, 1, 500, 200));
            settingSet('board_default_max_comments_per_hour', (string)intInRange($_POST['board_default_max_comments_per_hour'] ?? 500, 1, 1000, 500));
            $defaultCaptchaMode = (string)($_POST['board_default_captcha_mode'] ?? 'low_trust');
            if (!in_array($defaultCaptchaMode, ['off', 'low_trust', 'all'], true)) {
                $defaultCaptchaMode = 'low_trust';
            }
            settingSet('board_default_captcha_mode', $defaultCaptchaMode);
            settingSet('auto_approve_new_posts', isset($_POST['auto_approve_new_posts']) ? '1' : '0');

            settingSet('anti_spam_posts_per_10m', (string)intInRange($_POST['anti_spam_posts_per_10m'] ?? 8, 1, 100, 8));
            settingSet('anti_spam_comments_per_10m', (string)intInRange($_POST['anti_spam_comments_per_10m'] ?? 20, 1, 200, 20));
            settingSet('anti_spam_contact_per_hour', (string)intInRange($_POST['anti_spam_contact_per_hour'] ?? 5, 1, 100, 5));
            settingSet('low_trust_age_hours', (string)intInRange($_POST['low_trust_age_hours'] ?? 72, 1, 720, 72));
            settingSet('low_trust_min_activity', (string)intInRange($_POST['low_trust_min_activity'] ?? 5, 0, 1000, 5));

            settingSet('robots_custom_rules', trim((string)($_POST['robots_custom_rules'] ?? '')));
            settingSet('sitemap_auto_ping_enabled', isset($_POST['sitemap_auto_ping_enabled']) ? '1' : '0');
            settingSet('sitemap_ping_services', trim((string)($_POST['sitemap_ping_services'] ?? '')));

            settingSet('email_verification_enabled', isset($_POST['email_verification_enabled']) ? '1' : '0');
            settingSet('password_reset_enabled', isset($_POST['password_reset_enabled']) ? '1' : '0');
            settingSet('contact_forward_enabled', isset($_POST['contact_forward_enabled']) ? '1' : '0');
            settingSet('registration_enabled', isset($_POST['registration_enabled']) ? '1' : '0');

            settingSet('smtp_host', trim((string)($_POST['smtp_host'] ?? '')));
            settingSet('smtp_port', (string)intInRange($_POST['smtp_port'] ?? 587, 1, 65535, 587));
            settingSet('smtp_username', trim((string)($_POST['smtp_username'] ?? '')));
            settingSet('smtp_security', in_array((string)($_POST['smtp_security'] ?? 'tls'), ['tls', 'ssl', 'none'], true) ? (string)$_POST['smtp_security'] : 'tls');
            settingSet('smtp_from_email', trim((string)($_POST['smtp_from_email'] ?? '')));

            $newSmtpPassword = (string)($_POST['smtp_password'] ?? '');
            if ($newSmtpPassword !== '') {
                settingSet('smtp_password_enc', encryptSettingValue($newSmtpPassword));
            }

            logAudit($adminId, 'save_settings');

            if ((settingGet('email_verification_enabled', '0') === '1' || settingGet('password_reset_enabled', '0') === '1' || settingGet('contact_forward_enabled', '0') === '1') && !smtpIsConfigured()) {
                $warnings[] = 'Settings saved, but SMTP features are enabled without complete SMTP configuration.';
            }

            if (!empty($warnings)) {
                flashSet('error', implode(' ', $warnings));
            } else {
                flashSet('success', 'Settings updated.');
            }

            redirectTo('/admin/settings');
        }

        if ($action === 'admin_send_test_email') {
            $target = trim((string)($_POST['target_email'] ?? settingGet('legal_email', DEFAULT_LEGAL_EMAIL)));
            if (!filter_var($target, FILTER_VALIDATE_EMAIL)) {
                flashSet('error', 'Invalid target email.');
                redirectTo('/admin/settings');
            }

            [$ok, $msg] = smtpSendTextMail($target, 'SMTP Test Email', 'SMTP configuration test completed at ' . dbNow());
            flashSet($ok ? 'success' : 'error', $ok ? 'SMTP test email sent.' : ('SMTP test failed: ' . $msg));
            redirectTo('/admin/settings');
        }

        if ($action === 'admin_download_db') {
            $dbPath = appDbPath();
            if (!is_file($dbPath)) {
                flashSet('error', 'Database file not found.');
                redirectTo('/admin/settings');
            }

            $filename = 'site-backup-' . gmdate('Ymd-His') . '.sqlite';
            $size = (int)filesize($dbPath);

            logAudit($adminId, 'download_db_backup', $filename . ' bytes=' . $size);

            if (session_status() === PHP_SESSION_ACTIVE) {
                session_write_close();
            }

            header('Content-Description: File Transfer');
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Content-Transfer-Encoding: binary');
            header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
            header('Pragma: no-cache');
            header('Expires: 0');
            if ($size > 0) {
                header('Content-Length: ' . $size);
            }

            readfile($dbPath);
            exit;
        }

        if ($action === 'admin_cleanup_orphan_files') {
            $deletedUploads = 0;
            $deletedThumbs = 0;
            $deletedFavicons = 0;
            $reclaimedBytes = 0;

            $mediaRows = $db->query('SELECT id, storage_name, media_kind FROM media')->fetchAll();
            $keepImages = [];
            $keepVideos = [];
            $imageIdToStorage = [];

            foreach ($mediaRows as $mediaRow) {
                $storageName = basename((string)($mediaRow['storage_name'] ?? ''));
                if ($storageName === '') {
                    continue;
                }

                if ((string)($mediaRow['media_kind'] ?? 'image') === 'video') {
                    $keepVideos[$storageName] = true;
                } else {
                    $keepImages[$storageName] = true;
                    $imageIdToStorage[(int)$mediaRow['id']] = $storageName;
                }
            }

            $cleanupUploadDir = static function (string $dir, array $keepSet, int &$deletedCount, int &$bytesFreed): void {
                if (!is_dir($dir)) {
                    return;
                }

                $items = @scandir($dir);
                if (!is_array($items)) {
                    return;
                }

                foreach ($items as $item) {
                    if ($item === '.' || $item === '..') {
                        continue;
                    }

                    $path = $dir . '/' . $item;
                    if (!is_file($path)) {
                        continue;
                    }

                    if (isset($keepSet[$item])) {
                        continue;
                    }

                    $size = (int)@filesize($path);
                    if (@unlink($path)) {
                        $deletedCount++;
                        $bytesFreed += max(0, $size);
                    }
                }
            };

            $cleanupUploadDir(DATA_PATH . '/uploads/images', $keepImages, $deletedUploads, $reclaimedBytes);
            $cleanupUploadDir(DATA_PATH . '/uploads/videos', $keepVideos, $deletedUploads, $reclaimedBytes);

            $activeFavicon = basename(trim(settingGet('site_favicon_storage', '')));
            $faviconKeepSet = [];
            if ($activeFavicon !== '') {
                $faviconKeepSet[$activeFavicon] = true;
            }
            $cleanupUploadDir(DATA_PATH . '/uploads/favicons', $faviconKeepSet, $deletedFavicons, $reclaimedBytes);

            $thumbDir = DATA_PATH . '/cache/thumbs';
            if (is_dir($thumbDir)) {
                $thumbItems = @scandir($thumbDir);
                if (is_array($thumbItems)) {
                    foreach ($thumbItems as $thumbName) {
                        if ($thumbName === '.' || $thumbName === '..') {
                            continue;
                        }

                        $thumbPath = $thumbDir . '/' . $thumbName;
                        if (!is_file($thumbPath)) {
                            continue;
                        }

                        $deleteThumb = false;
                        if (!preg_match('/^(\d+)-[a-f0-9]{40}\.(?:webp|jpg|jpeg)$/i', $thumbName, $m)) {
                            $deleteThumb = true;
                        } else {
                            $thumbMediaId = (int)$m[1];
                            $imageStorage = $imageIdToStorage[$thumbMediaId] ?? '';
                            if ($imageStorage === '') {
                                $deleteThumb = true;
                            } else {
                                $imagePath = DATA_PATH . '/uploads/images/' . $imageStorage;
                                if (!is_file($imagePath)) {
                                    $deleteThumb = true;
                                }
                            }
                        }

                        if (!$deleteThumb) {
                            continue;
                        }

                        $size = (int)@filesize($thumbPath);
                        if (@unlink($thumbPath)) {
                            $deletedThumbs++;
                            $reclaimedBytes += max(0, $size);
                        }
                    }
                }
            }

            $reclaimedMb = number_format($reclaimedBytes / (1024 * 1024), 2);
            logAudit($adminId, 'cleanup_orphan_files', 'uploads=' . $deletedUploads . ' favicons=' . $deletedFavicons . ' thumbs=' . $deletedThumbs . ' bytes=' . $reclaimedBytes);
            flashSet('success', 'Cleanup finished. Removed ' . $deletedUploads . ' orphan upload file(s), ' . $deletedFavicons . ' orphan favicon file(s), and ' . $deletedThumbs . ' stale thumbnail file(s). Reclaimed ' . $reclaimedMb . ' MB.');
            redirectTo('/admin/settings');
        }

        if ($action === 'admin_add_allow_ip') {
            $ip = trim((string)($_POST['ip_address'] ?? ''));
            $label = trim((string)($_POST['label'] ?? ''));
            if (!filter_var($ip, FILTER_VALIDATE_IP)) {
                flashSet('error', 'Invalid IP.');
                redirectTo('/admin/settings');
            }
            $stmt = $db->prepare('INSERT INTO admin_allowed_ips (ip_address, label, added_at) VALUES (?, ?, ?) ON CONFLICT(ip_address) DO UPDATE SET label = excluded.label, added_at = excluded.added_at');
            $stmt->execute([$ip, $label, dbNow()]);
            logAudit($adminId, 'allow_ip_add', $ip);
            flashSet('success', 'IP added to admin allowlist.');
            redirectTo('/admin/settings');
        }

        if ($action === 'admin_remove_allow_ip') {
            $ip = trim((string)($_POST['ip_address'] ?? ''));
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                $db->prepare('DELETE FROM admin_allowed_ips WHERE ip_address = ?')->execute([$ip]);
                logAudit($adminId, 'allow_ip_remove', $ip);
                flashSet('success', 'IP removed from allowlist.');
            }
            redirectTo('/admin/settings');
        }

        if ($action === 'admin_create_admin') {
            $username = trim((string)($_POST['username'] ?? ''));
            $email = trim((string)($_POST['email'] ?? ''));
            $password = (string)($_POST['password'] ?? '');

            if (!preg_match('/^\w{3,30}$/', $username) || !filter_var($email, FILTER_VALIDATE_EMAIL) || strlen($password) < 10) {
                flashSet('error', 'Invalid admin user details. Password must be at least 10 characters.');
                redirectTo('/admin/users');
            }

            try {
                $stmt = $db->prepare('INSERT INTO users (username, email, password_hash, role, is_banned, email_verified_at, created_at) VALUES (?, ?, ?, "admin", 0, ?, ?)');
                $stmt->execute([$username, $email, securePasswordHash($password), dbNow(), dbNow()]);
                logAudit($adminId, 'create_admin_user', $username);
                flashSet('success', 'Admin account created.');
            } catch (Throwable $exception) {
                flashSet('error', 'Could not create admin account (username/email already used).');
            }
            redirectTo('/admin/users');
        }
    }

    flashSet('error', 'Unknown action.');
    redirectTo('/');
}

if (($routeParts[0] ?? '') === 'thumb' && isset($routeParts[1])) {
    $mediaId = (int)$routeParts[1];
    if ($mediaId <= 0) {
        render404();
    }

    $stmt = $db->prepare('SELECT * FROM media WHERE id = ? LIMIT 1');
    $stmt->execute([$mediaId]);
    $media = $stmt->fetch();

    if (!$media) {
        render404();
    }

    if (($media['media_kind'] ?? '') !== 'image') {
        redirectTo('/media/' . $mediaId);
    }

    $sourcePath = mediaStoragePath($media);
    if (!is_file($sourcePath)) {
        render404();
    }

    $thumbSize = intInRange($_GET['s'] ?? 360, 120, 640, 360);
    $cacheDir = DATA_PATH . '/cache/thumbs';
    if (!is_dir($cacheDir)) {
        @mkdir($cacheDir, 0755, true);
    }

    $signature = sha1($sourcePath . '|' . (string)@filesize($sourcePath) . '|' . (string)@filemtime($sourcePath) . '|' . $thumbSize);
    $useWebp = function_exists('imagewebp');
    $cacheExt = $useWebp ? 'webp' : 'jpg';
    $cachePath = $cacheDir . '/' . $mediaId . '-' . $signature . '.' . $cacheExt;
    $cacheMime = $useWebp ? 'image/webp' : 'image/jpeg';

    if (is_file($cachePath)) {
        header('Content-Type: ' . $cacheMime);
        header('Content-Length: ' . (string)filesize($cachePath));
        header('Cache-Control: public, max-age=604800');
        readfile($cachePath);
        exit;
    }

    if (!extension_loaded('gd') || !function_exists('imagecreatefromstring')) {
        redirectTo('/media/' . $mediaId);
    }

    $raw = @file_get_contents($sourcePath);
    if ($raw === false) {
        render404();
    }

    $src = @imagecreatefromstring($raw);
    if (!$src) {
        redirectTo('/media/' . $mediaId);
    }

    $srcW = imagesx($src);
    $srcH = imagesy($src);
    if ($srcW <= 0 || $srcH <= 0) {
        imagedestroy($src);
        redirectTo('/media/' . $mediaId);
    }

    $scale = min($thumbSize / $srcW, $thumbSize / $srcH, 1.0);
    $dstW = max(1, (int)round($srcW * $scale));
    $dstH = max(1, (int)round($srcH * $scale));

    $dst = imagecreatetruecolor($dstW, $dstH);
    imagealphablending($dst, false);
    imagesavealpha($dst, true);
    $transparent = imagecolorallocatealpha($dst, 0, 0, 0, 127);
    imagefilledrectangle($dst, 0, 0, $dstW, $dstH, $transparent);
    imagecopyresampled($dst, $src, 0, 0, 0, 0, $dstW, $dstH, $srcW, $srcH);

    ob_start();
    if ($useWebp) {
        imagewebp($dst, null, 80);
    } else {
        imagejpeg($dst, null, 82);
    }
    $thumbBinary = (string)ob_get_clean();

    imagedestroy($src);
    imagedestroy($dst);

    if ($thumbBinary === '') {
        redirectTo('/media/' . $mediaId);
    }

    @file_put_contents($cachePath, $thumbBinary, LOCK_EX);

    header('Content-Type: ' . $cacheMime);
    header('Content-Length: ' . (string)strlen($thumbBinary));
    header('Cache-Control: public, max-age=604800');
    echo $thumbBinary;
    exit;
}

if (($routeParts[0] ?? '') === 'media' && isset($routeParts[1])) {
    $mediaId = (int)$routeParts[1];
    $stmt = $db->prepare('SELECT * FROM media WHERE id = ? LIMIT 1');
    $stmt->execute([$mediaId]);
    $media = $stmt->fetch();

    if (!$media) {
        render404();
    }

    $path = mediaStoragePath($media);
    if (!is_file($path)) {
        render404();
    }

    $size = (int)filesize($path);
    $start = 0;
    $end = max(0, $size - 1);

    header('Content-Type: ' . $media['mime_type']);
    header('Accept-Ranges: bytes');
    header('Cache-Control: public, max-age=3600');

    $rangeHeader = (string)($_SERVER['HTTP_RANGE'] ?? '');
    if ($rangeHeader !== '' && preg_match('/bytes=(\d*)-(\d*)/i', $rangeHeader, $m)) {
        $rawStart = $m[1] ?? '';
        $rawEnd = $m[2] ?? '';

        if ($rawStart === '' && $rawEnd === '') {
            header('HTTP/1.1 416 Range Not Satisfiable');
            header('Content-Range: bytes */' . $size);
            exit;
        }

        if ($rawStart === '') {
            $suffix = (int)$rawEnd;
            if ($suffix <= 0) {
                header('HTTP/1.1 416 Range Not Satisfiable');
                header('Content-Range: bytes */' . $size);
                exit;
            }
            $start = max(0, $size - $suffix);
            $end = max(0, $size - 1);
        } else {
            $start = (int)$rawStart;
            $end = $rawEnd !== '' ? (int)$rawEnd : max(0, $size - 1);
        }

        $end = min($end, max(0, $size - 1));
        if ($start > $end || $start >= $size) {
            header('HTTP/1.1 416 Range Not Satisfiable');
            header('Content-Range: bytes */' . $size);
            exit;
        }

        $length = $end - $start + 1;
        header('HTTP/1.1 206 Partial Content');
        header('Content-Range: bytes ' . $start . '-' . $end . '/' . $size);
        header('Content-Length: ' . (string)$length);

        $fp = fopen($path, 'rb');
        if ($fp === false) {
            render404();
        }

        fseek($fp, $start);
        $remaining = $length;
        while (!feof($fp) && $remaining > 0) {
            $chunkSize = (int)min(8192, $remaining);
            $buffer = fread($fp, $chunkSize);
            if ($buffer === false) {
                break;
            }
            echo $buffer;
            $remaining -= strlen($buffer);
        }
        fclose($fp);
        exit;
    }

    header('Content-Length: ' . (string)$size);
    readfile($path);
    exit;
}

if ($route === 'verify-email') {
    $token = trim((string)($_GET['token'] ?? ''));
    if ($token === '' || !consumeEmailVerificationToken($token)) {
        flashSet('error', 'Invalid or expired verification token.');
    } else {
        flashSet('success', 'Email verified successfully.');
    }
    redirectTo('/login');
}

if ($route === 'logout') {
    doLogout();
    redirectTo('/');
}

if ($route === 'register') {
    if (!registrationEnabled()) {
        flashSet('error', 'Registration is currently disabled by admin.');
        redirectTo('/login');
    }

    $html = '<section class="card"><h1>Create account</h1><form method="POST" class="form-grid">'
        . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
        . '<input type="hidden" name="website" value="">'
        . '<input type="hidden" name="action" value="register">'
        . '<label>Username<input name="username" required minlength="3" maxlength="30"></label>'
        . '<label>Email<input name="email" type="email" required></label>'
        . '<label>Password<input name="password" type="password" required minlength="8"></label>'
        . '<button type="submit">Register</button>'
        . '</form></section>';
    renderLayout('Register', $html, ['description' => 'Create an account to post and comment on boards.']);
    exit;
}

if ($route === 'login') {
    $verifyNote = emailVerificationEnabled() ? '<p class="muted">Email verification is required before login.</p>' : '';
    $registrationNote = registrationEnabled() ? '' : '<p class="muted">Registration is currently disabled by admin.</p>';
    $html = '<section class="card"><h1>Login</h1>' . $verifyNote . $registrationNote . '<form method="POST" class="form-grid">'
        . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
        . '<input type="hidden" name="website" value="">'
        . '<input type="hidden" name="action" value="login">'
        . '<label>Username<input name="username" required></label>'
        . '<label>Password<input name="password" type="password" required></label>'
        . '<button type="submit">Login</button>'
        . '</form></section>';
    renderLayout('Login', $html, ['description' => 'Login to your account.']);
    exit;
}

if ($route === 'forgot-password') {
    if (!passwordResetEnabled()) {
        flashSet('error', 'Password reset is disabled.');
        redirectTo('/login');
    }

    $html = '<section class="card"><h1>Forgot password</h1><form method="POST" class="form-grid">'
        . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
        . '<input type="hidden" name="website" value="">'
        . '<input type="hidden" name="action" value="forgot_password">'
        . '<label>Account email<input type="email" name="email" required></label>'
        . '<button type="submit">Send reset link</button>'
        . '</form></section>';
    renderLayout('Forgot password', $html, ['noindex' => true]);
    exit;
}

if ($route === 'reset-password') {
    if (!passwordResetEnabled()) {
        flashSet('error', 'Password reset is disabled.');
        redirectTo('/login');
    }

    $token = trim((string)($_GET['token'] ?? ''));
    if ($token === '' || passwordResetTokenUser($token) === null) {
        renderLayout('Reset password', '<section class="card"><h1>Reset password</h1><p>Invalid or expired reset token.</p></section>', ['noindex' => true]);
        exit;
    }

    $html = '<section class="card"><h1>Reset password</h1><form method="POST" class="form-grid">'
        . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
        . '<input type="hidden" name="website" value="">'
        . '<input type="hidden" name="action" value="reset_password">'
        . '<input type="hidden" name="token" value="' . e($token) . '">'
        . '<label>New password<input type="password" name="password" required minlength="10"></label>'
        . '<label>Confirm password<input type="password" name="password_confirm" required minlength="10"></label>'
        . '<button type="submit">Update password</button>'
        . '</form></section>';
    renderLayout('Reset password', $html, ['noindex' => true]);
    exit;
}

if ($route === 'search') {
    $q = trim((string)($_GET['q'] ?? ''));
    $perPage = paginationPerPage();
    $page = queryPage();

    $html = '<section class="card"><h1>Search</h1><form method="GET" action="/search" class="form-grid">'
        . '<label>Query<input type="search" name="q" value="' . e($q) . '" required></label>'
        . '<button type="submit">Search</button>'
        . '</form></section>';

    if ($q !== '' && mb_strlen($q) >= 2) {
        $like = '%' . $q . '%';

        $boardStmt = $db->prepare('SELECT s.id AS sub_id, s.name AS sub_name, s.slug AS sub_slug, c.id AS cat_id, c.name AS cat_name, c.slug AS cat_slug,
            (SELECT COUNT(*) FROM posts p WHERE p.subcategory_id = s.id AND p.status = "active") AS post_count
            FROM subcategories s
            JOIN categories c ON c.id = s.category_id
            WHERE (
                LOWER(c.name) LIKE LOWER(:board_like1)
                OR LOWER(c.slug) LIKE LOWER(:board_like2)
                OR LOWER(COALESCE(c.description, "")) LIKE LOWER(:board_like3)
                OR LOWER(s.name) LIKE LOWER(:board_like4)
                OR LOWER(s.slug) LIKE LOWER(:board_like5)
                OR LOWER(COALESCE(s.description, "")) LIKE LOWER(:board_like6)
            )
            ORDER BY c.sort_order ASC, s.sort_order ASC, s.name ASC
            LIMIT 100');
        $boardStmt->bindValue(':board_like1', $like, PDO::PARAM_STR);
        $boardStmt->bindValue(':board_like2', $like, PDO::PARAM_STR);
        $boardStmt->bindValue(':board_like3', $like, PDO::PARAM_STR);
        $boardStmt->bindValue(':board_like4', $like, PDO::PARAM_STR);
        $boardStmt->bindValue(':board_like5', $like, PDO::PARAM_STR);
        $boardStmt->bindValue(':board_like6', $like, PDO::PARAM_STR);
        $boardStmt->execute();
        $boardRows = $boardStmt->fetchAll();

        $countStmt = $db->prepare('SELECT COUNT(*) FROM posts p
            JOIN subcategories s ON s.id = p.subcategory_id
            JOIN categories c ON c.id = s.category_id
            WHERE p.status = "active" AND (
                LOWER(p.title) LIKE LOWER(:count_like1)
                OR LOWER(p.body) LIKE LOWER(:count_like2)
                OR LOWER(c.name) LIKE LOWER(:count_like3)
                OR LOWER(c.slug) LIKE LOWER(:count_like4)
                OR LOWER(s.name) LIKE LOWER(:count_like5)
                OR LOWER(s.slug) LIKE LOWER(:count_like6)
                OR EXISTS (
                    SELECT 1 FROM comments cm
                    WHERE cm.post_id = p.id
                        AND cm.status = "active"
                        AND LOWER(cm.body) LIKE LOWER(:count_like7)
                )
            )');
        $countStmt->bindValue(':count_like1', $like, PDO::PARAM_STR);
        $countStmt->bindValue(':count_like2', $like, PDO::PARAM_STR);
        $countStmt->bindValue(':count_like3', $like, PDO::PARAM_STR);
        $countStmt->bindValue(':count_like4', $like, PDO::PARAM_STR);
        $countStmt->bindValue(':count_like5', $like, PDO::PARAM_STR);
        $countStmt->bindValue(':count_like6', $like, PDO::PARAM_STR);
        $countStmt->bindValue(':count_like7', $like, PDO::PARAM_STR);
        $countStmt->execute();
        $total = (int)$countStmt->fetchColumn();
        $pages = totalPages($total, $perPage);
        $page = min($page, $pages);

        $stmt = $db->prepare('SELECT p.id, p.title, p.body, p.created_at, u.id AS user_id, u.display_name, s.slug AS sub_slug, c.slug AS cat_slug,
            (SELECT m.id FROM media m WHERE m.post_id = p.id AND (m.comment_id IS NULL OR m.comment_id = 0) ORDER BY m.id ASC LIMIT 1) AS first_media_id,
            (SELECT m.media_kind FROM media m WHERE m.post_id = p.id AND (m.comment_id IS NULL OR m.comment_id = 0) ORDER BY m.id ASC LIMIT 1) AS first_media_kind,
            (SELECT COUNT(*) FROM comments cm WHERE cm.post_id = p.id AND cm.status = "active" AND LOWER(cm.body) LIKE LOWER(:like7)) AS comment_match_count
            FROM posts p
            JOIN users u ON u.id = p.user_id
            JOIN subcategories s ON s.id = p.subcategory_id
            JOIN categories c ON c.id = s.category_id
            WHERE p.status = "active" AND (
                LOWER(p.title) LIKE LOWER(:like1)
                OR LOWER(p.body) LIKE LOWER(:like2)
                OR LOWER(c.name) LIKE LOWER(:like3)
                OR LOWER(c.slug) LIKE LOWER(:like4)
                OR LOWER(s.name) LIKE LOWER(:like5)
                OR LOWER(s.slug) LIKE LOWER(:like6)
                OR EXISTS (
                    SELECT 1 FROM comments cm2
                    WHERE cm2.post_id = p.id
                        AND cm2.status = "active"
                        AND LOWER(cm2.body) LIKE LOWER(:like8)
                )
            )
            ORDER BY p.id DESC
            LIMIT :limit OFFSET :offset');
        $stmt->bindValue(':like1', $like, PDO::PARAM_STR);
        $stmt->bindValue(':like2', $like, PDO::PARAM_STR);
        $stmt->bindValue(':like3', $like, PDO::PARAM_STR);
        $stmt->bindValue(':like4', $like, PDO::PARAM_STR);
        $stmt->bindValue(':like5', $like, PDO::PARAM_STR);
        $stmt->bindValue(':like6', $like, PDO::PARAM_STR);
        $stmt->bindValue(':like7', $like, PDO::PARAM_STR);
        $stmt->bindValue(':like8', $like, PDO::PARAM_STR);
        $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', pageOffset($page, $perPage), PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll();

        $html .= '<section class="card"><h2>Matching boards (' . count($boardRows) . ')</h2>';
        if (empty($boardRows)) {
            $html .= '<p>No matching boards found.</p>';
        } else {
            $html .= '<ul class="clean-list">';
            foreach ($boardRows as $boardRow) {
                $boardPostCount = (int)($boardRow['post_count'] ?? 0);
                $html .= '<li><a href="/board/' . e((string)$boardRow['cat_slug']) . '/' . e((string)$boardRow['sub_slug']) . '">/' . e((string)$boardRow['cat_slug']) . '/' . e((string)$boardRow['sub_slug']) . '/</a> '
                    . '<small>' . e((string)$boardRow['cat_name']) . ' &gt; ' . e((string)$boardRow['sub_name']) . ' • ' . $boardPostCount . ' thread' . ($boardPostCount === 1 ? '' : 's') . '</small></li>';
            }
            $html .= '</ul>';
        }
        $html .= '</section>';

        $html .= '<section class="card"><h2>Matching threads (' . $total . ')</h2>';
        if (empty($rows)) {
            $html .= '<p>No matching threads found.</p>';
        } else {
            $html .= '<ul class="thread-list">';
            foreach ($rows as $row) {
                $firstMediaId = (int)($row['first_media_id'] ?? 0);
                $firstMediaKind = (string)($row['first_media_kind'] ?? 'image');
                $commentMatchCount = (int)($row['comment_match_count'] ?? 0);
                $commentMatchLabel = '';
                if ($commentMatchCount > 0) {
                    $commentMatchLabel = ' • ' . $commentMatchCount . ' comment match';
                    if ($commentMatchCount !== 1) {
                        $commentMatchLabel .= 'es';
                    }
                }
                $html .= '<li class="thread-row">';
                if ($firstMediaId > 0) {
                    if ($firstMediaKind === 'video') {
                        $html .= '<a class="thread-preview-thumb" href="/post/' . (int)$row['id'] . '"><video muted playsinline preload="metadata" src="' . e('/media/' . $firstMediaId) . '"></video></a>';
                    } else {
                        $html .= '<a class="thread-preview-thumb" href="/post/' . (int)$row['id'] . '"><img loading="lazy" decoding="async" fetchpriority="low" src="' . e(mediaThumbUrl($firstMediaId)) . '" alt="Thread preview"></a>';
                    }
                }
                $html .= '<div class="thread-row-body"><h3><a href="/post/' . (int)$row['id'] . '">' . e((string)$row['title']) . '</a></h3>'
                    . '<p>' . e(mb_substr((string)$row['body'], 0, 240)) . '</p>'
                    . '<small>/' . e((string)$row['cat_slug']) . '/' . e((string)$row['sub_slug']) . '/ by ' . authorLinkFromRow($row) . $commentMatchLabel . '</small></div></li>';
            }
            $html .= '</ul>';
            $html .= renderPagination('/search', ['q' => $q], $page, $pages);
        }
        $html .= '</section>';

        renderLayout('Search', $html, [
            'description' => 'Search threads and posts.',
            'canonical' => absoluteUrl('/search?q=' . rawurlencode($q) . ($page > 1 ? '&page=' . $page : '')),
            'noindex' => $page > 1,
        ]);
        exit;
    }

    renderLayout('Search', $html, ['description' => 'Search posts by title and content.']);
    exit;
}

if ($route === 'boards') {
    $categories = $db->query('SELECT c.id, c.name, c.slug, c.description FROM categories c ORDER BY c.sort_order ASC, c.name ASC')->fetchAll();
    $subs = $db->query('SELECT sc.id, sc.name, sc.slug, sc.description, sc.category_id, c.slug AS cat_slug, c.name AS cat_name,
        (SELECT COUNT(*) FROM posts p WHERE p.subcategory_id = sc.id AND p.status = "active") AS post_count
        FROM subcategories sc
        JOIN categories c ON c.id = sc.category_id
        ORDER BY c.sort_order ASC, sc.sort_order ASC, sc.name ASC')->fetchAll();

    $subsByCategory = [];
    foreach ($subs as $sub) {
        $subsByCategory[(int)$sub['category_id']][] = $sub;
    }

    $html = '<section class="card"><h1>Boards</h1></section>';
    foreach ($categories as $category) {
        $html .= '<section class="card"><h2>' . e((string)$category['name']) . '</h2>';
        if (!empty($category['description'])) {
            $html .= '<p>' . e((string)$category['description']) . '</p>';
        }

        $html .= '<ul class="clean-list">';
        foreach ($subsByCategory[(int)$category['id']] ?? [] as $sub) {
            $html .= '<li><a href="/board/' . e((string)$sub['cat_slug']) . '/' . e((string)$sub['slug']) . '">' . e((string)$sub['cat_name']) . ' &gt; ' . e((string)$sub['name']) . '</a> <small>(' . (int)$sub['post_count'] . ' threads)</small></li>';
        }
        $html .= '</ul></section>';
    }

    renderLayout('Boards', $html, [
        'description' => 'Browse all categories and boards.',
        'canonical' => absoluteUrl('/boards'),
    ]);
    exit;
}

if (($routeParts[0] ?? '') === 'admin') {
    requireAdminIpOr404();

    if (!isAdmin()) {
        $html = '<section class="card"><h1>Admin Login</h1><p>Only allowlisted IPs can access this page.</p>'
            . '<form method="POST" class="form-grid">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
            . '<input type="hidden" name="website" value="">'
            . '<input type="hidden" name="action" value="admin_login">'
            . '<label>Admin Username<input name="username" required></label>'
            . '<label>Password<input name="password" type="password" required></label>'
            . '<button type="submit">Enter Admin</button>'
            . '</form></section>';
        renderLayout('Admin Login', $html, ['noindex' => true]);
        exit;
    }

    $adminPage = $routeParts[1] ?? 'dashboard';

    if ($adminPage === 'dashboard') {
        $stats = [
            'users' => (int)$db->query('SELECT COUNT(*) FROM users')->fetchColumn(),
            'posts' => (int)$db->query('SELECT COUNT(*) FROM posts WHERE status = "active"')->fetchColumn(),
            'comments' => (int)$db->query('SELECT COUNT(*) FROM comments WHERE status = "active"')->fetchColumn(),
            'media' => (int)$db->query('SELECT COUNT(*) FROM media')->fetchColumn(),
        ];

        $logs = $db->query('SELECT al.*, u.username FROM audit_logs al LEFT JOIN users u ON u.id = al.actor_user_id ORDER BY al.id DESC LIMIT 15')->fetchAll();

        $html = '<section class="card"><h1>Admin Dashboard</h1>'
            . '<div class="stat-grid">'
            . '<article><h3>Users</h3><p>' . $stats['users'] . '</p></article>'
            . '<article><h3>Posts</h3><p>' . $stats['posts'] . '</p></article>'
            . '<article><h3>Comments</h3><p>' . $stats['comments'] . '</p></article>'
            . '<article><h3>Media Files</h3><p>' . $stats['media'] . '</p></article>'
            . '</div><h2>Recent admin actions</h2><ul class="clean-list">';

        foreach ($logs as $log) {
            $who = $log['username'] ? '@' . $log['username'] : 'system';
            $html .= '<li><strong>' . e($who) . '</strong> ' . e((string)$log['action']) . ' <small>' . e((string)$log['details']) . ' • ' . e((string)$log['created_at']) . '</small></li>';
        }
        $html .= '</ul></section>';
        renderLayout('Admin Dashboard', $html, ['is_admin' => true, 'noindex' => true]);
        exit;
    }

    if ($adminPage === 'categories') {
        $categories = $db->query('SELECT * FROM categories ORDER BY sort_order ASC, name ASC')->fetchAll();
        $subcategories = $db->query('SELECT sc.*, c.name AS category_name, c.slug AS category_slug, br.max_files, br.max_file_bytes, br.cooldown_seconds, br.max_posts_per_hour, br.max_comments_per_hour, br.allow_video, br.captcha_mode
            FROM subcategories sc
            JOIN categories c ON c.id = sc.category_id
            LEFT JOIN board_rules br ON br.subcategory_id = sc.id
            ORDER BY c.sort_order ASC, sc.sort_order ASC')->fetchAll();

        $subsByCategoryAdmin = [];
        foreach ($subcategories as $sub) {
            $subsByCategoryAdmin[(int)$sub['category_id']][] = $sub;
        }

        $html = '<section class="card"><h1>Categories</h1>'
            . '<div class="two-col">'
            . '<div><h2>Create category</h2><form method="POST" class="form-grid">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_create_category">'
            . '<label>Name<input name="name" required></label>'
            . '<label>Slug (optional)<input name="slug"></label>'
            . '<label>Description<textarea name="description"></textarea></label>'
            . '<label>Sort order<input type="number" name="sort_order" value="0"></label>'
            . '<button type="submit">Add Category</button></form></div>'
            . '<div><h2>Create subcategory</h2><form method="POST" class="form-grid">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_create_subcategory">'
            . '<label>Parent category<select name="category_id" required>';

        foreach ($categories as $category) {
            $html .= '<option value="' . (int)$category['id'] . '">' . e((string)$category['name']) . '</option>';
        }

        $html .= '</select></label>'
            . '<label>Name<input name="name" required></label>'
            . '<label>Slug (optional)<input name="slug"></label>'
            . '<label>Description<textarea name="description"></textarea></label>'
            . '<label>Sort order<input type="number" name="sort_order" value="0"></label>'
            . '<button type="submit">Add Subcategory</button></form></div>'
            . '</div></section>';

        $html .= '<section class="card"><h2>Manage categories and boards</h2><p class="muted">Collapsed layout: open only what you want to edit/delete.</p>'
            . '<form method="POST" class="inline-form">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
            . '<input type="hidden" name="website" value="">'
            . '<input type="hidden" name="action" value="admin_resync_category_slugs">'
            . '<button type="submit">Resync all category slugs with names</button>'
            . '</form>'
            . '<form method="POST" class="inline-form">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
            . '<input type="hidden" name="website" value="">'
            . '<input type="hidden" name="action" value="admin_resync_subcategory_slugs">'
            . '<button type="submit">Resync all subcategory slugs with names</button>'
            . '</form>';
        if (empty($categories)) {
            $html .= '<p>No categories yet.</p>';
        } else {
            foreach ($categories as $idx => $category) {
                $catSubs = $subsByCategoryAdmin[(int)$category['id']] ?? [];
                $html .= '<details class="admin-collapse" ' . ($idx === 0 ? 'open' : '') . '><summary><strong>' . e((string)$category['name']) . '</strong> <small>(' . e((string)$category['slug']) . ' • ' . count($catSubs) . ' boards)</small></summary><div class="collapse-body">';

                if (!empty($category['description'])) {
                    $html .= '<p class="muted">' . e((string)$category['description']) . '</p>';
                }

                $html .= '<form method="POST" class="form-grid">'
                    . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
                    . '<input type="hidden" name="website" value="">'
                    . '<input type="hidden" name="action" value="admin_rename_category">'
                    . '<input type="hidden" name="category_id" value="' . (int)$category['id'] . '">'
                    . '<label>Category name<input name="name" value="' . e((string)$category['name']) . '" required></label>'
                    . '<label>Description<textarea name="description" rows="3">' . e((string)($category['description'] ?? '')) . '</textarea></label>'
                    . '<button type="submit">Save Category</button>'
                    . '<p class="muted">Slug auto-updates from the name when the name changes.</p>'
                    . '</form>';

                $html .= '<form method="POST" class="inline-form admin-danger">'
                    . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
                    . '<input type="hidden" name="website" value="">'
                    . '<input type="hidden" name="action" value="admin_delete_category">'
                    . '<input type="hidden" name="category_id" value="' . (int)$category['id'] . '">'
                    . '<button type="submit">Delete Category</button>'
                    . '</form>';

                if (empty($catSubs)) {
                    $html .= '<p class="muted">No subcategories in this category yet.</p>';
                } else {
                    $html .= '<ul class="clean-list compact-list">';
                    foreach ($catSubs as $sub) {
                        $maxFiles = $sub['max_files'] !== null ? (int)$sub['max_files'] : intInRange(settingGet('board_default_max_files', '10'), 1, 20, 10);
                        $defaultFileMb = intInRange(settingGet('board_default_max_file_mb', '20'), 1, 50, 20);
                        $maxBytes = $sub['max_file_bytes'] !== null ? (int)$sub['max_file_bytes'] : ($defaultFileMb * 1024 * 1024);
                        $maxMb = bytesToMb($maxBytes);
                        $cooldown = $sub['cooldown_seconds'] !== null ? (int)$sub['cooldown_seconds'] : intInRange(settingGet('board_default_cooldown_seconds', '10'), 0, 3600, 10);
                        $mph = $sub['max_posts_per_hour'] !== null ? (int)$sub['max_posts_per_hour'] : intInRange(settingGet('board_default_max_posts_per_hour', '200'), 1, 500, 200);
                        $mch = $sub['max_comments_per_hour'] !== null ? (int)$sub['max_comments_per_hour'] : intInRange(settingGet('board_default_max_comments_per_hour', '500'), 1, 1000, 500);
                        $allowVideo = $sub['allow_video'] !== null ? (int)$sub['allow_video'] : 1;
                        $captchaMode = $sub['captcha_mode'] !== null ? (string)$sub['captcha_mode'] : settingGet('board_default_captcha_mode', 'low_trust');

                        $html .= '<li><details class="admin-sub-collapse"><summary><strong>' . e((string)$sub['slug']) . '</strong></summary><div class="collapse-body">'
                            . '<form method="POST" class="form-grid">'
                            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_rename_subcategory"><input type="hidden" name="subcategory_id" value="' . (int)$sub['id'] . '">'
                            . '<label>Subcategory name<input name="name" value="' . e((string)$sub['name']) . '" required></label>'
                            . '<label>Description<textarea name="description" rows="3">' . e((string)($sub['description'] ?? '')) . '</textarea></label>'
                            . '<button type="submit">Save Subcategory</button>'
                            . '<p class="muted">Slug auto-updates from the name when the name changes.</p>'
                            . '</form>'
                            . '<form method="POST" class="form-grid">'
                            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_save_board_rule"><input type="hidden" name="subcategory_id" value="' . (int)$sub['id'] . '">'
                            . '<label>Max files<input type="number" name="max_files" min="1" max="20" value="' . $maxFiles . '"></label>'
                            . '<label>Max file size (MB)<input type="number" name="max_file_mb" min="1" max="50" value="' . $maxMb . '"></label>'
                            . '<label>Cooldown seconds<input type="number" name="cooldown_seconds" min="0" max="3600" value="' . $cooldown . '"></label>'
                            . '<label>Max posts/hour<input type="number" name="max_posts_per_hour" min="1" max="500" value="' . $mph . '"></label>'
                            . '<label>Max comments/hour<input type="number" name="max_comments_per_hour" min="1" max="1000" value="' . $mch . '"></label>'
                            . '<label>CAPTCHA mode<select name="captcha_mode">'
                            . '<option value="off" ' . ($captchaMode === 'off' ? 'selected' : '') . '>Off</option>'
                            . '<option value="low_trust" ' . ($captchaMode === 'low_trust' ? 'selected' : '') . '>Low-trust users only</option>'
                            . '<option value="all" ' . ($captchaMode === 'all' ? 'selected' : '') . '>All users</option>'
                            . '</select></label>'
                            . '<label><input type="checkbox" name="allow_video" ' . ($allowVideo === 1 ? 'checked' : '') . '> Allow video uploads (webm)</label>'
                            . '<button type="submit">Save board rule</button></form>'
                            . '<form method="POST" class="inline-form admin-danger">'
                            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
                            . '<input type="hidden" name="website" value="">'
                            . '<input type="hidden" name="action" value="admin_delete_subcategory">'
                            . '<input type="hidden" name="subcategory_id" value="' . (int)$sub['id'] . '">'
                            . '<button type="submit">Delete Subcategory</button>'
                            . '</form>'
                            . '</div></details></li>';
                    }
                    $html .= '</ul>';
                }

                $html .= '</div></details>';
            }
        }
        $html .= '</section>';

        renderLayout('Admin Categories', $html, ['is_admin' => true, 'noindex' => true]);
        exit;
    }

    if ($adminPage === 'moderation') {
        $posts = $db->query('SELECT p.id, p.title, p.status, p.created_at, u.username FROM posts p JOIN users u ON u.id = p.user_id ORDER BY p.id DESC LIMIT 50')->fetchAll();
        $comments = $db->query('SELECT c.id, c.post_id, c.body, c.status, c.created_at, u.username FROM comments c JOIN users u ON u.id = c.user_id ORDER BY c.id DESC LIMIT 60')->fetchAll();
        $media = $db->query('SELECT m.id, m.post_id, m.original_name, m.mime_type, m.created_at, u.username FROM media m JOIN users u ON u.id = m.user_id ORDER BY m.id DESC LIMIT 60')->fetchAll();

        $html = '<section class="card"><h1>Moderation</h1><h2>Posts</h2><ul class="clean-list">';
        foreach ($posts as $post) {
            $html .= '<li>#' . (int)$post['id'] . ' ' . e((string)$post['title']) . ' by @' . e((string)$post['username']) . ' [' . e((string)$post['status']) . '] '
                . '<a href="/post/' . (int)$post['id'] . '">View</a>'
                . '<form method="POST" class="inline-form"><input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_delete_post"><input type="hidden" name="post_id" value="' . (int)$post['id'] . '"><button type="submit">Delete</button></form></li>';
        }

        $html .= '</ul><h2>Comments</h2><ul class="clean-list">';
        foreach ($comments as $comment) {
            $html .= '<li>#' . (int)$comment['id'] . ' on post ' . (int)$comment['post_id'] . ' by @' . e((string)$comment['username']) . ' [' . e((string)$comment['status']) . '] '
                . '<small>' . e(mb_substr((string)$comment['body'], 0, 120)) . '</small>'
                . '<form method="POST" class="inline-form"><input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_delete_comment"><input type="hidden" name="comment_id" value="' . (int)$comment['id'] . '"><button type="submit">Delete</button></form></li>';
        }

        $html .= '</ul><h2>Media</h2><ul class="clean-list">';
        foreach ($media as $file) {
            $html .= '<li>#' . (int)$file['id'] . ' ' . e((string)$file['original_name']) . ' (' . e((string)$file['mime_type']) . ') '
                . '<a href="/media/' . (int)$file['id'] . '" target="_blank" rel="noopener">Open</a>'
                . '<form method="POST" class="inline-form"><input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_delete_media"><input type="hidden" name="media_id" value="' . (int)$file['id'] . '"><button type="submit">Delete</button></form></li>';
        }

        $html .= '</ul></section>';
        renderLayout('Admin Moderation', $html, ['is_admin' => true, 'noindex' => true]);
        exit;
    }

    if ($adminPage === 'pending-posts') {
        $posts = $db->query('SELECT p.id, p.title, p.body, p.created_at, p.updated_at,
            u.id AS user_id, u.username, u.display_name,
                c.name AS cat_name, c.slug AS cat_slug,
                s.name AS sub_name, s.slug AS sub_slug,
                (SELECT COUNT(*) FROM media m WHERE m.post_id = p.id AND (m.comment_id IS NULL OR m.comment_id = 0)) AS media_count
            FROM posts p
            JOIN users u ON u.id = p.user_id
            JOIN subcategories s ON s.id = p.subcategory_id
            JOIN categories c ON c.id = s.category_id
            WHERE p.status = "pending"
            ORDER BY p.id DESC
            LIMIT 200')->fetchAll();

        $mediaByPost = [];
        if (!empty($posts)) {
            $postIds = array_map(static fn(array $row): int => (int)$row['id'], $posts);
            $placeholders = implode(',', array_fill(0, count($postIds), '?'));
            $mediaStmt = $db->prepare('SELECT id, post_id, original_name, media_kind, mime_type FROM media
                WHERE (comment_id IS NULL OR comment_id = 0)
                AND post_id IN (' . $placeholders . ')
                ORDER BY id ASC');
            foreach ($postIds as $idx => $pid) {
                $mediaStmt->bindValue($idx + 1, $pid, PDO::PARAM_INT);
            }
            $mediaStmt->execute();
            $mediaRows = $mediaStmt->fetchAll();

            foreach ($mediaRows as $mediaRow) {
                $postId = (int)($mediaRow['post_id'] ?? 0);
                if ($postId <= 0) {
                    continue;
                }
                $mediaByPost[$postId][] = $mediaRow;
            }
        }

        $html = '<section class="card"><h1>Pending Posts</h1><p class="muted">Approve to publish a post on the board, or decline to reject it.</p>';

        if (empty($posts)) {
            $html .= '<p>No pending posts right now.</p>';
        } else {
            $html .= '<ul class="thread-list">';
            foreach ($posts as $post) {
                $postId = (int)$post['id'];
                $mediaRows = $mediaByPost[$postId] ?? [];
                $html .= '<li class="thread-row"><div class="thread-row-body">'
                    . '<h3>#' . $postId . ' ' . e((string)$post['title']) . '</h3>'
                    . '<p>' . e(mb_substr(trim((string)$post['body']), 0, 400)) . '</p>'
                    . '<small>/' . e((string)$post['cat_slug']) . '/' . e((string)$post['sub_slug']) . '/ • ' . authorLinkFromRow($post) . ' • submitted ' . e((string)$post['created_at']) . '</small>';

                if (!empty($mediaRows)) {
                    $html .= '<div class="comment-media-list">';
                    foreach ($mediaRows as $media) {
                        $mediaUrl = '/media/' . (int)$media['id'];
                        $mediaLabel = mediaTypeLabel($media);
                        $mediaBadgeClass = mediaTypeBadgeClass($media);
                        if ((string)$media['media_kind'] === 'video') {
                            $html .= '<details class="comment-media-card"><summary><span class="media-thumb-wrap"><video muted playsinline preload="metadata" src="' . e($mediaUrl) . '"></video><span class="media-kind-badge media-kind-on-thumb ' . e($mediaBadgeClass) . '">' . e($mediaLabel) . '</span></span></summary><div class="comment-media-expanded"><video controls preload="metadata" src="' . e($mediaUrl) . '"></video><p class="media-caption"><span class="media-kind-badge ' . e($mediaBadgeClass) . '">' . e($mediaLabel) . '</span> ' . e((string)($media['original_name'] ?? '')) . '</p></div></details>';
                        } else {
                            $html .= '<details class="comment-media-card"><summary><span class="media-thumb-wrap"><img loading="lazy" decoding="async" fetchpriority="low" src="' . e(mediaThumbUrl((int)$media['id'])) . '" alt="Pending post media"><span class="media-kind-badge media-kind-on-thumb ' . e($mediaBadgeClass) . '">' . e($mediaLabel) . '</span></span></summary><div class="comment-media-expanded"><img loading="lazy" decoding="async" fetchpriority="low" src="' . e($mediaUrl) . '" alt="Pending post media expanded"><p class="media-caption"><span class="media-kind-badge ' . e($mediaBadgeClass) . '">' . e($mediaLabel) . '</span> ' . e((string)($media['original_name'] ?? '')) . '</p></div></details>';
                        }
                    }
                    $html .= '</div>';
                }

                $html .= '<div class="pending-actions">'
                    . '<form method="POST" class="inline-form">'
                    . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_approve_post"><input type="hidden" name="post_id" value="' . $postId . '">'
                    . '<button type="submit">Approve</button>'
                    . '</form>'
                    . '<form method="POST" class="inline-form admin-danger">'
                    . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_decline_post"><input type="hidden" name="post_id" value="' . $postId . '">'
                    . '<button type="submit">Decline</button>'
                    . '</form>'
                    . '</div>'
                    . '</div></li>';
            }
            $html .= '</ul>';
        }

        $html .= '</section>';
        renderLayout('Admin Pending Posts', $html, ['is_admin' => true, 'noindex' => true]);
        exit;
    }

    if ($adminPage === 'users') {
        $users = $db->query('SELECT id, username, email, role, is_banned, email_verified_at, created_at FROM users ORDER BY id DESC LIMIT 300')->fetchAll();

        $html = '<section class="card"><h1>User Management</h1>'
            . '<h2>Create admin user</h2>'
            . '<form method="POST" class="form-grid">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_create_admin">'
            . '<label>Username<input name="username" required></label>'
            . '<label>Email<input type="email" name="email" required></label>'
            . '<label>Password<input type="password" name="password" required minlength="10"></label>'
            . '<button type="submit">Create Admin</button>'
            . '</form><h2>All users</h2><ul class="clean-list">';

        foreach ($users as $user) {
            $verified = !empty($user['email_verified_at']) ? 'verified' : 'unverified';
            $html .= '<li>#' . (int)$user['id'] . ' @' . e((string)$user['username']) . ' [' . e((string)$user['role']) . '] ' . (((int)$user['is_banned']) === 1 ? '<strong>BANNED</strong>' : 'active') . ' • ' . $verified;
            if ($user['role'] !== 'admin') {
                $html .= '<form method="POST" class="inline-form"><input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value="">'
                    . '<input type="hidden" name="user_id" value="' . (int)$user['id'] . '">';
                if ((int)$user['is_banned'] === 1) {
                    $html .= '<input type="hidden" name="action" value="admin_unban_user"><button type="submit">Unban</button>';
                } else {
                    $html .= '<input type="hidden" name="action" value="admin_ban_user"><button type="submit">Ban</button>';
                }
                $html .= '</form>';
            }
            $html .= '</li>';
        }

        $html .= '</ul></section>';
        renderLayout('Admin Users', $html, ['is_admin' => true, 'noindex' => true]);
        exit;
    }

    if ($adminPage === 'settings') {
        $allowedIps = $db->query('SELECT ip_address, label, added_at FROM admin_allowed_ips ORDER BY id DESC')->fetchAll();

        $smtp = smtpSettings();

        $html = '<section class="card"><h1>Site Settings</h1><p class="muted">Hover the ⓘ icons for setting help.</p><form method="POST" enctype="multipart/form-data" class="form-grid settings-form">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_save_settings">'

            . '<h2>Brand & SEO</h2>'
            . '<label>Site name<input name="site_name" value="' . e(settingGet('site_name', DEFAULT_SITE_NAME)) . '" required></label>'
            . '<label>Tagline<input name="site_tagline" value="' . e(settingGet('site_tagline', '')) . '"></label>'
            . '<label>Legal contact email<input type="email" name="legal_email" value="' . e(settingGet('legal_email', DEFAULT_LEGAL_EMAIL)) . '" required></label>'
            . '<label>Default meta description<textarea name="site_meta_description" rows="3">' . e(settingGet('site_meta_description', '')) . '</textarea></label>'
            . '<label>Header brand display<select name="brand_display_mode">'
            . '<option value="favicon_and_name" ' . (settingGet('brand_display_mode', 'name_only') === 'favicon_and_name' ? 'selected' : '') . '>Favicon + Site name</option>'
            . '<option value="name_only" ' . (settingGet('brand_display_mode', 'name_only') === 'name_only' ? 'selected' : '') . '>Site name only</option>'
            . '<option value="favicon_only" ' . (settingGet('brand_display_mode', 'name_only') === 'favicon_only' ? 'selected' : '') . '>Favicon only</option>'
            . '</select></label>'
            . '<label>Favicon path (optional static path)<input name="site_favicon_url" value="' . e(settingGet('site_favicon_url', '')) . '" placeholder="/assets/favicon.ico"></label>'
            . '<label>Upload favicon (.ico, .png, .webp; max 1 MB)<input type="file" name="site_favicon_file" accept=".ico,image/x-icon,image/vnd.microsoft.icon,image/png,image/webp"></label>'
            . '<label><input type="checkbox" name="cursor_theme_enabled" ' . (settingGet('cursor_theme_enabled', '1') === '1' ? 'checked' : '') . '> Enable custom cursor theme (normal + hover)</label>'
            . '<label>Pagination per page<input type="number" name="pagination_per_page" min="5" max="100" value="' . e(settingGet('pagination_per_page', '20')) . '"></label>'

            . '<h2>Editable pages</h2>'
            . '<label>About page content<textarea name="about_page_content" rows="8">' . e(settingGet('about_page_content', '')) . '</textarea></label>'
            . '<label>DMCA page content<textarea name="dmca_page_content" rows="8">' . e(settingGet('dmca_page_content', '')) . '</textarea></label>'
            . '<label>Legal page content<textarea name="legal_page_content" rows="8">' . e(settingGet('legal_page_content', '')) . '</textarea></label>'
            . '<label>Terms of Service content<textarea name="terms_page_content" rows="8">' . e(settingGet('terms_page_content', '')) . '</textarea></label>'
            . '<label>Privacy Policy content<textarea name="privacy_page_content" rows="8">' . e(settingGet('privacy_page_content', '')) . '</textarea></label>'
            . '<label>Legal Notice content<textarea name="legal_notice_page_content" rows="8">' . e(settingGet('legal_notice_page_content', '')) . '</textarea></label>'

            . '<h2>Post disclaimer</h2>'
            . '<label><input type="checkbox" name="post_disclaimer_enabled" ' . (settingGet('post_disclaimer_enabled', '0') === '1' ? 'checked' : '') . '> Enable first-visit disclaimer before opening posts</label>'
            . '<label>Disclaimer title<input name="post_disclaimer_title" value="' . e(settingGet('post_disclaimer_title', 'Content Disclaimer')) . '" maxlength="120"></label>'
            . '<label>Disclaimer text<textarea name="post_disclaimer_body" rows="8">' . e(settingGet('post_disclaimer_body', '')) . '</textarea></label>'

            . '<h2>Default board rules</h2>'
            . '<label>Default max files per upload<input type="number" name="board_default_max_files" min="1" max="20" value="' . e(settingGet('board_default_max_files', '10')) . '"></label>'
            . '<label>Default max file size (MB)<input type="number" name="board_default_max_file_mb" min="1" max="50" value="' . e(settingGet('board_default_max_file_mb', '20')) . '"></label>'
            . '<label>Default cooldown seconds<input type="number" name="board_default_cooldown_seconds" min="0" max="3600" value="' . e(settingGet('board_default_cooldown_seconds', '10')) . '"></label>'
            . '<label>Default max posts/hour<input type="number" name="board_default_max_posts_per_hour" min="1" max="500" value="' . e(settingGet('board_default_max_posts_per_hour', '200')) . '"></label>'
            . '<label>Default max comments/hour<input type="number" name="board_default_max_comments_per_hour" min="1" max="1000" value="' . e(settingGet('board_default_max_comments_per_hour', '500')) . '"></label>'
            . '<label>Default CAPTCHA mode<select name="board_default_captcha_mode">'
            . '<option value="off" ' . (settingGet('board_default_captcha_mode', 'low_trust') === 'off' ? 'selected' : '') . '>Off</option>'
            . '<option value="low_trust" ' . (settingGet('board_default_captcha_mode', 'low_trust') === 'low_trust' ? 'selected' : '') . '>Low-trust users only</option>'
            . '<option value="all" ' . (settingGet('board_default_captcha_mode', 'low_trust') === 'all' ? 'selected' : '') . '>All users</option>'
            . '</select></label>'
            . '<label><input type="checkbox" name="auto_approve_new_posts" ' . (settingGet('auto_approve_new_posts', '1') === '1' ? 'checked' : '') . '> Auto-approve new posts (disable to require admin approval)</label>'

            . '<h2>Anti-spam limits</h2>'
            . '<label>Posts per 10 minutes (global)<input type="number" name="anti_spam_posts_per_10m" min="1" max="100" value="' . e(settingGet('anti_spam_posts_per_10m', '8')) . '"></label>'
            . '<label>Comments per 10 minutes (global)<input type="number" name="anti_spam_comments_per_10m" min="1" max="200" value="' . e(settingGet('anti_spam_comments_per_10m', '20')) . '"></label>'
            . '<label>Contact requests per hour<input type="number" name="anti_spam_contact_per_hour" min="1" max="100" value="' . e(settingGet('anti_spam_contact_per_hour', '5')) . '"></label>'
            . '<label>Low-trust account age (hours)<input type="number" name="low_trust_age_hours" min="1" max="720" value="' . e(settingGet('low_trust_age_hours', '72')) . '"></label>'
            . '<label>Low-trust minimum activity (posts+comments)<input type="number" name="low_trust_min_activity" min="0" max="1000" value="' . e(settingGet('low_trust_min_activity', '5')) . '"></label>'

            . '<h2>robots.txt & Sitemap Pings</h2>'
            . '<label>Custom robots rules<textarea name="robots_custom_rules" rows="8" placeholder="User-agent: *\nAllow: /\nDisallow: /admin\nSitemap: https://example.com/sitemap.xml">' . e(settingGet('robots_custom_rules', '')) . '</textarea></label>'
            . '<label><input type="checkbox" name="sitemap_auto_ping_enabled" ' . (settingGet('sitemap_auto_ping_enabled', '0') === '1' ? 'checked' : '') . '> Enable automatic sitemap ping hooks on content changes</label>'
            . '<label>Sitemap ping endpoints (one per line, use {sitemap_url})<textarea name="sitemap_ping_services" rows="4" placeholder="https://www.bing.com/ping?sitemap={sitemap_url}">' . e(settingGet('sitemap_ping_services', 'https://www.bing.com/ping?sitemap={sitemap_url}')) . '</textarea></label>'

            . '<h2>SMTP & Email Features</h2>'
            . '<label>SMTP Host<input name="smtp_host" value="' . e($smtp['host']) . '" placeholder="pro2.mail.ovh.net"></label>'
            . '<label>SMTP Port<input type="number" name="smtp_port" min="1" max="65535" value="' . (int)$smtp['port'] . '"></label>'
            . '<label>SMTP Username<input name="smtp_username" value="' . e($smtp['username']) . '" placeholder="support@example.com"></label>'
            . '<label>SMTP Password (leave empty to keep current)<input type="password" name="smtp_password" value=""></label>'
            . '<label>Security<select name="smtp_security">'
            . '<option value="tls" ' . ($smtp['security'] === 'tls' ? 'selected' : '') . '>tls (STARTTLS)</option>'
            . '<option value="ssl" ' . ($smtp['security'] === 'ssl' ? 'selected' : '') . '>ssl</option>'
            . '<option value="none" ' . ($smtp['security'] === 'none' ? 'selected' : '') . '>none</option>'
            . '</select></label>'
            . '<label>From email<input type="email" name="smtp_from_email" value="' . e($smtp['from_email']) . '"></label>'
            . '<label><input type="checkbox" name="email_verification_enabled" ' . (settingGet('email_verification_enabled', '0') === '1' ? 'checked' : '') . '> Enable email verification (requires SMTP)</label>'
            . '<label><input type="checkbox" name="password_reset_enabled" ' . (settingGet('password_reset_enabled', '0') === '1' ? 'checked' : '') . '> Enable password reset (requires SMTP)</label>'
            . '<label><input type="checkbox" name="contact_forward_enabled" ' . (settingGet('contact_forward_enabled', '0') === '1' ? 'checked' : '') . '> Forward contact form emails via SMTP</label>'
            . '<label><input type="checkbox" name="registration_enabled" ' . (settingGet('registration_enabled', '1') === '1' ? 'checked' : '') . '> Enable public registration</label>'

            . '<button type="submit">Save Settings</button>'
            . '</form>'
            . '<form method="POST" class="form-grid settings-form settings-form-spaced">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_send_test_email">'
            . '<label>SMTP test target email<input type="email" name="target_email" value="' . e(settingGet('legal_email', DEFAULT_LEGAL_EMAIL)) . '" required></label>'
            . '<button type="submit">Send SMTP test email</button>'
            . '</form></section>';

        $html .= '<section class="card"><h2>Database backup</h2>'
            . '<p class="muted">Download the current <code>site.sqlite</code> file.</p>'
            . '<form method="POST" class="form-grid">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_download_db">'
            . '<button type="submit">Download database (.sqlite)</button>'
            . '</form></section>';

        $html .= '<section class="card"><h2>Storage maintenance</h2>'
            . '<p class="muted">Remove files that are no longer referenced by the database/settings (orphan uploads, stale favicon files, stale thumbnail cache files).</p>'
            . '<form method="POST" class="form-grid">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_cleanup_orphan_files">'
            . '<button type="submit">Clean orphan files</button>'
            . '</form></section>';

        $html .= '<section class="card"><h2>Admin IP allowlist</h2>'
            . '<form method="POST" class="form-grid">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '"><input type="hidden" name="website" value=""><input type="hidden" name="action" value="admin_add_allow_ip">'
            . '<label>IP address<input name="ip_address" value="' . e(clientIp()) . '" required></label>'
            . '<label>Label<input name="label" placeholder="Home / Office"></label>'
            . '<button type="submit">Add / Update IP</button>'
            . '</form><ul class="clean-list">';

        foreach ($allowedIps as $ip) {
            $html .= '<li><code>' . e((string)$ip['ip_address']) . '</code> ' . e((string)($ip['label'] ?? '')) . ' '
                . '<form method="POST" class="inline-form">'
                . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
                . '<input type="hidden" name="website" value="">'
                . '<input type="hidden" name="action" value="admin_remove_allow_ip">'
                . '<input type="hidden" name="ip_address" value="' . e((string)$ip['ip_address']) . '">'
                . '<button type="submit">Remove</button></form></li>';
        }

        $html .= '</ul></section>';
        renderLayout('Admin Settings', $html, ['is_admin' => true, 'noindex' => true, 'scripts' => ['/assets/js/settings-tooltips.js']]);
        exit;
    }

    render404();
}

if ($route === 'account') {
    requireAuth();
    redirectTo(userProfileUrl((int)currentUser()['id']));
}

if (($routeParts[0] ?? '') === 'user' && isset($routeParts[1])) {
    $profileUserId = (int)$routeParts[1];
    if ($profileUserId <= 0) {
        render404();
    }

    $stmt = $db->prepare('SELECT id, username, display_name, role, created_at FROM users WHERE id = ? LIMIT 1');
    $stmt->execute([$profileUserId]);
    $profile = $stmt->fetch();
    if (!$profile) {
        render404();
    }

    $postCountStmt = $db->prepare('SELECT COUNT(*) FROM posts WHERE user_id = ? AND status = "active"');
    $postCountStmt->execute([$profileUserId]);
    $postCount = (int)$postCountStmt->fetchColumn();

    $commentCountStmt = $db->prepare('SELECT COUNT(*) FROM comments WHERE user_id = ? AND status = "active"');
    $commentCountStmt->execute([$profileUserId]);
    $commentCount = (int)$commentCountStmt->fetchColumn();

    $publicName = publicHandleFromRow($profile);
    $isOwner = isLoggedIn() && ((int)currentUser()['id'] === $profileUserId);

    $html = '<section class="card"><h1>@' . e($publicName) . '</h1>'
        . '<p class="muted">Member since ' . e((string)$profile['created_at']) . ' • ' . $postCount . ' posts • ' . $commentCount . ' comments</p>'
        . '<p>Account: @' . e((string)$profile['username']) . '</p>';

    if (trim((string)$profile['display_name']) === '') {
        $html .= '<p class="muted">This account currently posts as @Anonymous.</p>';
    }

    $html .= '</section>';

    if ($isOwner) {
        $html .= '<section class="card"><h2>Profile settings</h2><form method="POST" class="form-grid">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
            . '<input type="hidden" name="website" value="">'
            . '<input type="hidden" name="action" value="update_profile">'
            . '<label>Visible username for public posts/comments (leave empty for Anonymous)<input name="display_name" maxlength="30" pattern="\w{3,30}" value="' . e((string)$profile['display_name']) . '"></label>'
            . '<button type="submit">Save profile</button>'
            . '</form></section>';
    }

    renderLayout('User @' . $publicName, $html, [
        'description' => 'Public profile for @' . $publicName,
        'canonical' => absoluteUrl(userProfileUrl($profileUserId)),
    ]);
    exit;
}

if ($route === 'about') {
    $html = '<section class="card"><h1>About</h1><p>' . nl2br(e(settingGet('about_page_content', 'Write your About page in Admin Settings.'))) . '</p></section>';
    renderLayout('About', $html, ['description' => 'About this community imageboard.']);
    exit;
}

if ($route === 'dmca') {
    $email = settingGet('legal_email', DEFAULT_LEGAL_EMAIL);
    $html = '<section class="card"><h1>DMCA Policy</h1><p>' . nl2br(e(settingGet('dmca_page_content', 'Write your DMCA policy in Admin Settings.'))) . '</p><p>Contact: <a href="mailto:' . e($email) . '">' . e($email) . '</a></p></section>';
    renderLayout('DMCA', $html, ['description' => 'DMCA and copyright policy.']);
    exit;
}

if ($route === 'legal') {
    $html = '<section class="card"><h1>Legal</h1><p>' . nl2br(e(settingGet('legal_page_content', 'Write your Legal page in Admin Settings.'))) . '</p></section>';
    renderLayout('Legal', $html, ['description' => 'Legal terms and policies.']);
    exit;
}

if ($route === 'terms') {
    $html = '<section class="card"><h1>Terms of Service</h1><p>' . nl2br(e(settingGet('terms_page_content', 'Write your Terms of Service in Admin Settings.'))) . '</p></section>';
    renderLayout('Terms of Service', $html, ['description' => 'Terms of Service for using this website.']);
    exit;
}

if ($route === 'privacy') {
    $html = '<section class="card"><h1>Privacy Policy</h1><p>' . nl2br(e(settingGet('privacy_page_content', 'Write your Privacy Policy in Admin Settings.'))) . '</p></section>';
    renderLayout('Privacy Policy', $html, ['description' => 'Privacy Policy for this website.']);
    exit;
}

if ($route === 'legal-notice') {
    $html = '<section class="card"><h1>Legal Notice</h1><p>' . nl2br(e(settingGet('legal_notice_page_content', 'Write your Legal Notice in Admin Settings.'))) . '</p></section>';
    renderLayout('Legal Notice', $html, ['description' => 'Legal Notice and operator information.']);
    exit;
}

if ($route === 'cookie-settings') {
    $cookieChoice = cookieConsentState();
    if ($cookieChoice === '') {
        $cookieChoice = 'essential';
    }

    $html = '<section class="card"><h1>Cookie settings</h1>'
        . '<p>Essential cookies are required for secure login sessions and anti-abuse protections. You can optionally allow additional cookies.</p>'
        . '<form method="POST" class="form-grid">'
        . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
        . '<input type="hidden" name="website" value="">'
        . '<input type="hidden" name="action" value="cookie_settings_save">'
        . '<label><input type="radio" name="cookie_choice" value="accepted" ' . ($cookieChoice === 'accepted' ? 'checked' : '') . '> Accept cookies</label>'
        . '<label><input type="radio" name="cookie_choice" value="essential" ' . ($cookieChoice === 'essential' ? 'checked' : '') . '> Essential cookies only</label>'
        . '<button type="submit">Save cookie settings</button>'
        . '</form></section>';
    renderLayout('Cookie settings', $html, ['description' => 'Manage cookie consent preferences.']);
    exit;
}

if ($route === 'contact') {
    $html = '<section class="card"><h1>Contact</h1><form method="POST" class="form-grid">'
        . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
        . '<input type="hidden" name="website" value="">'
        . '<input type="hidden" name="action" value="contact_send">'
        . '<label>Name<input name="name" required></label>'
        . '<label>Email<input type="email" name="email" required></label>'
        . '<label>Message<textarea name="message" required rows="6"></textarea></label>'
        . '<button type="submit">Send</button>'
        . '</form></section>';
    renderLayout('Contact', $html, ['description' => 'Contact site administrators.']);
    exit;
}

if (($routeParts[0] ?? '') === 'board' && isset($routeParts[1], $routeParts[2])) {
    $catSlug = $routeParts[1];
    $subSlug = $routeParts[2];

    $stmt = $db->prepare('SELECT c.id AS category_id, c.name AS category_name, c.slug AS category_slug, sc.id AS sub_id, sc.name AS sub_name, sc.slug AS sub_slug, sc.description AS sub_desc
        FROM categories c
        JOIN subcategories sc ON sc.category_id = c.id
        WHERE c.slug = ? AND sc.slug = ?
        LIMIT 1');
    $stmt->execute([$catSlug, $subSlug]);
    $board = $stmt->fetch();

    if (!$board) {
        render404();
    }

    $rule = boardRuleForSubcategory((int)$board['sub_id']);
    $boardUserId = (int)(currentUser()['id'] ?? 0);
    $boardCaptchaNeeded = isLoggedIn() && captchaRequiredForBoardRule($rule, $boardUserId);
    $boardCaptcha = $boardCaptchaNeeded ? getCaptchaChallengeForBoard((int)$board['sub_id'], $boardUserId) : null;

    $perPage = paginationPerPage();
    $page = queryPage();

    $countStmt = $db->prepare('SELECT COUNT(*) FROM posts WHERE subcategory_id = ? AND status = "active"');
    $countStmt->execute([(int)$board['sub_id']]);
    $total = (int)$countStmt->fetchColumn();
    $pages = totalPages($total, $perPage);
    $page = min($page, $pages);

    $postsStmt = $db->prepare('SELECT p.id, p.title, p.body, p.created_at, u.id AS user_id, u.display_name,
        (SELECT COUNT(*) FROM comments c WHERE c.post_id = p.id AND c.status = "active") AS comment_count,
        (SELECT COUNT(*) FROM media m WHERE m.post_id = p.id AND (m.comment_id IS NULL OR m.comment_id = 0)) AS media_count,
        (SELECT m.id FROM media m WHERE m.post_id = p.id AND (m.comment_id IS NULL OR m.comment_id = 0) ORDER BY m.id ASC LIMIT 1) AS first_media_id,
        (SELECT m.media_kind FROM media m WHERE m.post_id = p.id AND (m.comment_id IS NULL OR m.comment_id = 0) ORDER BY m.id ASC LIMIT 1) AS first_media_kind
        FROM posts p JOIN users u ON u.id = p.user_id
        WHERE p.subcategory_id = ? AND p.status = "active"
        ORDER BY p.id DESC
        LIMIT :limit OFFSET :offset');
    $postsStmt->bindValue(1, (int)$board['sub_id'], PDO::PARAM_INT);
    $postsStmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
    $postsStmt->bindValue(':offset', pageOffset($page, $perPage), PDO::PARAM_INT);
    $postsStmt->execute();
    $posts = $postsStmt->fetchAll();

    $html = '<section class="card board-header"><h1>/' . e((string)$board['category_slug']) . '/' . e((string)$board['sub_slug']) . '/ - ' . e((string)$board['sub_name']) . '</h1>'
        . '<p>' . e((string)($board['sub_desc'] ?? '')) . '</p>'
        . '<p class="muted">Rules: max files ' . $rule['max_files'] . ', max ' . bytesToMb((int)$rule['max_file_bytes']) . ' MB/file, cooldown ' . $rule['cooldown_seconds'] . 's, video ' . ($rule['allow_video'] ? 'allowed' : 'disabled') . '.</p>'
        . (isLoggedIn() ? '<a class="corner-create" href="#create-post-modal">+ Post</a>' : '')
        . '</section>';

    if ((string)($_GET['post_pending'] ?? '') === '1') {
        $html .= '<section class="card"><div class="alert alert-success">Post submitted successfully and is pending admin approval.</div></section>';
    }

    if (isLoggedIn()) {
        $accept = $rule['allow_video'] ? 'image/jpeg,image/png,image/gif,image/webp,video/*' : 'image/jpeg,image/png,image/gif,image/webp';
        $supportedTip = $rule['allow_video']
            ? 'Supported files: JPEG, PNG, GIF, WEBP, WEBM. Non-WEBM videos can be converted to WEBM in-browser (FFmpeg first, fallback converter if needed), but conversion may reduce quality/FPS; for best quality upload WEBM directly.'
            : 'Supported files: JPEG, PNG, GIF, WEBP.';
        $html .= '<div id="create-post-modal" class="modal-overlay"><div class="modal-card">'
            . '<a class="modal-close" href="#" aria-label="Close">×</a>'
            . '<h2>Create post</h2><form id="create-post-form" method="POST" enctype="multipart/form-data" class="form-grid" data-upload-form="1" data-max-files="' . (int)$rule['max_files'] . '" data-max-file-bytes="' . (int)$rule['max_file_bytes'] . '">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
            . '<input type="hidden" name="website" value="">'
            . '<input type="hidden" name="action" value="create_post">'
            . '<input type="hidden" name="subcategory_id" value="' . (int)$board['sub_id'] . '">'
            . '<label>Title<input name="title" required minlength="3" maxlength="150"></label>'
            . '<label>Body<textarea name="body" rows="5" placeholder="Text is optional if you upload media."></textarea></label>'
            . '<label>Media (max ' . $rule['max_files'] . ' files, ' . bytesToMb((int)$rule['max_file_bytes']) . ' MB each)<input id="create-post-media-input" type="file" name="media[]" multiple accept="' . e($accept) . '" data-media-input="1"></label>'
            . '<p class="muted upload-supported-tip">' . e($supportedTip) . '</p>'
            . '<div class="upload-dropzone" data-upload-dropzone tabindex="0" role="button" aria-label="Drag and drop media for your post">Drag &amp; drop media here, or click to choose files.</div>'
            . '<p class="upload-feedback muted" data-upload-feedback hidden></p>'
            . '<div class="upload-preview-list" data-upload-preview></div>'
            . '<div class="upload-progress" data-upload-progress hidden><progress class="upload-progress-bar" max="100" value="0"></progress><p class="upload-progress-text" data-upload-progress-text>Uploading: 0%</p></div>'
            . ($boardCaptchaNeeded ? '<label class="captcha-box">CAPTCHA: ' . e((string)$boardCaptcha['question']) . '<input name="captcha_answer" required inputmode="numeric" pattern="[0-9]+"></label>' : '')
            . '<p class="muted legal-ack">By creating a post, you agree to our <a href="/terms">Terms of Service</a> and <a href="/privacy">Privacy Policy</a>.</p>'
            . '<button type="submit">Publish Post</button>'
            . '</form></div></div>';
    } else {
        $html .= '<section class="card"><p><a href="/login">Login</a> to publish posts and comments.</p></section>';
    }

    $html .= '<section class="card"><h2>Threads</h2>';
    if (empty($posts)) {
        $html .= '<p>No threads yet.</p>';
    } else {
        $html .= '<ul class="thread-list">';
        foreach ($posts as $post) {
            $firstMediaId = (int)($post['first_media_id'] ?? 0);
            $firstMediaKind = (string)($post['first_media_kind'] ?? 'image');
            $html .= '<li class="thread-row">';
            if ($firstMediaId > 0) {
                if ($firstMediaKind === 'video') {
                    $html .= '<a class="thread-preview-thumb" href="/post/' . (int)$post['id'] . '"><video muted playsinline preload="metadata" src="' . e('/media/' . $firstMediaId) . '"></video></a>';
                } else {
                    $html .= '<a class="thread-preview-thumb" href="/post/' . (int)$post['id'] . '"><img loading="lazy" decoding="async" fetchpriority="low" src="' . e(mediaThumbUrl($firstMediaId)) . '" alt="Thread preview"></a>';
                }
            }
            $html .= '<div class="thread-row-body"><h3><a href="/post/' . (int)$post['id'] . '">' . e((string)$post['title']) . '</a></h3>'
                . '<p>' . e(mb_substr((string)$post['body'], 0, 280)) . '</p>'
                . '<small>By ' . authorLinkFromRow($post) . ' • ' . e((string)$post['created_at']) . ' • ' . (int)$post['comment_count'] . ' comments • ' . (int)$post['media_count'] . ' files</small></div></li>';
        }
        $html .= '</ul>';
        $html .= renderPagination('/board/' . $board['category_slug'] . '/' . $board['sub_slug'], [], $page, $pages);
    }
    $html .= '</section>';

    renderLayout('Board ' . $board['sub_name'], $html, [
        'description' => 'Threads for board /' . $board['category_slug'] . '/' . $board['sub_slug'] . '/.',
        'canonical' => absoluteUrl('/board/' . $board['category_slug'] . '/' . $board['sub_slug'] . ($page > 1 ? '?page=' . $page : '')),
        'noindex' => $page > 1,
        'scripts' => ['/assets/js/comment-upload.js'],
    ]);
    exit;
}

if (($routeParts[0] ?? '') === 'post' && isset($routeParts[1])) {
    $postId = (int)$routeParts[1];

    $disclaimerEnabled = settingGet('post_disclaimer_enabled', '0') === '1';
    if ($disclaimerEnabled && !hasAcceptedPostDisclaimer()) {
        $defaultDisclaimer = "To access this section of the website, you understand and agree to the following:\n\n"
            . "The content in this section may be intended for mature audiences and may not be suitable for minors. If you are not legally permitted to view this type of content, do not proceed.\n\n"
            . "This website is provided AS IS without warranties. By selecting I Agree, you accept responsibility for your use of the site and acknowledge that user content is posted by community members.\n\n"
            . "As a condition of use, you agree to comply with this website's Rules and legal policies.";
        $disclaimerTitle = trim(settingGet('post_disclaimer_title', 'Content Disclaimer'));
        if ($disclaimerTitle === '') {
            $disclaimerTitle = 'Content Disclaimer';
        }
        $disclaimerBody = trim(settingGet('post_disclaimer_body', $defaultDisclaimer));
        $redirectPath = safeLocalPath((string)($_SERVER['REQUEST_URI'] ?? ('/post/' . $postId)), '/post/' . $postId);

        $html = '<section class="card"><h1>' . e($disclaimerTitle) . '</h1>'
            . '<p>' . nl2br(e($disclaimerBody)) . '</p>'
            . '<div class="disclaimer-actions">'
            . '<form method="POST" class="inline-form">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
            . '<input type="hidden" name="website" value="">'
            . '<input type="hidden" name="action" value="accept_post_disclaimer">'
            . '<input type="hidden" name="redirect_path" value="' . e($redirectPath) . '">'
            . '<button type="submit">I Agree</button>'
            . '</form>'
            . '<form method="POST" class="inline-form">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
            . '<input type="hidden" name="website" value="">'
            . '<input type="hidden" name="action" value="decline_post_disclaimer">'
            . '<button type="submit" class="button-secondary">Cancel</button>'
            . '</form>'
            . '</div></section>';

        renderLayout($disclaimerTitle, $html, [
            'noindex' => true,
            'canonical' => absoluteUrl('/post/' . $postId),
        ]);
        exit;
    }

    $stmt = $db->prepare('SELECT p.*, u.id AS user_id, u.display_name, sc.slug AS sub_slug, c.slug AS cat_slug, sc.name AS sub_name FROM posts p
        JOIN users u ON u.id = p.user_id
        JOIN subcategories sc ON sc.id = p.subcategory_id
        JOIN categories c ON c.id = sc.category_id
        WHERE p.id = ? AND p.status = "active"
        LIMIT 1');
    $stmt->execute([$postId]);
    $post = $stmt->fetch();

    if (!$post) {
        render404();
    }

    $mediaStmt = $db->prepare('SELECT * FROM media WHERE post_id = ? AND (comment_id IS NULL OR comment_id = 0) ORDER BY id ASC');
    $mediaStmt->execute([$postId]);
    $mediaRows = $mediaStmt->fetchAll();

    $commentsPerPage = max(10, min(100, paginationPerPage()));
    $commentPage = queryPage();

    $countCommentsStmt = $db->prepare('SELECT COUNT(*) FROM comments WHERE post_id = ? AND status = "active"');
    $countCommentsStmt->execute([$postId]);
    $commentsTotal = (int)$countCommentsStmt->fetchColumn();
    $commentPages = totalPages($commentsTotal, $commentsPerPage);
    $commentPage = min($commentPage, $commentPages);

    $commentsStmt = $db->prepare('SELECT c.*, u.id AS user_id, u.display_name FROM comments c JOIN users u ON u.id = c.user_id WHERE c.post_id = ? AND c.status = "active" ORDER BY c.id ASC LIMIT :limit OFFSET :offset');
    $commentsStmt->bindValue(1, $postId, PDO::PARAM_INT);
    $commentsStmt->bindValue(':limit', $commentsPerPage, PDO::PARAM_INT);
    $commentsStmt->bindValue(':offset', pageOffset($commentPage, $commentsPerPage), PDO::PARAM_INT);
    $commentsStmt->execute();
    $comments = $commentsStmt->fetchAll();

    $commentMediaStmt = $db->prepare('SELECT * FROM media WHERE post_id = ? AND comment_id IS NOT NULL AND comment_id > 0 ORDER BY id ASC');
    $commentMediaStmt->execute([$postId]);
    $commentMediaRows = $commentMediaStmt->fetchAll();
    $commentMediaByComment = [];
    foreach ($commentMediaRows as $mediaRow) {
        $cid = (int)($mediaRow['comment_id'] ?? 0);
        if ($cid <= 0) {
            continue;
        }
        $commentMediaByComment[$cid][] = $mediaRow;
    }

    $commentRule = boardRuleForSubcategory((int)$post['subcategory_id']);
    $commentUserId = (int)(currentUser()['id'] ?? 0);
    $commentCaptchaNeeded = isLoggedIn() && captchaRequiredForBoardRule($commentRule, $commentUserId);
    $commentCaptcha = $commentCaptchaNeeded ? getCaptchaChallengeForBoard((int)$post['subcategory_id'], $commentUserId) : null;

    $replyToUserId = max(0, (int)($_GET['reply_to'] ?? 0));
    $replyToCommentId = (int)($_GET['reply_to_comment'] ?? -1);
    $replyToUser = null;
    if ($replyToUserId > 0) {
        $replyStmt = $db->prepare('SELECT id, display_name FROM users WHERE id = ? LIMIT 1');
        $replyStmt->execute([$replyToUserId]);
        $replyToUser = $replyStmt->fetch() ?: null;
    }
    $replyPrefix = '';
    if ($replyToCommentId === 0) {
        $replyPrefix = '>>OP ';
    } elseif ($replyToCommentId > 0) {
        $replyPrefix = '>>' . $replyToCommentId . ' ';
    }

    $html = '<section id="post-op" class="card thread-post"><h1>' . e((string)$post['title']) . '</h1>'
        . '<p class="crumb"><a href="/board/' . e((string)$post['cat_slug']) . '/' . e((string)$post['sub_slug']) . '">Back to /' . e((string)$post['cat_slug']) . '/' . e((string)$post['sub_slug']) . '/</a></p>'
        . '<p>' . nl2br(e((string)$post['body'])) . '</p>'
        . '<small>By ' . authorLinkFromRow($post) . ' • ' . e((string)$post['created_at'])
        . ((int)$post['user_id'] > 0 ? ' • <a href="/post/' . $postId . '?reply_to=' . (int)$post['user_id'] . '&reply_to_comment=0#reply-form">Reply</a>' : '')
        . '</small></section>';

    if (!empty($mediaRows)) {
        $html .= '<section class="card"><h2>Media</h2><div class="media-grid">';
        foreach ($mediaRows as $media) {
            $url = '/media/' . (int)$media['id'];
            $mediaLabel = mediaTypeLabel($media);
            $mediaBadgeClass = mediaTypeBadgeClass($media);
            if ($media['media_kind'] === 'video') {
                $html .= '<details class="media-card"><summary><span class="media-thumb-wrap"><video muted playsinline preload="metadata" src="' . e($url) . '"></video><span class="media-kind-badge media-kind-on-thumb ' . e($mediaBadgeClass) . '">' . e($mediaLabel) . '</span></span></summary><div class="media-expanded"><video controls preload="metadata" src="' . e($url) . '"></video><p class="media-caption"><span class="media-kind-badge ' . e($mediaBadgeClass) . '">' . e($mediaLabel) . '</span> ' . e((string)$media['original_name']) . '</p></div></details>';
            } else {
                $html .= '<details class="media-card"><summary><span class="media-thumb-wrap"><img loading="lazy" decoding="async" fetchpriority="low" src="' . e(mediaThumbUrl((int)$media['id'])) . '" alt="Uploaded image"><span class="media-kind-badge media-kind-on-thumb ' . e($mediaBadgeClass) . '">' . e($mediaLabel) . '</span></span></summary><div class="media-expanded"><img loading="lazy" decoding="async" fetchpriority="low" src="' . e($url) . '" alt="Uploaded image expanded"><p class="media-caption"><span class="media-kind-badge ' . e($mediaBadgeClass) . '">' . e($mediaLabel) . '</span> ' . e((string)$media['original_name']) . '</p></div></details>';
            }
        }
        $html .= '</div></section>';
    }

    $html .= '<section class="card"><h2>Comments</h2>';
    if (empty($comments)) {
        $html .= '<p>No comments yet.</p>';
    } else {
        $html .= '<ul class="clean-list">';
        foreach ($comments as $comment) {
            $commentId = (int)$comment['id'];
            $html .= '<li id="comment-' . $commentId . '" class="comment-item"><strong>' . authorLinkFromRow($comment) . '</strong> <small>' . e((string)$comment['created_at'])
                . ((int)$comment['user_id'] > 0 ? ' • <a href="/post/' . $postId . '?reply_to=' . (int)$comment['user_id'] . '&reply_to_comment=' . (int)$comment['id'] . '#reply-form">Reply</a>' : '')
                . '</small><p>' . renderCommentBodyWithMentions((string)$comment['body']) . '</p>';

            $commentMedia = $commentMediaByComment[$commentId] ?? [];
            if (!empty($commentMedia)) {
                $html .= '<div class="comment-media-list">';
                foreach ($commentMedia as $cMedia) {
                    $mediaUrl = '/media/' . (int)$cMedia['id'];
                    $commentMediaLabel = mediaTypeLabel($cMedia);
                    $commentMediaBadgeClass = mediaTypeBadgeClass($cMedia);
                    if (($cMedia['media_kind'] ?? 'image') === 'video') {
                        $html .= '<details class="comment-media-card"><summary><span class="media-thumb-wrap"><video muted playsinline preload="metadata" src="' . e($mediaUrl) . '"></video><span class="media-kind-badge media-kind-on-thumb ' . e($commentMediaBadgeClass) . '">' . e($commentMediaLabel) . '</span></span></summary><div class="comment-media-expanded"><video controls preload="metadata" src="' . e($mediaUrl) . '"></video><p class="media-caption"><span class="media-kind-badge ' . e($commentMediaBadgeClass) . '">' . e($commentMediaLabel) . '</span> ' . e((string)($cMedia['original_name'] ?? '')) . '</p></div></details>';
                    } else {
                        $html .= '<details class="comment-media-card"><summary><span class="media-thumb-wrap"><img loading="lazy" decoding="async" fetchpriority="low" src="' . e(mediaThumbUrl((int)$cMedia['id'])) . '" alt="Comment media"><span class="media-kind-badge media-kind-on-thumb ' . e($commentMediaBadgeClass) . '">' . e($commentMediaLabel) . '</span></span></summary><div class="comment-media-expanded"><img loading="lazy" decoding="async" fetchpriority="low" src="' . e($mediaUrl) . '" alt="Comment media expanded"><p class="media-caption"><span class="media-kind-badge ' . e($commentMediaBadgeClass) . '">' . e($commentMediaLabel) . '</span> ' . e((string)($cMedia['original_name'] ?? '')) . '</p></div></details>';
                    }
                }
                $html .= '</div>';
            }

            $html .= '</li>';
        }
        $html .= '</ul>';
        $html .= renderPagination('/post/' . $postId, [], $commentPage, $commentPages);
    }

    if (isLoggedIn()) {
        $commentAccept = $commentRule['allow_video'] ? 'image/jpeg,image/png,image/gif,image/webp,video/*' : 'image/jpeg,image/png,image/gif,image/webp';
        $commentSupportedTip = $commentRule['allow_video']
            ? 'Supported files: JPEG, PNG, GIF, WEBP, WEBM. Non-WEBM videos can be converted to WEBM in-browser (FFmpeg first, fallback converter if needed), but conversion may reduce quality/FPS; for best quality upload WEBM directly.'
            : 'Supported files: JPEG, PNG, GIF, WEBP.';
        $html .= '<form id="reply-form" method="POST" class="form-grid" data-upload-form="1" data-max-files="' . (int)$commentRule['max_files'] . '" data-max-file-bytes="' . (int)$commentRule['max_file_bytes'] . '">'
            . '<input type="hidden" name="csrf_token" value="' . e(csrfToken()) . '">'
            . '<input type="hidden" name="website" value="">'
            . '<input type="hidden" name="action" value="add_comment">'
            . '<input type="hidden" name="post_id" value="' . (int)$post['id'] . '">'
            . ($replyToUser ? '<p class="muted">Replying to @' . e(publicHandleFromRow($replyToUser)) . '</p>' : '')
            . '<label>Add comment<textarea name="body" rows="4">' . e($replyPrefix) . '</textarea></label>'
            . '<label>Optional media (max ' . $commentRule['max_files'] . ' files, ' . bytesToMb((int)$commentRule['max_file_bytes']) . ' MB each)<input id="comment-media-input" type="file" name="comment_media[]" multiple accept="' . e($commentAccept) . '" data-media-input="1"></label>'
            . '<p class="muted upload-supported-tip">' . e($commentSupportedTip) . '</p>'
            . '<div class="upload-dropzone" data-upload-dropzone tabindex="0" role="button" aria-label="Drag and drop media for your comment">Drag &amp; drop media here, or click to choose files.</div>'
            . '<p class="upload-feedback muted" data-upload-feedback hidden></p>'
            . '<div class="upload-preview-list" data-upload-preview></div>'
            . '<div class="upload-progress" data-upload-progress hidden><progress class="upload-progress-bar" max="100" value="0"></progress><p class="upload-progress-text" data-upload-progress-text>Uploading: 0%</p></div>'
            . ($commentCaptchaNeeded ? '<label class="captcha-box">CAPTCHA: ' . e((string)$commentCaptcha['question']) . '<input name="captcha_answer" required inputmode="numeric" pattern="[0-9]+"></label>' : '')
            . '<p class="muted legal-ack">By posting a comment, you agree to our <a href="/terms">Terms of Service</a> and <a href="/privacy">Privacy Policy</a>.</p>'
            . '<button type="submit">Post Comment</button>'
            . '</form>';
    } else {
        $html .= '<p><a href="/login">Login</a> to comment.</p>';
    }

    $html .= '</section>';

    renderLayout('Post #' . $postId, $html, [
        'description' => mb_substr(trim((string)$post['body']), 0, 160) !== '' ? mb_substr(trim((string)$post['body']), 0, 160) : ('Thread: ' . (string)$post['title']),
        'canonical' => absoluteUrl('/post/' . $postId . ($commentPage > 1 ? '?page=' . $commentPage : '')),
        'noindex' => $commentPage > 1,
        'body_class' => 'page-post',
        'scripts' => ['/assets/js/comment-upload.js', '/assets/js/media-viewer.js'],
    ]);
    exit;
}

$categories = $db->query('SELECT c.id, c.name, c.slug, c.description,
    (SELECT COUNT(*) FROM subcategories s WHERE s.category_id = c.id) AS sub_count
    FROM categories c ORDER BY c.sort_order ASC, c.name ASC')->fetchAll();

$subs = $db->query('SELECT sc.id, sc.name, sc.slug, sc.description, sc.category_id, c.slug AS cat_slug, c.name AS cat_name,
    (SELECT COUNT(*) FROM posts p WHERE p.subcategory_id = sc.id AND p.status = "active") AS post_count
    FROM subcategories sc
    JOIN categories c ON c.id = sc.category_id
    ORDER BY c.sort_order ASC, sc.sort_order ASC, sc.name ASC')->fetchAll();

$subsByCategory = [];
foreach ($subs as $sub) {
    $subsByCategory[(int)$sub['category_id']][] = $sub;
}

$perPage = paginationPerPage();
$page = queryPage();
$totalThreads = (int)$db->query('SELECT COUNT(*) FROM posts WHERE status = "active"')->fetchColumn();
$pages = totalPages($totalThreads, $perPage);
$page = min($page, $pages);

$latestStmt = $db->prepare('SELECT p.id, p.title, p.body, p.created_at, u.id AS user_id, u.display_name, c.slug AS cat_slug, s.slug AS sub_slug
    FROM posts p
    JOIN users u ON u.id = p.user_id
    JOIN subcategories s ON s.id = p.subcategory_id
    JOIN categories c ON c.id = s.category_id
    WHERE p.status = "active"
    ORDER BY p.id DESC
    LIMIT :limit OFFSET :offset');
$latestStmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$latestStmt->bindValue(':offset', pageOffset($page, $perPage), PDO::PARAM_INT);
$latestStmt->execute();
$latestThreads = $latestStmt->fetchAll();

$html = '';

foreach ($categories as $category) {
    $html .= '<section class="card"><h2>' . e((string)$category['name']) . '</h2>';
    if (!empty($category['description'])) {
        $html .= '<p>' . e((string)$category['description']) . '</p>';
    }

    $html .= '<ul class="clean-list">';
    foreach ($subsByCategory[(int)$category['id']] ?? [] as $sub) {
        $html .= '<li><a href="/board/' . e((string)$sub['cat_slug']) . '/' . e((string)$sub['slug']) . '">' . e((string)$sub['cat_name']) . ' &gt; ' . e((string)$sub['name']) . '</a> <small>(' . (int)$sub['post_count'] . ' threads)</small></li>';
    }
    $html .= '</ul></section>';
}

$html .= '<section class="card"><h2>Latest threads</h2>';
if (empty($latestThreads)) {
    $html .= '<p>No threads yet.</p>';
} else {
    $html .= '<ul class="thread-list">';
    foreach ($latestThreads as $thread) {
        $html .= '<li class="thread-row">';
        $html .= '<div class="thread-row-body"><h3><a href="/post/' . (int)$thread['id'] . '">' . e((string)$thread['title']) . '</a></h3>'
            . '<p>' . e(mb_substr((string)$thread['body'], 0, 220)) . '</p>'
            . '<small>/' . e((string)$thread['cat_slug']) . '/' . e((string)$thread['sub_slug']) . '/ • ' . authorLinkFromRow($thread) . '</small></div></li>';
    }
    $html .= '</ul>';
    $html .= renderPagination('/', [], $page, $pages);
}
$html .= '</section>';

renderLayout('Home', $html, [
    'description' => settingGet('site_meta_description', 'Community imageboard with secure media uploads and moderation.'),
    'canonical' => absoluteUrl('/' . ($page > 1 ? '?page=' . $page : '')),
    'noindex' => $page > 1,
]);
