(() => {
    const VIDEO_EXT_RE = /\.(mp4|mov|m4v|avi|mkv|wmv|mpeg|mpg|ogv|webm)$/i;
    const BROWSER_CONVERSION_WARNING = 'Browser conversion may reduce quality, FPS smoothness, and compression efficiency. For best quality, upload WEBM directly when possible.';

    function absoluteAssetUrl(path) {
        try {
            return new URL(path, globalThis.location.href).toString();
        } catch {
            return path;
        }
    }

    const FFMPEG_SCRIPT_URL = absoluteAssetUrl('/assets/vendor/ffmpeg/ffmpeg.min.js');
    const FFMPEG_CORE_URL = absoluteAssetUrl('/assets/vendor/ffmpeg/ffmpeg-core.js');
    const FFMPEG_WORKER_URL = absoluteAssetUrl('/assets/vendor/ffmpeg/ffmpeg-core.worker.js');
    const FFMPEG_LOAD_TIMEOUT_MS = 30000;

    const loadedScriptPromises = new Map();
    let ffmpegLoaderPromise = null;
    let ffmpegWorkerProbePromise = null;
    let ffmpegDisabledReason = '';

    function formatBytes(bytes) {
        const size = Number(bytes) || 0;
        if (size <= 0) {
            return '0 B';
        }

        const units = ['B', 'KB', 'MB', 'GB'];
        let value = size;
        let unitIndex = 0;
        while (value >= 1024 && unitIndex < units.length - 1) {
            value /= 1024;
            unitIndex += 1;
        }
        return `${value.toFixed(value >= 10 || unitIndex === 0 ? 0 : 1)} ${units[unitIndex]}`;
    }

    function setFeedback(feedbackEl, message, isError) {
        if (!feedbackEl) {
            return;
        }

        const text = String(message || '').trim();
        if (text === '') {
            feedbackEl.hidden = true;
            feedbackEl.textContent = '';
            feedbackEl.classList.remove('is-error');
            return;
        }

        feedbackEl.hidden = false;
        feedbackEl.textContent = text;
        feedbackEl.classList.toggle('is-error', Boolean(isError));
    }

    function setInputFiles(fileInput, files) {
        if (!globalThis.DataTransfer) {
            return false;
        }

        const dt = new DataTransfer();
        files.forEach((file) => {
            dt.items.add(file);
        });
        fileInput.files = dt.files;
        return true;
    }

    function loadScriptOnce(src) {
        if (!src) {
            return Promise.reject(new Error('Missing script source.'));
        }

        const existing = loadedScriptPromises.get(src);
        if (existing) {
            return existing;
        }

        const promise = new Promise((resolve, reject) => {
            const script = document.createElement('script');
            script.src = src;
            script.async = true;
            script.onload = () => resolve();
            script.onerror = () => reject(new Error(`Failed to load ${src}`));
            document.head.appendChild(script);
        });

        loadedScriptPromises.set(src, promise);
        return promise;
    }

    function withTimeout(promise, timeoutMs, message) {
        return new Promise((resolve, reject) => {
            let settled = false;
            const timer = globalThis.setTimeout(() => {
                if (settled) {
                    return;
                }
                settled = true;
                reject(new Error(message));
            }, Math.max(1000, Number(timeoutMs) || 1000));

            Promise.resolve(promise)
                .then((value) => {
                    if (settled) {
                        return;
                    }
                    settled = true;
                    clearTimeout(timer);
                    resolve(value);
                })
                .catch((error) => {
                    if (settled) {
                        return;
                    }
                    settled = true;
                    clearTimeout(timer);
                    reject(error);
                });
        });
    }

    async function ensureFfmpegWorkerAsset() {
        if (ffmpegWorkerProbePromise) {
            return ffmpegWorkerProbePromise;
        }

        ffmpegWorkerProbePromise = (async () => {
            if (typeof globalThis.fetch !== 'function') {
                return;
            }

            const response = await withTimeout(
                globalThis.fetch(FFMPEG_WORKER_URL, {
                    method: 'GET',
                    credentials: 'same-origin',
                    cache: 'no-store',
                }),
                10000,
                'Timed out while checking FFmpeg worker asset.'
            );

            if (!response || !response.ok) {
                throw new Error(`FFmpeg worker asset missing at ${FFMPEG_WORKER_URL}`);
            }

            const contentType = String(response.headers.get('content-type') || '').toLowerCase();
            if (contentType.includes('text/html')) {
                throw new Error(`FFmpeg worker asset path resolved to HTML instead of JavaScript: ${FFMPEG_WORKER_URL}`);
            }
        })();

        try {
            await ffmpegWorkerProbePromise;
        } catch (error) {
            ffmpegWorkerProbePromise = null;
            throw error;
        }
    }

    function randomToken(length) {
        const chars = 'abcdefghijklmnopqrstuvwxyz0123456789';
        let out = '';
        for (let i = 0; i < length; i += 1) {
            out += chars[Math.floor(Math.random() * chars.length)];
        }
        return out;
    }

    function ffmpegTempName(prefix, fileName, fallbackExt) {
        const base = fileNameWithoutExt(fileName || 'video');
        const extMatch = String(fileName || '').match(/\.([a-z0-9]+)$/i);
        const ext = (extMatch && extMatch[1] ? extMatch[1] : fallbackExt).toLowerCase();
        return `${prefix}-${Date.now()}-${randomToken(6)}-${base.replace(/[^a-z0-9_-]+/gi, '_')}.${ext}`;
    }

    async function getFfmpegRuntime() {
        if (ffmpegDisabledReason !== '') {
            throw new Error(ffmpegDisabledReason);
        }

        if (typeof globalThis.SharedArrayBuffer === 'undefined' || globalThis.crossOriginIsolated !== true) {
            throw new Error('SharedArrayBuffer is unavailable. FFmpeg requires Cross-Origin-Opener-Policy: same-origin and Cross-Origin-Embedder-Policy: require-corp.');
        }

        if (!ffmpegLoaderPromise) {
            ffmpegLoaderPromise = (async () => {
                if (!(globalThis.FFmpeg && typeof globalThis.FFmpeg.createFFmpeg === 'function')) {
                    await loadScriptOnce(FFMPEG_SCRIPT_URL);
                }

                const ffmpegApi = globalThis.FFmpeg;
                if (!ffmpegApi || typeof ffmpegApi.createFFmpeg !== 'function' || typeof ffmpegApi.fetchFile !== 'function') {
                    throw new Error(`FFmpeg WASM assets unavailable. Expected ${FFMPEG_SCRIPT_URL} and core files.`);
                }

                await ensureFfmpegWorkerAsset();

                const ffmpeg = ffmpegApi.createFFmpeg({
                    log: false,
                    corePath: FFMPEG_CORE_URL,
                });

                await withTimeout(
                    ffmpeg.load(),
                    FFMPEG_LOAD_TIMEOUT_MS,
                    'FFmpeg initialization timed out (possible CSP worker/script restriction).'
                );

                return {
                    ffmpeg,
                    fetchFile: ffmpegApi.fetchFile,
                };
            })();
        }

        try {
            return await ffmpegLoaderPromise;
        } catch (error) {
            ffmpegLoaderPromise = null;
            ffmpegDisabledReason = error instanceof Error ? error.message : 'FFmpeg unavailable';
            throw error;
        }
    }

    function isVideoFile(file) {
        const type = String(file?.type || '').toLowerCase();
        if (type.startsWith('video/')) {
            return true;
        }

        return VIDEO_EXT_RE.test(String(file?.name || ''));
    }

    function isWebmVideo(file) {
        const type = String(file?.type || '').toLowerCase();
        if (type === 'video/webm') {
            return true;
        }

        return /\.webm$/i.test(String(file?.name || ''));
    }

    function fileNameWithoutExt(name) {
        const value = String(name || 'video').trim();
        const dotIndex = value.lastIndexOf('.');
        if (dotIndex <= 0) {
            return value || 'video';
        }
        return value.slice(0, dotIndex);
    }

    function waitForMediaEvent(target, eventName, timeoutMs, errorMessage) {
        return new Promise((resolve, reject) => {
            let settled = false;
            let timer = null;

            const onSuccess = () => {
                if (settled) {
                    return;
                }
                settled = true;
                cleanup();
                resolve();
            };

            const onError = () => {
                if (settled) {
                    return;
                }
                settled = true;
                cleanup();
                reject(new Error(errorMessage));
            };

            const cleanup = () => {
                target.removeEventListener(eventName, onSuccess);
                target.removeEventListener('error', onError);
                if (timer !== null) {
                    clearTimeout(timer);
                }
            };

            timer = globalThis.setTimeout(() => {
                if (settled) {
                    return;
                }
                settled = true;
                cleanup();
                reject(new Error(errorMessage));
            }, Math.max(1000, timeoutMs));

            target.addEventListener(eventName, onSuccess, { once: true });
            target.addEventListener('error', onError, { once: true });
        });
    }

    async function convertVideoToWebmInBrowser(file, onProgress) {
        if (typeof MediaRecorder === 'undefined') {
            throw new Error('This browser does not support MediaRecorder conversion.');
        }

        const sourceUrl = URL.createObjectURL(file);
        const video = document.createElement('video');
        video.preload = 'auto';
        video.playsInline = true;
        video.muted = true;
        video.hidden = true;
        video.src = sourceUrl;
        document.body.appendChild(video);

        let mediaStream = null;

        try {
            await waitForMediaEvent(video, 'loadedmetadata', 20000, 'Could not read video metadata.');

            if (!Number.isFinite(video.duration) || video.duration <= 0) {
                throw new Error('Unsupported or empty video.');
            }

            if (typeof video.captureStream !== 'function') {
                throw new Error('captureStream is not available in this browser.');
            }

            mediaStream = video.captureStream();

            const mimeCandidates = [
                'video/webm;codecs=vp9,opus',
                'video/webm;codecs=vp8,opus',
                'video/webm',
            ];

            let selectedMime = '';
            for (const candidate of mimeCandidates) {
                if (MediaRecorder.isTypeSupported(candidate)) {
                    selectedMime = candidate;
                    break;
                }
            }

            if (selectedMime === '') {
                throw new Error('No WEBM encoder available in this browser.');
            }

            const chunks = [];
            const recorder = new MediaRecorder(mediaStream, { mimeType: selectedMime });

            recorder.addEventListener('dataavailable', (event) => {
                if (event.data && event.data.size > 0) {
                    chunks.push(event.data);
                }
            });

            const stopPromise = new Promise((resolve, reject) => {
                recorder.addEventListener('stop', resolve, { once: true });
                recorder.addEventListener('error', () => reject(new Error('Video conversion failed.')), { once: true });
            });

            const endedPromise = waitForMediaEvent(video, 'ended', Math.max(30000, Math.ceil(video.duration * 1000) + 10000), 'Video playback ended unexpectedly during conversion.');

            video.addEventListener('timeupdate', () => {
                if (!Number.isFinite(video.duration) || video.duration <= 0 || typeof onProgress !== 'function') {
                    return;
                }
                const percent = Math.max(1, Math.min(99, Math.round((video.currentTime / video.duration) * 100)));
                onProgress(percent);
            });

            recorder.start(1000);

            const playPromise = video.play();
            if (playPromise && typeof playPromise.then === 'function') {
                await playPromise;
            }

            await endedPromise;
            if (recorder.state !== 'inactive') {
                recorder.stop();
            }
            await stopPromise;

            if (typeof onProgress === 'function') {
                onProgress(100);
            }

            const outputBlob = new Blob(chunks, { type: 'video/webm' });
            if (outputBlob.size <= 0) {
                throw new Error('Conversion produced an empty file.');
            }

            const outputName = `${fileNameWithoutExt(file.name)}.webm`;
            return new File([outputBlob], outputName, {
                type: 'video/webm',
                lastModified: Date.now(),
            });
        } finally {
            if (mediaStream) {
                mediaStream.getTracks().forEach((track) => {
                    try {
                        track.stop();
                    } catch {
                        // no-op cleanup guard
                    }
                });
            }

            URL.revokeObjectURL(sourceUrl);
            video.removeAttribute('src');
            video.load();
            video.remove();
        }
    }

    async function convertVideoToWebmWithFfmpeg(file, onProgress) {
        const runtime = await getFfmpegRuntime();
        const ffmpeg = runtime.ffmpeg;
        const fetchFile = runtime.fetchFile;

        const inputName = ffmpegTempName('input', file.name, 'mp4');
        const outputName = ffmpegTempName('output', `${fileNameWithoutExt(file.name)}.webm`, 'webm');

        if (typeof ffmpeg.setProgress === 'function' && typeof onProgress === 'function') {
            ffmpeg.setProgress(({ ratio }) => {
                const raw = Number(ratio) || 0;
                const percent = Math.max(1, Math.min(99, Math.round(raw * 100)));
                onProgress(percent);
            });
        }

        try {
            ffmpeg.FS('writeFile', inputName, await fetchFile(file));
            await ffmpeg.run(
                '-i', inputName,
                '-c:v', 'libvpx-vp9',
                '-b:v', '0',
                '-crf', '32',
                '-deadline', 'good',
                '-row-mt', '1',
                '-c:a', 'libopus',
                '-b:a', '128k',
                outputName
            );

            const outData = ffmpeg.FS('readFile', outputName);
            if (!outData || outData.length === 0) {
                throw new Error('FFmpeg output was empty.');
            }

            if (typeof onProgress === 'function') {
                onProgress(100);
            }

            return new File([outData.buffer], `${fileNameWithoutExt(file.name)}.webm`, {
                type: 'video/webm',
                lastModified: Date.now(),
            });
        } finally {
            try {
                ffmpeg.FS('unlink', inputName);
            } catch {
                // ignore cleanup failures
            }
            try {
                ffmpeg.FS('unlink', outputName);
            } catch {
                // ignore cleanup failures
            }
        }
    }

    async function convertVideoToWebmBestEffort(file, onProgress) {
        try {
            const convertedWithFfmpeg = await convertVideoToWebmWithFfmpeg(file, onProgress);
            return {
                file: convertedWithFfmpeg,
                method: 'ffmpeg',
                ffmpegError: '',
            };
        } catch (ffmpegError) {
            const fallbackFile = await convertVideoToWebmInBrowser(file, onProgress);
            return {
                file: fallbackFile,
                method: 'fallback',
                ffmpegError: ffmpegError instanceof Error ? ffmpegError.message : 'FFmpeg conversion failed',
            };
        }
    }

    async function normalizeAndConvertFiles(files, options) {
        const {
            allowVideo,
            maxFiles,
            maxFileBytes,
            feedbackEl,
        } = options;

        let working = Array.from(files || []);
        const messages = [];

        if (!allowVideo) {
            const removedVideos = working.filter((file) => isVideoFile(file));
            if (removedVideos.length > 0) {
                working = working.filter((file) => !isVideoFile(file));
                messages.push(`Video files are disabled for this board. Skipped: ${removedVideos.map((file) => file.name).join(', ')}`);
            }
        } else {
            const nonWebmVideos = working.filter((file) => isVideoFile(file) && !isWebmVideo(file));
            if (nonWebmVideos.length > 0) {
                const wantsConversion = typeof globalThis.confirm === 'function'
                    ? globalThis.confirm(
                        `This board accepts WEBM for videos. Convert ${nonWebmVideos.length} non-WEBM video${nonWebmVideos.length > 1 ? 's' : ''} in your browser now?\n\nFFmpeg WASM is attempted first (better control), then fallback converter if needed.\n\n${BROWSER_CONVERSION_WARNING}`
                    )
                    : false;

                if (!wantsConversion) {
                    working = working.filter((file) => !nonWebmVideos.includes(file));
                    messages.push(`Skipped non-WEBM videos: ${nonWebmVideos.map((file) => file.name).join(', ')}`);
                } else {
                    const convertedMap = new Map();
                    const failed = [];
                    const ffmpegFallbackReasons = [];
                    let ffmpegConvertedCount = 0;
                    let fallbackConvertedCount = 0;

                    for (let i = 0; i < nonWebmVideos.length; i += 1) {
                        const sourceVideo = nonWebmVideos[i];
                        setFeedback(feedbackEl, `Converting video ${i + 1}/${nonWebmVideos.length}: ${sourceVideo.name}…`, false);

                        try {
                            const conversion = await convertVideoToWebmBestEffort(sourceVideo, (progress) => {
                                setFeedback(feedbackEl, `Converting video ${i + 1}/${nonWebmVideos.length}: ${sourceVideo.name} (${progress}%)`, false);
                            });
                            convertedMap.set(sourceVideo, conversion.file);

                            if (conversion.method === 'ffmpeg') {
                                ffmpegConvertedCount += 1;
                            } else {
                                fallbackConvertedCount += 1;
                                if (conversion.ffmpegError) {
                                    ffmpegFallbackReasons.push(`${sourceVideo.name} (${conversion.ffmpegError})`);
                                }
                            }
                        } catch (error) {
                            const reason = error instanceof Error ? error.message : 'Unknown conversion error';
                            failed.push(`${sourceVideo.name} (${reason})`);
                        }
                    }

                    working = working
                        .map((file) => convertedMap.get(file) || file)
                        .filter((file) => isWebmVideo(file) || !isVideoFile(file));

                    if (ffmpegConvertedCount > 0) {
                        messages.push(`Converted ${ffmpegConvertedCount} video${ffmpegConvertedCount > 1 ? 's' : ''} with FFmpeg WASM.`);
                    }

                    if (fallbackConvertedCount > 0) {
                        messages.push(`Used fallback converter for ${fallbackConvertedCount} video${fallbackConvertedCount > 1 ? 's' : ''}. ${BROWSER_CONVERSION_WARNING}`);
                    }

                    if (ffmpegFallbackReasons.length > 0) {
                        messages.push(`FFmpeg fallback details: ${ffmpegFallbackReasons.join('; ')}`);
                    }

                    if (failed.length > 0) {
                        messages.push(`Some videos could not be converted and were skipped: ${failed.join('; ')}`);
                    }
                }
            }
        }

        if (maxFileBytes > 0) {
            const oversized = working.filter((file) => Number(file.size || 0) > maxFileBytes);
            if (oversized.length > 0) {
                working = working.filter((file) => Number(file.size || 0) <= maxFileBytes);
                messages.push(`Skipped oversized file${oversized.length > 1 ? 's' : ''}: ${oversized.map((file) => `${file.name} (${formatBytes(file.size)}) exceeds ${formatBytes(maxFileBytes)}`).join('; ')}`);
            }
        }

        if (maxFiles > 0 && working.length > maxFiles) {
            const removed = working.slice(maxFiles);
            working = working.slice(0, maxFiles);
            messages.push(`Skipped extra file${removed.length > 1 ? 's' : ''} (max ${maxFiles}): ${removed.map((file) => file.name).join(', ')}`);
        }

        return {
            files: working,
            messages,
        };
    }

    function makePreviewItem(file, previewUrls) {
        const details = document.createElement('details');
        details.className = 'upload-preview-item';

        const summary = document.createElement('summary');
        summary.className = 'upload-preview-summary';

        const expanded = document.createElement('div');
        expanded.className = 'upload-preview-expanded';

        const isImage = (file.type || '').startsWith('image/');
        const isVideo = (file.type || '').startsWith('video/');
        const fileUrl = URL.createObjectURL(file);
        previewUrls.push(fileUrl);

        if (isImage) {
            const thumb = document.createElement('img');
            thumb.src = fileUrl;
            thumb.alt = file.name;
            thumb.loading = 'lazy';
            summary.appendChild(thumb);

            const full = document.createElement('img');
            full.src = fileUrl;
            full.alt = file.name;
            full.loading = 'lazy';
            expanded.appendChild(full);
        } else if (isVideo) {
            const thumb = document.createElement('video');
            thumb.src = fileUrl;
            thumb.muted = true;
            thumb.playsInline = true;
            thumb.preload = 'metadata';
            summary.appendChild(thumb);

            const full = document.createElement('video');
            full.src = fileUrl;
            full.controls = true;
            full.preload = 'metadata';
            expanded.appendChild(full);
        } else {
            const genericThumb = document.createElement('div');
            genericThumb.className = 'upload-preview-generic';
            genericThumb.textContent = 'FILE';
            summary.appendChild(genericThumb);

            const genericFull = document.createElement('p');
            genericFull.className = 'upload-preview-file';
            genericFull.textContent = file.name;
            expanded.appendChild(genericFull);
        }

        const caption = document.createElement('p');
        caption.className = 'upload-preview-caption';
        caption.textContent = `${file.name} • ${formatBytes(file.size)}`;
        expanded.appendChild(caption);

        details.appendChild(summary);
        details.appendChild(expanded);
        return details;
    }

    function initUploadForm(form) {
        const fileInput = form.querySelector('input[type="file"][data-media-input], input[type="file"][multiple]');
        if (!fileInput) {
            return;
        }

        const maxFiles = Math.max(0, Number.parseInt(form.dataset.maxFiles || '0', 10) || 0);
        const maxFileBytes = Math.max(0, Number.parseInt(form.dataset.maxFileBytes || '0', 10) || 0);
        const dropzone = form.querySelector('[data-upload-dropzone]');
        const previewWrap = form.querySelector('[data-upload-preview]');
        const feedbackEl = form.querySelector('[data-upload-feedback]');
        const progressWrap = form.querySelector('[data-upload-progress]');
        const progressBar = progressWrap ? progressWrap.querySelector('.upload-progress-bar') : null;
        const progressText = progressWrap ? progressWrap.querySelector('[data-upload-progress-text], .upload-progress-text') : null;
        const submitButton = form.querySelector('button[type="submit"]');
        const allowVideo = String(fileInput.getAttribute('accept') || '').toLowerCase().includes('video/');

        let previewUrls = [];
        let isNormalizing = false;
        let isSubmitting = false;

        const clearPreviewUrls = () => {
            previewUrls.forEach((url) => {
                URL.revokeObjectURL(url);
            });
            previewUrls = [];
        };

        const renderPreview = () => {
            if (!previewWrap) {
                return;
            }

            clearPreviewUrls();
            previewWrap.innerHTML = '';
            const files = Array.from(fileInput.files || []);
            files.forEach((file) => {
                previewWrap.appendChild(makePreviewItem(file, previewUrls));
            });
        };

        const enforceSelectionRules = async () => {
            if (isNormalizing) {
                return false;
            }

            const files = Array.from(fileInput.files || []);
            if (files.length === 0) {
                setFeedback(feedbackEl, '', false);
                renderPreview();
                return true;
            }

            isNormalizing = true;
            if (submitButton && !isSubmitting) {
                submitButton.disabled = true;
            }

            try {
                const result = await normalizeAndConvertFiles(files, {
                    allowVideo,
                    maxFiles,
                    maxFileBytes,
                    feedbackEl,
                });

                const changed = result.files.length !== files.length || result.files.some((file, index) => file !== files[index]);
                if (changed) {
                    if (!setInputFiles(fileInput, result.files)) {
                        setFeedback(feedbackEl, `${result.messages.join(' ')} Please remove invalid files manually (browser limitation).`, true);
                        return false;
                    }
                }

                const messageText = result.messages.join(' ').trim();
                setFeedback(feedbackEl, messageText, messageText !== '');

                renderPreview();
                return true;
            } finally {
                isNormalizing = false;
                if (submitButton && !isSubmitting) {
                    submitButton.disabled = false;
                }
            }
        };

        const mergeDroppedFiles = async (droppedFiles) => {
            const existing = Array.from(fileInput.files || []);
            const merged = existing.concat(droppedFiles);
            if (!setInputFiles(fileInput, merged)) {
                setFeedback(feedbackEl, 'Your browser does not support drag/drop file merge. Use the file picker.', true);
                return;
            }
            await enforceSelectionRules();
        };

        fileInput.addEventListener('change', () => {
            void enforceSelectionRules();
        });

        if (dropzone) {
            dropzone.addEventListener('click', () => {
                fileInput.click();
            });

            dropzone.addEventListener('keydown', (event) => {
                if (event.key === 'Enter' || event.key === ' ') {
                    event.preventDefault();
                    fileInput.click();
                }
            });

            ['dragenter', 'dragover'].forEach((eventName) => {
                dropzone.addEventListener(eventName, (event) => {
                    event.preventDefault();
                    dropzone.classList.add('is-dragover');
                });
            });

            ['dragleave', 'dragend', 'drop'].forEach((eventName) => {
                dropzone.addEventListener(eventName, () => {
                    dropzone.classList.remove('is-dragover');
                });
            });

            dropzone.addEventListener('drop', (event) => {
                event.preventDefault();
                const dropped = Array.from(event.dataTransfer?.files || []);
                if (dropped.length === 0) {
                    return;
                }
                void mergeDroppedFiles(dropped);
            });
        }

        form.addEventListener('submit', async (event) => {
            event.preventDefault();

            if (isSubmitting || isNormalizing) {
                return;
            }

            const isValid = await enforceSelectionRules();
            if (!isValid) {
                event.preventDefault();
                return;
            }

            const selectedFiles = Array.from(fileInput.files || []);

            if (selectedFiles.length === 0) {
                form.submit();
                return;
            }

            if (!globalThis.XMLHttpRequest || !globalThis.FormData || !submitButton || !progressWrap || !progressBar || !progressText) {
                form.submit();
                return;
            }

            isSubmitting = true;

            submitButton.disabled = true;
            progressWrap.hidden = false;
            progressBar.value = 0;
            progressText.textContent = 'Uploading: 0%';

            const xhr = new XMLHttpRequest();
            const action = form.getAttribute('action') || globalThis.location.href;
            xhr.open('POST', action, true);

            xhr.upload.addEventListener('progress', (progressEvent) => {
                if (!progressEvent.lengthComputable || progressEvent.total <= 0) {
                    return;
                }

                const percent = Math.max(0, Math.min(100, Math.round((progressEvent.loaded / progressEvent.total) * 100)));
                progressBar.value = percent;
                progressText.textContent = `Uploading: ${percent}%`;
            });

            const restoreSubmitState = () => {
                isSubmitting = false;
                submitButton.disabled = false;
                progressWrap.hidden = true;
            };

            xhr.addEventListener('load', () => {
                if (xhr.status >= 200 && xhr.status < 400) {
                    progressBar.value = 100;
                    progressText.textContent = 'Uploading: 100%';
                    const destination = xhr.responseURL && xhr.responseURL !== '' ? xhr.responseURL : globalThis.location.href;
                    globalThis.location.assign(destination);
                    return;
                }

                restoreSubmitState();
                setFeedback(feedbackEl, 'Upload failed. Please try again.', true);
            });

            xhr.addEventListener('error', () => {
                restoreSubmitState();
                setFeedback(feedbackEl, 'Upload failed. Please try again.', true);
            });

            const formData = new FormData(form);
            xhr.send(formData);
        });

        window.addEventListener('beforeunload', clearPreviewUrls);
        renderPreview();
    }

    function initUploadEnhancements() {
        document.querySelectorAll('form[data-upload-form="1"]').forEach((form) => {
            initUploadForm(form);
        });
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initUploadEnhancements);
    } else {
        initUploadEnhancements();
    }
})();
