<?php

declare(strict_types=1);

function appSecretKeyMaterial(): string
{
    $secret = (string)appConfig('APP_SECRET', '');
    if ($secret === '') {
        return hash('sha256', 'website22-fallback-secret', true);
    }
    return hash('sha256', $secret, true);
}

function encryptSettingValue(string $plainText): string
{
    if ($plainText === '') {
        return '';
    }

    if (!function_exists('openssl_encrypt')) {
        return $plainText;
    }

    $key = appSecretKeyMaterial();
    $iv = random_bytes(12);
    $tag = '';
    $cipher = openssl_encrypt($plainText, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
    if ($cipher === false) {
        return $plainText;
    }

    return 'enc:v1:' . base64_encode($iv . $tag . $cipher);
}

function decryptSettingValue(string $stored): string
{
    if ($stored === '') {
        return '';
    }

    if (!str_starts_with($stored, 'enc:v1:')) {
        return $stored;
    }

    if (!function_exists('openssl_decrypt')) {
        return '';
    }

    $raw = base64_decode(substr($stored, 7), true);
    if (!is_string($raw) || strlen($raw) < 29) {
        return '';
    }

    $iv = substr($raw, 0, 12);
    $tag = substr($raw, 12, 16);
    $cipher = substr($raw, 28);

    $plain = openssl_decrypt($cipher, 'aes-256-gcm', appSecretKeyMaterial(), OPENSSL_RAW_DATA, $iv, $tag);
    return $plain === false ? '' : $plain;
}

function smtpSettings(): array
{
    $host = trim(settingGet('smtp_host', ''));
    $port = (int)settingGet('smtp_port', '587');
    $username = trim(settingGet('smtp_username', ''));
    $password = decryptSettingValue(settingGet('smtp_password_enc', ''));
    $security = strtolower(trim(settingGet('smtp_security', 'tls')));
    $fromEmail = trim(settingGet('smtp_from_email', $username));

    if ($security !== 'ssl' && $security !== 'tls' && $security !== 'none') {
        $security = 'tls';
    }

    if ($port <= 0) {
        $port = 587;
    }

    if ($fromEmail === '') {
        $fromEmail = $username;
    }

    return [
        'host' => $host,
        'port' => $port,
        'username' => $username,
        'password' => $password,
        'security' => $security,
        'from_email' => $fromEmail,
    ];
}

function smtpIsConfigured(): bool
{
    $cfg = smtpSettings();
    return $cfg['host'] !== '' && $cfg['username'] !== '' && $cfg['password'] !== '' && $cfg['port'] > 0;
}

function smtpReadReply($socket): array
{
    $response = '';
    $code = 0;

    while (!feof($socket)) {
        $line = fgets($socket, 515);
        if ($line === false) {
            break;
        }
        $response .= $line;
        if (preg_match('/^(\d{3})([\s-])/', $line, $matches)) {
            $code = (int)$matches[1];
            if ($matches[2] === ' ') {
                break;
            }
        } else {
            break;
        }
    }

    return [$code, trim($response)];
}

function smtpExpect($socket, array $acceptedCodes): array
{
    [$code, $message] = smtpReadReply($socket);
    if (!in_array($code, $acceptedCodes, true)) {
        return [false, $message === '' ? ('SMTP error code ' . $code) : $message];
    }
    return [true, $message];
}

function smtpWrite($socket, string $command): array
{
    $written = fwrite($socket, $command . "\r\n");
    if ($written === false) {
        return [false, 'Could not write SMTP command.'];
    }
    return [true, ''];
}

function smtpSendTextMail(string $toEmail, string $subject, string $body): array
{
    if (!smtpIsConfigured()) {
        return [false, 'SMTP is not configured.'];
    }

    if (!filter_var($toEmail, FILTER_VALIDATE_EMAIL)) {
        return [false, 'Invalid recipient email.'];
    }

    $cfg = smtpSettings();
    $transportHost = $cfg['security'] === 'ssl' ? 'ssl://' . $cfg['host'] : $cfg['host'];
    $socket = @stream_socket_client($transportHost . ':' . $cfg['port'], $errno, $errstr, 20, STREAM_CLIENT_CONNECT);
    if (!$socket) {
        return [false, 'SMTP connect failed: ' . $errstr . ' (' . $errno . ')'];
    }

    stream_set_timeout($socket, 20);

    [$ok, $msg] = smtpExpect($socket, [220]);
    if (!$ok) {
        fclose($socket);
        return [false, $msg];
    }

    $hostname = preg_replace('/[^a-zA-Z0-9.-]/', '', (string)($_SERVER['HTTP_HOST'] ?? 'localhost'));
    [$ok, $msg] = smtpWrite($socket, 'EHLO ' . ($hostname !== '' ? $hostname : 'localhost'));
    if (!$ok) {
        fclose($socket);
        return [false, $msg];
    }
    [$ok, $msg] = smtpExpect($socket, [250]);
    if (!$ok) {
        fclose($socket);
        return [false, $msg];
    }

    if ($cfg['security'] === 'tls') {
        [$ok, $msg] = smtpWrite($socket, 'STARTTLS');
        if (!$ok) {
            fclose($socket);
            return [false, $msg];
        }
        [$ok, $msg] = smtpExpect($socket, [220]);
        if (!$ok) {
            fclose($socket);
            return [false, $msg];
        }

        $tlsMethod = 0;
        if (defined('STREAM_CRYPTO_METHOD_TLSv1_3_CLIENT')) {
            $tlsMethod |= STREAM_CRYPTO_METHOD_TLSv1_3_CLIENT;
        }
        if (defined('STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT')) {
            $tlsMethod |= STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT;
        }
        if ($tlsMethod === 0) {
            $tlsMethod = STREAM_CRYPTO_METHOD_TLS_CLIENT;
        }

        $crypto = stream_socket_enable_crypto($socket, true, $tlsMethod);
        if ($crypto !== true) {
            fclose($socket);
            return [false, 'SMTP TLS negotiation failed.'];
        }

        smtpWrite($socket, 'EHLO ' . ($hostname !== '' ? $hostname : 'localhost'));
        [$ok, $msg] = smtpExpect($socket, [250]);
        if (!$ok) {
            fclose($socket);
            return [false, $msg];
        }
    }

    smtpWrite($socket, 'AUTH LOGIN');
    [$ok, $msg] = smtpExpect($socket, [334]);
    if (!$ok) {
        fclose($socket);
        return [false, $msg];
    }

    smtpWrite($socket, base64_encode($cfg['username']));
    [$ok, $msg] = smtpExpect($socket, [334]);
    if (!$ok) {
        fclose($socket);
        return [false, $msg];
    }

    smtpWrite($socket, base64_encode($cfg['password']));
    [$ok, $msg] = smtpExpect($socket, [235]);
    if (!$ok) {
        fclose($socket);
        return [false, $msg];
    }

    smtpWrite($socket, 'MAIL FROM:<' . $cfg['from_email'] . '>');
    [$ok, $msg] = smtpExpect($socket, [250]);
    if (!$ok) {
        fclose($socket);
        return [false, $msg];
    }

    smtpWrite($socket, 'RCPT TO:<' . $toEmail . '>');
    [$ok, $msg] = smtpExpect($socket, [250, 251]);
    if (!$ok) {
        fclose($socket);
        return [false, $msg];
    }

    smtpWrite($socket, 'DATA');
    [$ok, $msg] = smtpExpect($socket, [354]);
    if (!$ok) {
        fclose($socket);
        return [false, $msg];
    }

    $safeSubject = trim(preg_replace('/[\r\n]+/', ' ', $subject) ?? 'Message');
    $headers = [
        'Date: ' . gmdate('D, d M Y H:i:s') . ' +0000',
        'From: <' . $cfg['from_email'] . '>',
        'To: <' . $toEmail . '>',
        'Subject: ' . $safeSubject,
        'MIME-Version: 1.0',
        'Content-Type: text/plain; charset=UTF-8',
        'Content-Transfer-Encoding: 8bit',
    ];

    $payload = implode("\r\n", $headers) . "\r\n\r\n" . str_replace(["\r\n", "\r"], "\n", $body);
    $payload = str_replace("\n.", "\n..", $payload);

    fwrite($socket, $payload . "\r\n.\r\n");
    [$ok, $msg] = smtpExpect($socket, [250]);

    smtpWrite($socket, 'QUIT');
    fclose($socket);

    if (!$ok) {
        return [false, $msg];
    }

    return [true, 'Sent'];
}

function emailVerificationEnabled(): bool
{
    return settingGet('email_verification_enabled', '0') === '1' && smtpIsConfigured();
}

function passwordResetEnabled(): bool
{
    return settingGet('password_reset_enabled', '0') === '1' && smtpIsConfigured();
}

function issueEmailVerificationToken(int $userId): string
{
    $token = bin2hex(random_bytes(24));
    $hash = hash('sha256', $token);
    $now = time();

    $stmt = appDb()->prepare('INSERT INTO email_verification_tokens (user_id, token_hash, expires_at, used_at, created_at) VALUES (?, ?, ?, NULL, ?)');
    $stmt->execute([$userId, $hash, $now + 86400, $now]);

    return $token;
}

function consumeEmailVerificationToken(string $token): bool
{
    $hash = hash('sha256', $token);
    $now = time();

    $stmt = appDb()->prepare('SELECT id, user_id FROM email_verification_tokens WHERE token_hash = ? AND used_at IS NULL AND expires_at >= ? ORDER BY id DESC LIMIT 1');
    $stmt->execute([$hash, $now]);
    $row = $stmt->fetch();
    if (!$row) {
        return false;
    }

    $tokenId = (int)$row['id'];
    $userId = (int)$row['user_id'];

    appDb()->prepare('UPDATE email_verification_tokens SET used_at = ? WHERE id = ?')->execute([$now, $tokenId]);
    appDb()->prepare('UPDATE users SET email_verified_at = ? WHERE id = ?')->execute([dbNow(), $userId]);

    return true;
}

function issuePasswordResetToken(int $userId): string
{
    $token = bin2hex(random_bytes(24));
    $hash = hash('sha256', $token);
    $now = time();

    $stmt = appDb()->prepare('INSERT INTO password_reset_tokens (user_id, token_hash, expires_at, used_at, created_at) VALUES (?, ?, ?, NULL, ?)');
    $stmt->execute([$userId, $hash, $now + 3600, $now]);

    return $token;
}

function passwordResetTokenUser(string $token): ?int
{
    $hash = hash('sha256', $token);
    $now = time();
    $stmt = appDb()->prepare('SELECT user_id FROM password_reset_tokens WHERE token_hash = ? AND used_at IS NULL AND expires_at >= ? ORDER BY id DESC LIMIT 1');
    $stmt->execute([$hash, $now]);
    $userId = $stmt->fetchColumn();
    return $userId === false ? null : (int)$userId;
}

function consumePasswordResetToken(string $token): bool
{
    $hash = hash('sha256', $token);
    $now = time();

    $stmt = appDb()->prepare('SELECT id FROM password_reset_tokens WHERE token_hash = ? AND used_at IS NULL AND expires_at >= ? ORDER BY id DESC LIMIT 1');
    $stmt->execute([$hash, $now]);
    $id = $stmt->fetchColumn();
    if ($id === false) {
        return false;
    }

    appDb()->prepare('UPDATE password_reset_tokens SET used_at = ? WHERE id = ?')->execute([$now, (int)$id]);
    return true;
}
