<?php

declare(strict_types=1);

if (!defined('APP_ROOT')) {
    define('APP_ROOT', dirname(__DIR__));
}
if (!defined('DATA_PATH')) {
    define('DATA_PATH', APP_ROOT . '/data');
}
if (!defined('INCLUDES_PATH')) {
    define('INCLUDES_PATH', APP_ROOT . '/includes');
}
if (!defined('PUBLIC_PATH')) {
    define('PUBLIC_PATH', APP_ROOT . '/public');
}

require_once INCLUDES_PATH . '/database.php';

$__appConfig = [];
$configPath = DATA_PATH . '/config.php';
if (is_file($configPath)) {
    $loaded = include $configPath;
    if (is_array($loaded)) {
        $__appConfig = $loaded;
    }
}

function appConfig(string $key, $default = null)
{
    global $__appConfig;
    return array_key_exists($key, $__appConfig) ? $__appConfig[$key] : $default;
}

include_once INCLUDES_PATH . '/mailer.php';

function forceHttpsIfRequired(): void
{
    if ((string)appConfig('FORCE_HTTPS', '0') !== '1') {
        return;
    }

    if (php_sapi_name() === 'cli') {
        return;
    }

    $isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
        || ((int)($_SERVER['SERVER_PORT'] ?? 0) === 443)
        || (strtolower((string)($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '')) === 'https');

    if ($isHttps || headers_sent()) {
        return;
    }

    $host = (string)($_SERVER['HTTP_HOST'] ?? '');
    $uri = (string)($_SERVER['REQUEST_URI'] ?? '/');
    if ($host !== '') {
        header('Location: https://' . $host . $uri, true, 301);
        exit;
    }
}

forceHttpsIfRequired();

ini_set('display_errors', '0');
ini_set('log_errors', '1');
error_reporting(E_ALL);
ini_set('expose_php', '0');
date_default_timezone_set('UTC');

function isHttpsRequest(): bool
{
    return (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
        || ((int)($_SERVER['SERVER_PORT'] ?? 0) === 443)
        || (strtolower((string)($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '')) === 'https');
}

if (session_status() !== PHP_SESSION_ACTIVE) {
    ini_set('session.use_only_cookies', '1');
    ini_set('session.use_strict_mode', '1');
    ini_set('session.cookie_httponly', '1');
    ini_set('session.cookie_samesite', 'Lax');
    ini_set('session.cookie_secure', isHttpsRequest() ? '1' : '0');

    session_name('W22SESSID');

    $cookieDomain = (string)appConfig('COOKIE_DOMAIN', '');
    session_set_cookie_params([
        'lifetime' => 0,
        'path' => '/',
        'domain' => $cookieDomain,
        'secure' => isHttpsRequest(),
        'httponly' => true,
        'samesite' => 'Lax',
    ]);

    session_start();
}

if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

if (php_sapi_name() !== 'cli' && !headers_sent()) {
    header_remove('X-Powered-By');
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: SAMEORIGIN');
    header('Referrer-Policy: strict-origin-when-cross-origin');
    header('Permissions-Policy: geolocation=(), microphone=(), camera=()');
    header('Cross-Origin-Opener-Policy: same-origin');
    header('Cross-Origin-Embedder-Policy: require-corp');
    header('Cross-Origin-Resource-Policy: same-origin');
    header("Content-Security-Policy: default-src 'self'; script-src 'self' blob: 'wasm-unsafe-eval'; script-src-elem 'self' blob:; connect-src 'self' blob:; style-src 'self'; img-src 'self' data: blob:; media-src 'self' blob:; worker-src 'self' blob:; object-src 'none'; base-uri 'self'; form-action 'self'; frame-ancestors 'none'");

    if (isHttpsRequest()) {
        header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
    }
}

if (appIsInstalled()) {
    appInstallSchema(appDb());
}

function routePath(): string
{
    $route = trim((string)($_GET['route'] ?? ''), '/');
    if ($route === '') {
        $path = trim((string)parse_url((string)($_SERVER['REQUEST_URI'] ?? '/'), PHP_URL_PATH), '/');
        if ($path !== '' && basename($path) !== 'index.php') {
            $route = $path;
        }
    }
    return trim($route, '/');
}

function e(string $value): string
{
    return htmlspecialchars($value, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function redirectTo(string $url): void
{
    header('Location: ' . $url);
    exit;
}

function baseUrl(): string
{
    $scheme = isHttpsRequest() ? 'https' : 'http';
    $host = (string)($_SERVER['HTTP_HOST'] ?? 'localhost');
    return $scheme . '://' . $host;
}

function absoluteUrl(string $path): string
{
    $p = $path;
    if (!str_starts_with($p, '/')) {
        $p = '/' . $p;
    }
    return baseUrl() . $p;
}

function csrfToken(): string
{
    return (string)($_SESSION['csrf_token'] ?? '');
}

function ensureCsrfForPost(): void
{
    if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET')) !== 'POST') {
        return;
    }

    $token = (string)($_POST['csrf_token'] ?? '');
    $session = csrfToken();
    if ($token === '' || $session === '' || !hash_equals($session, $token)) {
        http_response_code(403);
        echo 'Invalid CSRF token.';
        exit;
    }
}

function clientIp(): string
{
    $keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
    foreach ($keys as $key) {
        if (empty($_SERVER[$key])) {
            continue;
        }

        $raw = (string)$_SERVER[$key];
        $first = trim(explode(',', $raw)[0]);
        if (filter_var($first, FILTER_VALIDATE_IP)) {
            return $first;
        }
    }

    return '0.0.0.0';
}

function checkHoneypotField(string $fieldName = 'website'): bool
{
    return trim((string)($_POST[$fieldName] ?? '')) === '';
}

function cleanupRateLimits(): void
{
    appDb()->prepare('DELETE FROM rate_limits WHERE reset_at <= ?')->execute([time()]);
}

function consumeRateLimit(string $bucket, int $maxRequests, int $windowSeconds): bool
{
    cleanupRateLimits();

    $now = time();
    $stmt = appDb()->prepare('SELECT request_count, reset_at FROM rate_limits WHERE rate_key = ? LIMIT 1');
    $stmt->execute([$bucket]);
    $row = $stmt->fetch();

    if (!$row) {
        appDb()->prepare('INSERT INTO rate_limits (rate_key, request_count, reset_at) VALUES (?, ?, ?)')
            ->execute([$bucket, 1, $now + $windowSeconds]);
        return true;
    }

    $count = (int)$row['request_count'];
    $resetAt = (int)$row['reset_at'];

    if ($resetAt <= $now) {
        appDb()->prepare('UPDATE rate_limits SET request_count = 1, reset_at = ? WHERE rate_key = ?')
            ->execute([$now + $windowSeconds, $bucket]);
        return true;
    }

    if ($count >= $maxRequests) {
        return false;
    }

    appDb()->prepare('UPDATE rate_limits SET request_count = request_count + 1 WHERE rate_key = ?')
        ->execute([$bucket]);

    return true;
}

function actorRateKey(string $action, ?int $userId = null): string
{
    $uid = $userId ?? (int)(currentUser()['id'] ?? 0);
    return $action . '|ip:' . clientIp() . '|u:' . $uid;
}

function publicHandle(?string $displayName): string
{
    $name = trim((string)$displayName);
    if ($name === '') {
        return 'Anonymous';
    }
    return mb_substr($name, 0, 30);
}

function publicHandleFromRow(array $row): string
{
    return publicHandle((string)($row['display_name'] ?? ''));
}

function userProfileUrl(int $userId): string
{
    return '/user/' . max(1, $userId);
}

function boardRuleForSubcategory(int $subcategoryId): array
{
    $defaultMb = (int)settingGet('board_default_max_file_mb', '0');
    if ($defaultMb <= 0) {
        $legacyBytes = max(1024 * 1024, (int)settingGet('board_default_max_file_bytes', '20971520'));
        $defaultMb = max(1, (int)ceil($legacyBytes / (1024 * 1024)));
    }
    $defaultBytes = max(1024 * 1024, $defaultMb * 1024 * 1024);

    $defaults = [
        'max_files' => max(1, (int)settingGet('board_default_max_files', '10')),
        'max_file_bytes' => $defaultBytes,
        'cooldown_seconds' => max(0, (int)settingGet('board_default_cooldown_seconds', '10')),
        'max_posts_per_hour' => max(1, (int)settingGet('board_default_max_posts_per_hour', '200')),
        'max_comments_per_hour' => max(1, (int)settingGet('board_default_max_comments_per_hour', '500')),
        'allow_video' => 1,
        'captcha_mode' => settingGet('board_default_captcha_mode', 'low_trust'),
    ];

    $stmt = appDb()->prepare('SELECT max_files, max_file_bytes, cooldown_seconds, max_posts_per_hour, max_comments_per_hour, allow_video, captcha_mode FROM board_rules WHERE subcategory_id = ? LIMIT 1');
    $stmt->execute([$subcategoryId]);
    $row = $stmt->fetch();
    if (!$row) {
        return $defaults;
    }

    return [
        'max_files' => max(1, (int)$row['max_files']),
        'max_file_bytes' => max(1024 * 1024, (int)$row['max_file_bytes']),
        'cooldown_seconds' => max(0, (int)$row['cooldown_seconds']),
        'max_posts_per_hour' => max(1, (int)$row['max_posts_per_hour']),
        'max_comments_per_hour' => max(1, (int)$row['max_comments_per_hour']),
        'allow_video' => (int)$row['allow_video'] === 1 ? 1 : 0,
        'captcha_mode' => in_array((string)($row['captcha_mode'] ?? 'low_trust'), ['off', 'low_trust', 'all'], true)
            ? (string)$row['captcha_mode']
            : 'low_trust',
    ];
}

function userIsLowTrust(int $userId): bool
{
    $stmt = appDb()->prepare('SELECT id, role, created_at, email_verified_at FROM users WHERE id = ? LIMIT 1');
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    if (!$user) {
        return true;
    }

    if (($user['role'] ?? '') === 'admin') {
        return false;
    }

    $ageHoursThreshold = max(1, (int)settingGet('low_trust_age_hours', '72'));
    $minActivity = max(0, (int)settingGet('low_trust_min_activity', '5'));

    $createdAt = strtotime((string)$user['created_at']);
    if ($createdAt === false || ((time() - $createdAt) < ($ageHoursThreshold * 3600))) {
        return true;
    }

    if (emailVerificationEnabled() && empty($user['email_verified_at'])) {
        return true;
    }

    $activityStmt = appDb()->prepare('SELECT
        (SELECT COUNT(*) FROM posts WHERE user_id = ? AND status = "active")
        +
        (SELECT COUNT(*) FROM comments WHERE user_id = ? AND status = "active") AS total_activity');
    $activityStmt->execute([$userId, $userId]);
    $activity = (int)$activityStmt->fetchColumn();

    return $activity < $minActivity;
}

function captchaRequiredForBoardRule(array $rule, ?int $userId = null): bool
{
    $mode = (string)($rule['captcha_mode'] ?? 'low_trust');
    if (!in_array($mode, ['off', 'low_trust', 'all'], true)) {
        $mode = 'low_trust';
    }

    if ($mode === 'off') {
        return false;
    }

    if ($mode === 'all') {
        return true;
    }

    if ($userId === null || $userId <= 0) {
        return true;
    }

    return userIsLowTrust($userId);
}

function captchaSessionKey(int $subcategoryId, int $userId): string
{
    return 'captcha_' . $subcategoryId . '_' . $userId;
}

function getCaptchaChallengeForBoard(int $subcategoryId, ?int $userId = null): array
{
    $uid = $userId ?? (int)(currentUser()['id'] ?? 0);
    $key = captchaSessionKey($subcategoryId, $uid);
    $item = $_SESSION[$key] ?? null;
    $valid = is_array($item)
        && isset($item['question'], $item['answer'], $item['expires_at'])
        && (int)$item['expires_at'] > time();

    if (!$valid) {
        $a = random_int(1, 9);
        $b = random_int(1, 9);
        $question = $a . ' + ' . $b . ' = ?';
        $_SESSION[$key] = [
            'question' => $question,
            'answer' => (string)($a + $b),
            'expires_at' => time() + 600,
        ];
    }

    $current = $_SESSION[$key];
    return [
        'key' => $key,
        'question' => (string)$current['question'],
    ];
}

function verifyCaptchaForBoard(int $subcategoryId, ?int $userId, string $answer): bool
{
    $uid = $userId ?? 0;
    $key = captchaSessionKey($subcategoryId, $uid);
    $item = $_SESSION[$key] ?? null;
    if (!is_array($item)) {
        return false;
    }

    $expires = (int)($item['expires_at'] ?? 0);
    $expected = trim((string)($item['answer'] ?? ''));
    $provided = trim($answer);

    if ($expires <= time() || $expected === '' || $provided === '') {
        unset($_SESSION[$key]);
        return false;
    }

    $ok = hash_equals($expected, $provided);
    if ($ok) {
        unset($_SESSION[$key]);
    }
    return $ok;
}

function triggerSitemapPing(string $reason = ''): void
{
    if (settingGet('sitemap_auto_ping_enabled', '0') !== '1') {
        return;
    }

    $raw = trim(settingGet('sitemap_ping_services', ''));
    if ($raw === '') {
        return;
    }

    $sitemapUrl = rawurlencode(absoluteUrl('/sitemap.xml'));
    $lines = preg_split('/\r\n|\r|\n/', $raw) ?: [];

    foreach ($lines as $line) {
        $template = trim($line);
        if ($template === '') {
            continue;
        }

        $target = str_replace('{sitemap_url}', $sitemapUrl, $template);
        if (!preg_match('/^https?:\/\//i', $target)) {
            continue;
        }

        if (function_exists('curl_init')) {
            $ch = curl_init($target);
            if ($ch !== false) {
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 3);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
                curl_setopt($ch, CURLOPT_USERAGENT, 'Website22-SitemapPing/1.0');
                curl_exec($ch);
                curl_close($ch);
            }
        } else {
            $ctx = stream_context_create(['http' => ['method' => 'GET', 'timeout' => 3]]);
            @file_get_contents($target, false, $ctx);
        }
    }

    if ($reason !== '') {
        logAudit((int)(currentUser()['id'] ?? 0), 'sitemap_ping', $reason);
    }
}

function countUploadedFiles(string $inputName = 'media'): int
{
    if (empty($_FILES[$inputName]) || !is_array($_FILES[$inputName]['error'])) {
        return 0;
    }

    $count = 0;
    foreach ($_FILES[$inputName]['error'] as $errorCode) {
        if ((int)$errorCode !== UPLOAD_ERR_NO_FILE) {
            $count++;
        }
    }
    return $count;
}

function paginationPerPage(): int
{
    $value = (int)settingGet('pagination_per_page', '20');
    if ($value < 5) {
        $value = 5;
    }
    if ($value > 100) {
        $value = 100;
    }
    return $value;
}

function flashSet(string $type, string $message): void
{
    $_SESSION['flash_' . $type] = $message;
}

function flashGet(string $type): string
{
    $key = 'flash_' . $type;
    $value = (string)($_SESSION[$key] ?? '');
    unset($_SESSION[$key]);
    return $value;
}

function currentUser(): ?array
{
    static $cache = null;

    if ($cache !== null) {
        return $cache;
    }

    $id = (int)($_SESSION['user_id'] ?? 0);
    if ($id <= 0) {
        $cache = null;
        return null;
    }

    $stmt = appDb()->prepare('SELECT id, username, display_name, email, role, is_banned FROM users WHERE id = ? LIMIT 1');
    $stmt->execute([$id]);
    $user = $stmt->fetch();

    if (!$user) {
        unset($_SESSION['user_id']);
        $cache = null;
        return null;
    }

    if ((int)$user['is_banned'] === 1) {
        unset($_SESSION['user_id']);
        $cache = null;
        return null;
    }

    $cache = $user;
    return $cache;
}

function isLoggedIn(): bool
{
    return currentUser() !== null;
}

function isAdmin(): bool
{
    $user = currentUser();
    return $user !== null && ($user['role'] ?? '') === 'admin';
}

function adminIpAllowed(): bool
{
    $ip = clientIp();
    if ($ip === '127.0.0.1' || $ip === '::1') {
        return true;
    }

    $stmt = appDb()->prepare('SELECT COUNT(*) FROM admin_allowed_ips WHERE ip_address = ?');
    $stmt->execute([$ip]);
    return ((int)$stmt->fetchColumn()) > 0;
}

function requireAdminIpOr404(): void
{
    if (!adminIpAllowed()) {
        http_response_code(404);
        echo '<h1>404 Not Found</h1>';
        exit;
    }
}

function requireAuth(): void
{
    if (!isLoggedIn()) {
        flashSet('error', 'Please login first.');
        redirectTo('/login');
    }
}

function requireAdminAuth(): void
{
    requireAdminIpOr404();
    if (!isAdmin()) {
        redirectTo('/admin?login=1');
    }
}

function securePasswordHash(string $password): string
{
    if (defined('PASSWORD_ARGON2ID')) {
        return password_hash($password, PASSWORD_ARGON2ID);
    }

    return password_hash($password, PASSWORD_BCRYPT);
}

function clearOldLoginAttempts(): void
{
    $before = time() - 600;
    $stmt = appDb()->prepare('DELETE FROM login_attempts WHERE attempted_at < ?');
    $stmt->execute([$before]);
}

function tooManyLoginAttempts(string $username, string $ip): bool
{
    clearOldLoginAttempts();

    $stmt = appDb()->prepare('SELECT COUNT(*) FROM login_attempts WHERE ip_address = ? OR username = ?');
    $stmt->execute([$ip, $username]);
    return ((int)$stmt->fetchColumn()) >= 20;
}

function recordLoginAttempt(string $username, string $ip): void
{
    $stmt = appDb()->prepare('INSERT INTO login_attempts (username, ip_address, attempted_at) VALUES (?, ?, ?)');
    $stmt->execute([$username, $ip, time()]);
}

function doLogin(string $username, string $password, bool $adminOnly = false): bool
{
    $username = trim($username);
    $ip = clientIp();

    if (tooManyLoginAttempts($username, $ip)) {
        $_SESSION['login_block_reason'] = 'Too many login attempts. Please wait up to 10 minutes and try again.';
        return false;
    }

    $userCount = (int)appDb()->query('SELECT COUNT(*) FROM users')->fetchColumn();
    if ($userCount <= 0) {
        $_SESSION['login_block_reason'] = 'No user accounts were found. Database may be reset or setup may be incomplete.';
        return false;
    }

    $stmt = appDb()->prepare('SELECT * FROM users WHERE username = ? LIMIT 1');
    $stmt->execute([$username]);
    $user = $stmt->fetch();

    if (!$user || (int)$user['is_banned'] === 1) {
        recordLoginAttempt($username, $ip);
        return false;
    }

    if (!password_verify($password, (string)$user['password_hash'])) {
        recordLoginAttempt($username, $ip);
        return false;
    }

    if ($adminOnly && ($user['role'] ?? '') !== 'admin') {
        recordLoginAttempt($username, $ip);
        return false;
    }

    if (emailVerificationEnabled() && ($user['role'] ?? '') !== 'admin' && empty($user['email_verified_at'])) {
        $_SESSION['login_block_reason'] = 'Please verify your email before logging in.';
        recordLoginAttempt($username, $ip);
        return false;
    }

    session_regenerate_id(true);
    $_SESSION['user_id'] = (int)$user['id'];

    $stmt = appDb()->prepare('UPDATE users SET last_login_at = ? WHERE id = ?');
    $stmt->execute([dbNow(), (int)$user['id']]);

    return true;
}

function doRegister(string $username, string $email, string $password): array
{
    $username = trim($username);
    $email = trim($email);

    if (!preg_match('/^\w{3,30}$/', $username)) {
        return [false, 'Username must be 3-30 chars and only letters/numbers/_'];
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        return [false, 'Invalid email address.'];
    }

    if (strlen($password) < 8) {
        return [false, 'Password must be at least 8 characters.'];
    }

    $hash = securePasswordHash($password);
    $verificationNeeded = emailVerificationEnabled();
    $emailVerifiedAt = $verificationNeeded ? null : dbNow();

    try {
        $stmt = appDb()->prepare('INSERT INTO users (username, email, password_hash, role, is_banned, email_verified_at, created_at) VALUES (?, ?, ?, "user", 0, ?, ?)');
        $stmt->execute([$username, $email, $hash, $emailVerifiedAt, dbNow()]);
    } catch (Throwable $exception) {
        return [false, 'Username or email already exists.'];
    }

    $userId = (int)appDb()->lastInsertId();

    if ($verificationNeeded) {
        $token = issueEmailVerificationToken($userId);
        $verifyUrl = absoluteUrl('/verify-email?token=' . rawurlencode($token));
        $subject = 'Verify your account email';
        $body = "Hello {$username},\n\nPlease verify your email by opening this link:\n{$verifyUrl}\n\nThis link expires in 24 hours.";
        [$sent] = smtpSendTextMail($email, $subject, $body);
        if ($sent) {
            return [true, 'Account created. Check your email for verification link.'];
        }
        return [true, 'Account created, but verification email could not be sent yet.'];
    }

    return [true, 'Account created. You can now login.'];
}

function doLogout(): void
{
    $_SESSION = [];
    if (ini_get('session.use_cookies')) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000, $params['path'], $params['domain'], (bool)$params['secure'], (bool)$params['httponly']);
    }
    session_destroy();
}

function allowedMediaTypes(): array
{
    return [
        'image/jpeg' => ['kind' => 'image', 'ext' => 'jpg'],
        'image/png' => ['kind' => 'image', 'ext' => 'png'],
        'image/gif' => ['kind' => 'image', 'ext' => 'gif'],
        'image/webp' => ['kind' => 'image', 'ext' => 'webp'],
        'video/webm' => ['kind' => 'video', 'ext' => 'webm'],
    ];
}

function maxUploadBytes(): int
{
    $value = (int)appConfig('MAX_UPLOAD_BYTES', 20971520);
    return $value > 0 ? $value : 20971520;
}

function mediaStoragePath(array $row): string
{
    $kind = ($row['media_kind'] ?? 'image') === 'video' ? 'videos' : 'images';
    return DATA_PATH . '/uploads/' . $kind . '/' . $row['storage_name'];
}

function preGenerateMediaThumb(int $mediaId, string $mediaKind, string $sourcePath, int $thumbSize = 360): void
{
    if ($mediaId <= 0 || $mediaKind !== 'image' || !is_file($sourcePath)) {
        return;
    }

    if (!extension_loaded('gd') || !function_exists('imagecreatefromstring')) {
        return;
    }

    $thumbSize = max(120, min(640, $thumbSize));
    $cacheDir = DATA_PATH . '/cache/thumbs';
    if (!is_dir($cacheDir)) {
        @mkdir($cacheDir, 0755, true);
    }

    $signature = sha1($sourcePath . '|' . (string)@filesize($sourcePath) . '|' . (string)@filemtime($sourcePath) . '|' . $thumbSize);
    $useWebp = function_exists('imagewebp');
    $cacheExt = $useWebp ? 'webp' : 'jpg';
    $cachePath = $cacheDir . '/' . $mediaId . '-' . $signature . '.' . $cacheExt;

    if (is_file($cachePath)) {
        return;
    }

    $raw = @file_get_contents($sourcePath);
    if ($raw === false) {
        return;
    }

    $src = @imagecreatefromstring($raw);
    if (!$src) {
        return;
    }

    $srcW = imagesx($src);
    $srcH = imagesy($src);
    if ($srcW <= 0 || $srcH <= 0) {
        imagedestroy($src);
        return;
    }

    $scale = min($thumbSize / $srcW, $thumbSize / $srcH, 1.0);
    $dstW = max(1, (int)round($srcW * $scale));
    $dstH = max(1, (int)round($srcH * $scale));

    $dst = imagecreatetruecolor($dstW, $dstH);
    imagealphablending($dst, false);
    imagesavealpha($dst, true);
    $transparent = imagecolorallocatealpha($dst, 0, 0, 0, 127);
    imagefilledrectangle($dst, 0, 0, $dstW, $dstH, $transparent);
    imagecopyresampled($dst, $src, 0, 0, 0, 0, $dstW, $dstH, $srcW, $srcH);

    if ($useWebp) {
        @imagewebp($dst, $cachePath, 80);
    } else {
        @imagejpeg($dst, $cachePath, 82);
    }

    imagedestroy($src);
    imagedestroy($dst);
}

function storeMediaFromRequest(int $postId, int $userId, string $inputName = 'media'): array
{
    if (empty($_FILES[$inputName]) || !is_array($_FILES[$inputName]['name'])) {
        return [[], []];
    }

    $options = [];
    if (func_num_args() >= 4) {
        $maybeOptions = func_get_arg(3);
        if (is_array($maybeOptions)) {
            $options = $maybeOptions;
        }
    }

    $files = $_FILES[$inputName];
    $count = count($files['name']);
    $saved = [];
    $errors = [];

    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $allowed = allowedMediaTypes();
    $maxBytes = isset($options['max_file_bytes']) ? max(1, (int)$options['max_file_bytes']) : maxUploadBytes();
    $maxFiles = isset($options['max_files']) ? max(1, (int)$options['max_files']) : 4;
    $allowVideo = !isset($options['allow_video']) || (int)$options['allow_video'] === 1;
    $commentId = isset($options['comment_id']) ? max(0, (int)$options['comment_id']) : 0;

    if (!$allowVideo) {
        unset($allowed['video/webm']);
    }

    if (countUploadedFiles($inputName) > $maxFiles) {
        return [[], ['Too many files for this board. Maximum is ' . $maxFiles . '.']];
    }

    for ($i = 0; $i < $count; $i++) {
        if (($files['error'][$i] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE) {
            continue;
        }

        $errorCode = (int)($files['error'][$i] ?? UPLOAD_ERR_NO_FILE);
        if ($errorCode !== UPLOAD_ERR_OK) {
            $errors[] = 'One upload failed with error code ' . $errorCode;
            continue;
        }

        $tmpName = (string)$files['tmp_name'][$i];
        $originalName = (string)$files['name'][$i];
        $size = (int)$files['size'][$i];

        if ($size <= 0 || $size > $maxBytes) {
            $errors[] = 'File too large or empty: ' . $originalName;
            continue;
        }

        $mime = (string)$finfo->file($tmpName);
        if (!isset($allowed[$mime])) {
            $errors[] = 'Unsupported file type: ' . $originalName;
            continue;
        }

        $kind = $allowed[$mime]['kind'];
        $ext = $allowed[$mime]['ext'];
        $storageName = bin2hex(random_bytes(16)) . '.' . $ext;
        $targetDir = DATA_PATH . '/uploads/' . ($kind === 'video' ? 'videos' : 'images');
        if (!is_dir($targetDir)) {
            mkdir($targetDir, 0755, true);
        }
        $targetPath = $targetDir . '/' . $storageName;

        if (!move_uploaded_file($tmpName, $targetPath)) {
            $errors[] = 'Could not store file: ' . $originalName;
            continue;
        }

        $stmt = appDb()->prepare('INSERT INTO media (post_id, comment_id, user_id, storage_name, original_name, mime_type, media_kind, size_bytes, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)');
        $stmt->execute([
            $postId,
            $commentId > 0 ? $commentId : null,
            $userId,
            $storageName,
            $originalName,
            $mime,
            $kind,
            $size,
            dbNow(),
        ]);

        $mediaId = (int)appDb()->lastInsertId();
        preGenerateMediaThumb($mediaId, $kind, $targetPath, 360);

        $saved[] = [
            'id' => $mediaId,
            'mime_type' => $mime,
            'media_kind' => $kind,
            'storage_name' => $storageName,
        ];
    }

    return [$saved, $errors];
}

function render404(): void
{
    http_response_code(404);
    echo '<!DOCTYPE html><html><head><meta charset="utf-8"><title>404</title></head><body><h1>404 Not Found</h1></body></html>';
    exit;
}

if (!appIsInstalled()) {
    $route = routePath();
    $script = basename((string)($_SERVER['SCRIPT_NAME'] ?? ''));

    if ($script !== 'setup.php' && $route !== 'setup') {
        redirectTo('/setup.php');
    }
}

function loginBlockReason(): string
{
    $reason = (string)($_SESSION['login_block_reason'] ?? '');
    unset($_SESSION['login_block_reason']);
    return $reason;
}

function sendPasswordResetLink(string $email): bool
{
    if (!passwordResetEnabled()) {
        return false;
    }

    $stmt = appDb()->prepare('SELECT id, username, email, is_banned FROM users WHERE email = ? LIMIT 1');
    $stmt->execute([trim($email)]);
    $user = $stmt->fetch();
    if (!$user || (int)$user['is_banned'] === 1) {
        return true;
    }

    $token = issuePasswordResetToken((int)$user['id']);
    $url = absoluteUrl('/reset-password?token=' . rawurlencode($token));
    $subject = 'Password reset request';
    $body = "Hello {$user['username']},\n\nReset your password with this link:\n{$url}\n\nThis link expires in 1 hour.";
    [$ok] = smtpSendTextMail((string)$user['email'], $subject, $body);

    return (bool)$ok;
}

function resetPasswordFromToken(string $token, string $newPassword): bool
{
    if (strlen($newPassword) < 10) {
        return false;
    }

    $userId = passwordResetTokenUser($token);
    if ($userId === null) {
        return false;
    }

    if (!consumePasswordResetToken($token)) {
        return false;
    }

    $stmt = appDb()->prepare('UPDATE users SET password_hash = ? WHERE id = ?');
    $stmt->execute([securePasswordHash($newPassword), $userId]);
    return true;
}
