<?php

declare(strict_types=1);

$bootstrapPath = dirname(__DIR__) . '/includes/bootstrap.php';
if (!is_file($bootstrapPath)) {
    $bootstrapPath = dirname(__DIR__, 2) . '/includes/bootstrap.php';
}
require_once $bootstrapPath;

if (!appIsInstalled()) {
    render404();
}

$token = (string)appConfig('ALLOWLIST_TOOL_TOKEN', '');
if ($token === '') {
    http_response_code(500);
    echo 'ALLOWLIST_TOOL_TOKEN missing in data/config.php';
    exit;
}

if (empty($_SESSION['allowlist_tool_csrf'])) {
    $_SESSION['allowlist_tool_csrf'] = bin2hex(random_bytes(32));
}

$authorized = !empty($_SESSION['allowlist_tool_ok']) && $_SESSION['allowlist_tool_ok'] === true;
$message = '';
$error = '';

if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET')) === 'POST') {
    $action = (string)($_POST['action'] ?? 'unlock');

    if ($action === 'unlock') {
        $provided = (string)($_POST['token'] ?? '');
        if (hash_equals($token, $provided)) {
            $_SESSION['allowlist_tool_ok'] = true;
            $authorized = true;
            $message = 'Tool unlocked.';
        } else {
            $error = 'Invalid token.';
        }
    } else {
        if (!$authorized) {
            $error = 'Not authorized.';
        } else {
            $csrf = (string)($_POST['csrf_token'] ?? '');
            if ($csrf === '' || !hash_equals((string)$_SESSION['allowlist_tool_csrf'], $csrf)) {
                $error = 'Invalid CSRF token.';
            } else {
                $db = appDb();

                if ($action === 'add_ip') {
                    $ip = trim((string)($_POST['ip_address'] ?? ''));
                    $label = trim((string)($_POST['label'] ?? ''));
                    if (!filter_var($ip, FILTER_VALIDATE_IP)) {
                        $error = 'Invalid IP address.';
                    } else {
                        $stmt = $db->prepare('INSERT INTO admin_allowed_ips (ip_address, label, added_at) VALUES (?, ?, ?) ON CONFLICT(ip_address) DO UPDATE SET label = excluded.label, added_at = excluded.added_at');
                        $stmt->execute([$ip, $label, dbNow()]);
                        $message = 'IP allowlisted: ' . $ip;
                    }
                }

                if ($action === 'remove_ip') {
                    $ip = trim((string)($_POST['ip_address'] ?? ''));
                    if (!filter_var($ip, FILTER_VALIDATE_IP)) {
                        $error = 'Invalid IP address.';
                    } else {
                        $db->prepare('DELETE FROM admin_allowed_ips WHERE ip_address = ?')->execute([$ip]);
                        $message = 'IP removed: ' . $ip;
                    }
                }

                if ($action === 'create_admin') {
                    $username = trim((string)($_POST['username'] ?? ''));
                    $email = trim((string)($_POST['email'] ?? ''));
                    $password = (string)($_POST['password'] ?? '');

                    if (!preg_match('/^\w{3,30}$/', $username) || !filter_var($email, FILTER_VALIDATE_EMAIL) || strlen($password) < 10) {
                        $error = 'Invalid admin details. Username 3-30, valid email, password min 10.';
                    } else {
                        try {
                            $stmt = $db->prepare('INSERT INTO users (username, email, password_hash, role, is_banned, created_at) VALUES (?, ?, ?, "admin", 0, ?)');
                            $stmt->execute([$username, $email, securePasswordHash($password), dbNow()]);
                            $message = 'Admin created: ' . $username;
                        } catch (Throwable $exception) {
                            $error = 'Failed to create admin (username/email may already exist).';
                        }
                    }
                }
            }
        }
    }
}

$allowed = appDb()->query('SELECT ip_address, label, added_at FROM admin_allowed_ips ORDER BY id DESC')->fetchAll();
$currentIp = clientIp();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Allowlist Tool</title>
    <style>
        body{font-family:Inter,Arial,sans-serif;background:#0b1020;color:#e8efff;margin:0;padding:20px}
        .card{max-width:860px;margin:0 auto;background:#121b36;border:1px solid #34467a;border-radius:14px;padding:18px}
        .grid{display:grid;gap:10px}
        input{padding:10px;border-radius:8px;border:1px solid #30406f;background:#0b1430;color:#ecf2ff}
        button{padding:9px 14px;border:0;border-radius:8px;background:#69c8ff;color:#0a142a;font-weight:700;cursor:pointer}
        .error{background:#3d1725;border:1px solid #7c2d4b;color:#ffbdd1;padding:10px;border-radius:8px}
        .ok{background:#163a2e;border:1px solid #2c7a5f;color:#b9ffe0;padding:10px;border-radius:8px}
        code{background:#0a1430;padding:2px 5px;border-radius:6px}
        ul{padding-left:20px}
    </style>
</head>
<body>
<div class="card">
    <h1>Admin Allowlist Emergency Tool</h1>
    <p>Upload this file to <code>/public/allowlist_tool.php</code>, use it, then delete it from public root.</p>

    <?php if ($message !== ''): ?><div class="ok"><?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?></div><?php endif; ?>
    <?php if ($error !== ''): ?><div class="error"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></div><?php endif; ?>

    <?php if (!$authorized): ?>
        <form method="post" class="grid">
            <input type="hidden" name="action" value="unlock">
            <label>Access token from <code>data/config.php</code>
                <input type="password" name="token" required>
            </label>
            <button type="submit">Unlock Tool</button>
        </form>
    <?php else: ?>
        <h2>Add/Update allowlisted IP</h2>
        <form method="post" class="grid">
            <input type="hidden" name="action" value="add_ip">
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars((string)$_SESSION['allowlist_tool_csrf'], ENT_QUOTES, 'UTF-8'); ?>">
            <label>IP address
                <input name="ip_address" required value="<?php echo htmlspecialchars($currentIp, ENT_QUOTES, 'UTF-8'); ?>">
            </label>
            <label>Label
                <input name="label" placeholder="Home / Office / VPN">
            </label>
            <button type="submit">Save IP</button>
        </form>

        <h2>Remove IP</h2>
        <form method="post" class="grid">
            <input type="hidden" name="action" value="remove_ip">
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars((string)$_SESSION['allowlist_tool_csrf'], ENT_QUOTES, 'UTF-8'); ?>">
            <label>IP address
                <input name="ip_address" required>
            </label>
            <button type="submit">Remove IP</button>
        </form>

        <h2>Create admin account</h2>
        <form method="post" class="grid">
            <input type="hidden" name="action" value="create_admin">
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars((string)$_SESSION['allowlist_tool_csrf'], ENT_QUOTES, 'UTF-8'); ?>">
            <label>Username
                <input name="username" required minlength="3" maxlength="30">
            </label>
            <label>Email
                <input type="email" name="email" required>
            </label>
            <label>Password (min 10 chars)
                <input type="password" name="password" required minlength="10">
            </label>
            <button type="submit">Create Admin</button>
        </form>

        <h2>Current allowlisted IPs</h2>
        <ul>
            <?php foreach ($allowed as $row): ?>
                <li><code><?php echo htmlspecialchars((string)$row['ip_address'], ENT_QUOTES, 'UTF-8'); ?></code>
                    - <?php echo htmlspecialchars((string)($row['label'] ?? ''), ENT_QUOTES, 'UTF-8'); ?>
                    (<?php echo htmlspecialchars((string)$row['added_at'], ENT_QUOTES, 'UTF-8'); ?>)
                </li>
            <?php endforeach; ?>
        </ul>
    <?php endif; ?>
</div>
</body>
</html>
