(() => {
    function createViewer() {
        const overlay = document.createElement('div');
        overlay.className = 'media-viewer-overlay';
        overlay.hidden = true;

        const modal = document.createElement('div');
        modal.className = 'media-viewer-modal';
        modal.setAttribute('role', 'dialog');
        modal.setAttribute('aria-modal', 'true');

        const toolbar = document.createElement('div');
        toolbar.className = 'media-viewer-toolbar';

        const title = document.createElement('p');
        title.className = 'media-viewer-title';
        title.textContent = 'Media viewer';

        const controls = document.createElement('div');
        controls.className = 'media-viewer-controls';

        const zoomOutButton = document.createElement('button');
        zoomOutButton.type = 'button';
        zoomOutButton.className = 'media-viewer-btn';
        zoomOutButton.textContent = '−';
        zoomOutButton.setAttribute('aria-label', 'Zoom out');

        const zoomLabel = document.createElement('span');
        zoomLabel.className = 'media-viewer-zoom';
        zoomLabel.textContent = '100%';

        const zoomInButton = document.createElement('button');
        zoomInButton.type = 'button';
        zoomInButton.className = 'media-viewer-btn';
        zoomInButton.textContent = '+';
        zoomInButton.setAttribute('aria-label', 'Zoom in');

        const resetButton = document.createElement('button');
        resetButton.type = 'button';
        resetButton.className = 'media-viewer-btn';
        resetButton.textContent = 'Reset';

        const downloadButton = document.createElement('button');
        downloadButton.type = 'button';
        downloadButton.className = 'media-viewer-btn';
        downloadButton.textContent = '⬇';
        downloadButton.setAttribute('aria-label', 'Download media');
        downloadButton.title = 'Download';
        downloadButton.disabled = true;

        const closeButton = document.createElement('button');
        closeButton.type = 'button';
        closeButton.className = 'media-viewer-btn media-viewer-close';
        closeButton.textContent = 'Close';

        controls.appendChild(zoomOutButton);
        controls.appendChild(zoomLabel);
        controls.appendChild(zoomInButton);
        controls.appendChild(resetButton);
        controls.appendChild(downloadButton);
        controls.appendChild(closeButton);

        toolbar.appendChild(title);
        toolbar.appendChild(controls);

        const stage = document.createElement('div');
        stage.className = 'media-viewer-stage';

        modal.appendChild(toolbar);
        modal.appendChild(stage);
        overlay.appendChild(modal);
        document.body.appendChild(overlay);

        return {
            overlay,
            title,
            stage,
            zoomOutButton,
            zoomInButton,
            resetButton,
            downloadButton,
            closeButton,
            zoomLabel,
        };
    }

    function initMediaViewer() {
        const clickTargets = document.querySelectorAll('.media-card > summary, .comment-media-card > summary');
        if (clickTargets.length === 0) {
            return;
        }

        const viewer = createViewer();

        let activeMediaEl = null;
        let lastFocused = null;
        let zoom = 1;
        let panX = 0;
        let panY = 0;
        let isDragging = false;
        let dragStartX = 0;
        let dragStartY = 0;
        let dragOriginX = 0;
        let dragOriginY = 0;
        let activeDownloadSrc = '';
        let activeDownloadName = '';

        const minZoom = 0.25;
        const maxZoom = 8;

        const clampZoom = (value) => {
            return Math.max(minZoom, Math.min(maxZoom, value));
        };

        const sanitizeFileName = (value) => {
            const raw = String(value || '').trim();
            if (raw === '') {
                return '';
            }

            return raw
                .replace(/[\\/:*?"<>|]+/g, '_')
                .replace(/\s+/g, ' ')
                .slice(0, 180)
                .trim();
        };

        const inferDownloadName = (src, kind, preferredName) => {
            const cleanPreferred = sanitizeFileName(preferredName);
            if (cleanPreferred !== '') {
                return cleanPreferred;
            }

            const fallbackExt = kind === 'video' ? 'webm' : 'jpg';

            try {
                const url = new URL(src, globalThis.location.href);
                const path = decodeURIComponent(url.pathname || '');
                const segment = path.split('/').filter(Boolean).pop() || '';
                const cleanSegment = sanitizeFileName(segment);
                if (cleanSegment !== '') {
                    if (/\.[a-z0-9]{2,6}$/i.test(cleanSegment)) {
                        return cleanSegment;
                    }
                    return `${cleanSegment}.${fallbackExt}`;
                }
            } catch {
                // ignore parsing errors and fallback below
            }

            return `media-${Date.now()}.${fallbackExt}`;
        };

        const stopDragging = () => {
            if (!isDragging) {
                return;
            }

            isDragging = false;
            viewer.stage.classList.remove('is-dragging');
            if (activeMediaEl?.tagName?.toLowerCase() === 'img') {
                if (zoom > 1) {
                    activeMediaEl.style.cursor = 'grab';
                } else {
                    activeMediaEl.style.cursor = 'zoom-in';
                }
            }
        };

        const applyZoom = () => {
            if (activeMediaEl?.tagName?.toLowerCase() !== 'img') {
                viewer.stage.classList.remove('is-dragging');
                viewer.zoomLabel.textContent = '100%';
                viewer.zoomInButton.disabled = true;
                viewer.zoomOutButton.disabled = true;
                viewer.resetButton.disabled = true;
                return;
            }

            if (zoom <= 1) {
                panX = 0;
                panY = 0;
                stopDragging();
            }

            activeMediaEl.style.transform = `translate(${panX}px, ${panY}px) scale(${zoom})`;
            let nextCursor = 'zoom-in';
            if (isDragging) {
                nextCursor = 'grabbing';
            } else if (zoom > 1) {
                nextCursor = 'grab';
            }
            activeMediaEl.style.cursor = nextCursor;
            viewer.zoomLabel.textContent = `${Math.round(zoom * 100)}%`;
            viewer.zoomInButton.disabled = zoom >= maxZoom;
            viewer.zoomOutButton.disabled = zoom <= minZoom;
            viewer.resetButton.disabled = Math.abs(zoom - 1) < 0.001;
        };

        const clearStage = () => {
            if (activeMediaEl?.tagName?.toLowerCase() === 'video') {
                try {
                    activeMediaEl.pause();
                    activeMediaEl.currentTime = 0;
                } catch {
                    // ignore cleanup errors
                }
            }

            stopDragging();
            viewer.stage.innerHTML = '';
            activeMediaEl = null;
            activeDownloadSrc = '';
            activeDownloadName = '';
            viewer.downloadButton.disabled = true;
            zoom = 1;
            panX = 0;
            panY = 0;
            applyZoom();
        };

        const closeViewer = () => {
            viewer.overlay.hidden = true;
            document.body.classList.remove('viewer-open');
            clearStage();
            if (lastFocused && typeof lastFocused.focus === 'function') {
                lastFocused.focus();
            }
        };

        const openMedia = (src, kind, label, preferredName) => {
            clearStage();

            let mediaEl;
            if (kind === 'video') {
                mediaEl = document.createElement('video');
                mediaEl.controls = true;
                mediaEl.preload = 'metadata';
                mediaEl.autoplay = true;
                mediaEl.loop = true;
                mediaEl.playsInline = true;
                mediaEl.src = src;
                mediaEl.className = 'media-viewer-video';
            } else {
                mediaEl = document.createElement('img');
                mediaEl.loading = 'eager';
                mediaEl.draggable = false;
                mediaEl.src = src;
                mediaEl.alt = label || 'Expanded image';
                mediaEl.className = 'media-viewer-image';
            }

            activeMediaEl = mediaEl;
            activeDownloadSrc = src;
            activeDownloadName = inferDownloadName(src, kind, preferredName);
            viewer.downloadButton.disabled = activeDownloadSrc === '';
            viewer.stage.appendChild(mediaEl);
            viewer.title.textContent = label || 'Media viewer';

            viewer.overlay.hidden = false;
            document.body.classList.add('viewer-open');
            zoom = 1;
            applyZoom();

            if (kind === 'video' && activeMediaEl instanceof HTMLVideoElement) {
                const startPlayback = () => {
                    const maybePromise = activeMediaEl.play();
                    if (maybePromise && typeof maybePromise.catch === 'function') {
                        maybePromise.catch(() => {
                            activeMediaEl.muted = true;
                            const retry = activeMediaEl.play();
                            if (retry && typeof retry.catch === 'function') {
                                retry.catch(() => {
                                    // user can press play manually if autoplay is blocked
                                });
                            }
                        });
                    }
                };

                if (activeMediaEl.readyState >= 2) {
                    startPlayback();
                } else {
                    activeMediaEl.addEventListener('canplay', startPlayback, { once: true });
                }
            }
        };

        clickTargets.forEach((summaryEl) => {
            summaryEl.addEventListener('click', (event) => {
                const details = summaryEl.parentElement;
                if (!(details instanceof HTMLElement)) {
                    return;
                }

                const thumbImg = summaryEl.querySelector('img');
                const thumbVideo = summaryEl.querySelector('video');
                const fullImg = details.querySelector('.media-expanded img, .comment-media-expanded img');
                const fullVideo = details.querySelector('.media-expanded video, .comment-media-expanded video');
                const mediaCaption = details.querySelector('.media-caption');

                let src = '';
                let kind = 'image';
                let label = 'Media viewer';
                let preferredName = '';

                if (thumbImg) {
                    src = (fullImg && (fullImg.currentSrc || fullImg.src))
                        || thumbImg.currentSrc
                        || thumbImg.src;
                    label = thumbImg.alt || 'Expanded image';
                } else if (thumbVideo) {
                    src = (fullVideo && (fullVideo.currentSrc || fullVideo.src))
                        || thumbVideo.currentSrc
                        || thumbVideo.src;
                    kind = 'video';
                    label = 'Expanded video';
                }

                if (mediaCaption) {
                    preferredName = mediaCaption.textContent || '';
                }

                if (src === '') {
                    return;
                }

                event.preventDefault();
                details.open = false;
                lastFocused = summaryEl;
                openMedia(src, kind, label, preferredName);
            });
        });

        viewer.closeButton.addEventListener('click', closeViewer);

        viewer.overlay.addEventListener('click', (event) => {
            if (event.target === viewer.overlay) {
                closeViewer();
            }
        });

        document.addEventListener('keydown', (event) => {
            if (viewer.overlay.hidden) {
                return;
            }

            if (event.key === 'Escape') {
                event.preventDefault();
                closeViewer();
            }
        });

        const bumpZoom = (delta) => {
            if (activeMediaEl?.tagName?.toLowerCase() !== 'img') {
                return;
            }

            zoom = clampZoom(zoom + delta);
            applyZoom();
        };

        viewer.zoomInButton.addEventListener('click', () => {
            bumpZoom(0.2);
        });

        viewer.zoomOutButton.addEventListener('click', () => {
            bumpZoom(-0.2);
        });

        viewer.resetButton.addEventListener('click', () => {
            zoom = 1;
            applyZoom();
        });

        viewer.downloadButton.addEventListener('click', () => {
            if (activeDownloadSrc === '') {
                return;
            }

            const anchor = document.createElement('a');
            anchor.href = activeDownloadSrc;
            if (activeDownloadName !== '') {
                anchor.download = activeDownloadName;
            }
            anchor.rel = 'noopener';
            document.body.appendChild(anchor);
            anchor.click();
            anchor.remove();
        });

        viewer.stage.addEventListener('wheel', (event) => {
            if (viewer.overlay.hidden || activeMediaEl?.tagName?.toLowerCase() !== 'img') {
                return;
            }

            event.preventDefault();
            const delta = event.deltaY < 0 ? 0.12 : -0.12;
            bumpZoom(delta);
        }, { passive: false });

        viewer.stage.addEventListener('mousedown', (event) => {
            if (event.button !== 0 || viewer.overlay.hidden || activeMediaEl?.tagName?.toLowerCase() !== 'img' || zoom <= 1) {
                return;
            }

            if (event.target !== activeMediaEl) {
                return;
            }

            isDragging = true;
            viewer.stage.classList.add('is-dragging');
            dragStartX = event.clientX;
            dragStartY = event.clientY;
            dragOriginX = panX;
            dragOriginY = panY;
            activeMediaEl.style.cursor = 'grabbing';
            event.preventDefault();
        });

        document.addEventListener('mousemove', (event) => {
            if (!isDragging || viewer.overlay.hidden || activeMediaEl?.tagName?.toLowerCase() !== 'img') {
                return;
            }

            panX = dragOriginX + (event.clientX - dragStartX);
            panY = dragOriginY + (event.clientY - dragStartY);
            applyZoom();
        });

        document.addEventListener('mouseup', () => {
            stopDragging();
        });
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initMediaViewer);
    } else {
        initMediaViewer();
    }
})();
