<?php

declare(strict_types=1);

if (!defined('APP_ROOT')) {
    define('APP_ROOT', dirname(__DIR__));
}
if (!defined('DATA_PATH')) {
    define('DATA_PATH', APP_ROOT . '/data');
}

function appIsInstalled(): bool
{
    return is_file(DATA_PATH . '/.installed.lock') && is_file(DATA_PATH . '/config.php');
}

function appDbPath(): string
{
    return DATA_PATH . '/site.sqlite';
}

function appDb(): PDO
{
    static $pdo = null;

    if ($pdo instanceof PDO) {
        return $pdo;
    }

    if (!is_dir(DATA_PATH)) {
        mkdir(DATA_PATH, 0755, true);
    }

    $pdo = new PDO('sqlite:' . appDbPath());
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
    $pdo->exec('PRAGMA foreign_keys = ON');
    $pdo->exec('PRAGMA journal_mode = WAL');

    return $pdo;
}

function dbNow(): string
{
    return gmdate('Y-m-d H:i:s');
}

function appInstallSchema(PDO $db): void
{
    $db->exec('CREATE TABLE IF NOT EXISTS settings (
        setting_key TEXT PRIMARY KEY,
        setting_value TEXT NOT NULL,
        updated_at TEXT NOT NULL
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT NOT NULL UNIQUE,
        email TEXT NOT NULL UNIQUE,
        display_name TEXT NOT NULL DEFAULT "",
        password_hash TEXT NOT NULL,
        role TEXT NOT NULL DEFAULT "user",
        is_banned INTEGER NOT NULL DEFAULT 0,
        email_verified_at TEXT,
        created_at TEXT NOT NULL,
        last_login_at TEXT
    )');

    try {
        $db->exec('ALTER TABLE users ADD COLUMN email_verified_at TEXT');
    } catch (Throwable $exception) {
        // already exists
    }

    try {
        $db->exec('ALTER TABLE users ADD COLUMN display_name TEXT NOT NULL DEFAULT ""');
    } catch (Throwable $exception) {
        // already exists
    }

    $db->exec('CREATE TABLE IF NOT EXISTS admin_allowed_ips (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ip_address TEXT NOT NULL UNIQUE,
        label TEXT,
        added_at TEXT NOT NULL
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS categories (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        slug TEXT NOT NULL UNIQUE,
        description TEXT,
        sort_order INTEGER NOT NULL DEFAULT 0,
        created_at TEXT NOT NULL
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS subcategories (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        category_id INTEGER NOT NULL,
        name TEXT NOT NULL,
        slug TEXT NOT NULL,
        description TEXT,
        sort_order INTEGER NOT NULL DEFAULT 0,
        created_at TEXT NOT NULL,
        UNIQUE(category_id, slug),
        FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS posts (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        subcategory_id INTEGER NOT NULL,
        user_id INTEGER NOT NULL,
        title TEXT NOT NULL,
        body TEXT,
        status TEXT NOT NULL DEFAULT "active",
        created_at TEXT NOT NULL,
        updated_at TEXT NOT NULL,
        FOREIGN KEY (subcategory_id) REFERENCES subcategories(id) ON DELETE CASCADE,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS media (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        post_id INTEGER NOT NULL,
        comment_id INTEGER,
        user_id INTEGER NOT NULL,
        storage_name TEXT NOT NULL UNIQUE,
        original_name TEXT NOT NULL,
        mime_type TEXT NOT NULL,
        media_kind TEXT NOT NULL,
        size_bytes INTEGER NOT NULL,
        created_at TEXT NOT NULL,
        FOREIGN KEY (post_id) REFERENCES posts(id) ON DELETE CASCADE,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    )');

    try {
        $db->exec('ALTER TABLE media ADD COLUMN comment_id INTEGER');
    } catch (Throwable $exception) {
        // already exists
    }

    $db->exec('CREATE TABLE IF NOT EXISTS comments (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        post_id INTEGER NOT NULL,
        user_id INTEGER NOT NULL,
        body TEXT NOT NULL,
        status TEXT NOT NULL DEFAULT "active",
        created_at TEXT NOT NULL,
        FOREIGN KEY (post_id) REFERENCES posts(id) ON DELETE CASCADE,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS contact_messages (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        email TEXT NOT NULL,
        message TEXT NOT NULL,
        ip_address TEXT,
        created_at TEXT NOT NULL
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS login_attempts (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT,
        ip_address TEXT NOT NULL,
        attempted_at INTEGER NOT NULL
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS rate_limits (
        rate_key TEXT PRIMARY KEY,
        request_count INTEGER NOT NULL,
        reset_at INTEGER NOT NULL
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS board_rules (
        subcategory_id INTEGER PRIMARY KEY,
        max_files INTEGER NOT NULL DEFAULT 4,
        max_file_bytes INTEGER NOT NULL DEFAULT 20971520,
        cooldown_seconds INTEGER NOT NULL DEFAULT 30,
        max_posts_per_hour INTEGER NOT NULL DEFAULT 30,
        max_comments_per_hour INTEGER NOT NULL DEFAULT 90,
        allow_video INTEGER NOT NULL DEFAULT 1,
        captcha_mode TEXT NOT NULL DEFAULT "low_trust",
        updated_at TEXT NOT NULL,
        FOREIGN KEY (subcategory_id) REFERENCES subcategories(id) ON DELETE CASCADE
    )');

    try {
        $db->exec('ALTER TABLE board_rules ADD COLUMN captcha_mode TEXT NOT NULL DEFAULT "low_trust"');
    } catch (Throwable $exception) {
        // already exists
    }

    $db->exec('CREATE TABLE IF NOT EXISTS email_verification_tokens (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL,
        token_hash TEXT NOT NULL,
        expires_at INTEGER NOT NULL,
        used_at INTEGER,
        created_at INTEGER NOT NULL,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS password_reset_tokens (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL,
        token_hash TEXT NOT NULL,
        expires_at INTEGER NOT NULL,
        used_at INTEGER,
        created_at INTEGER NOT NULL,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    )');

    $db->exec('CREATE TABLE IF NOT EXISTS audit_logs (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        actor_user_id INTEGER,
        action TEXT NOT NULL,
        details TEXT,
        created_at TEXT NOT NULL,
        FOREIGN KEY (actor_user_id) REFERENCES users(id) ON DELETE SET NULL
    )');

    $db->exec('CREATE INDEX IF NOT EXISTS idx_users_role ON users(role)');
    $db->exec('CREATE UNIQUE INDEX IF NOT EXISTS idx_users_display_name_unique ON users(display_name) WHERE display_name <> ""');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_posts_subcategory_created ON posts(subcategory_id, created_at DESC)');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_media_post ON media(post_id)');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_media_comment ON media(comment_id)');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_comments_post ON comments(post_id, created_at ASC)');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_login_attempts_ip_time ON login_attempts(ip_address, attempted_at)');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_rate_limits_reset ON rate_limits(reset_at)');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_verify_tokens_hash ON email_verification_tokens(token_hash)');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_verify_tokens_user ON email_verification_tokens(user_id, expires_at)');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_reset_tokens_hash ON password_reset_tokens(token_hash)');
    $db->exec('CREATE INDEX IF NOT EXISTS idx_reset_tokens_user ON password_reset_tokens(user_id, expires_at)');

    seedDefaultContent($db);
    upgradeLegacyBoardDefaults($db);
    normalizeLegacySlugs($db);
}

function upgradeLegacyBoardDefaults(PDO $db): void
{
    $flagKey = 'board_defaults_upgraded_20260214';
    $flagStmt = $db->prepare('SELECT setting_value FROM settings WHERE setting_key = ? LIMIT 1');
    $flagStmt->execute([$flagKey]);
    $flagValue = $flagStmt->fetchColumn();
    if ($flagValue === '1') {
        return;
    }

    $replacements = [
        'board_default_max_files' => ['old' => '4', 'new' => '10'],
        'board_default_cooldown_seconds' => ['old' => '30', 'new' => '10'],
        'board_default_max_posts_per_hour' => ['old' => '30', 'new' => '200'],
        'board_default_max_comments_per_hour' => ['old' => '90', 'new' => '500'],
    ];

    $readStmt = $db->prepare('SELECT setting_value FROM settings WHERE setting_key = ? LIMIT 1');
    $writeStmt = $db->prepare('INSERT INTO settings (setting_key, setting_value, updated_at) VALUES (?, ?, ?) ON CONFLICT(setting_key) DO UPDATE SET setting_value = excluded.setting_value, updated_at = excluded.updated_at');
    $now = dbNow();

    foreach ($replacements as $key => $pair) {
        $readStmt->execute([$key]);
        $current = $readStmt->fetchColumn();

        if ($current === false) {
            $writeStmt->execute([$key, (string)$pair['new'], $now]);
            continue;
        }

        if ((string)$current === (string)$pair['old']) {
            $writeStmt->execute([$key, (string)$pair['new'], $now]);
        }
    }

    $writeStmt->execute([$flagKey, '1', $now]);
}

function normalizeLegacySlugs(PDO $db): void
{
    $categories = $db->query('SELECT id, name, slug FROM categories')->fetchAll();
    foreach ($categories as $category) {
        $slug = (string)$category['slug'];
        $name = trim((string)$category['name']);
        if ($name === '' || !preg_match('/^item-[a-f0-9]{6}$/', $slug)) {
            continue;
        }

        $newSlug = uniqueSlug($db, 'categories', $name);
        if ($newSlug !== $slug) {
            $stmt = $db->prepare('UPDATE categories SET slug = ? WHERE id = ?');
            $stmt->execute([$newSlug, (int)$category['id']]);
        }
    }

    $subs = $db->query('SELECT id, category_id, name, slug FROM subcategories')->fetchAll();
    foreach ($subs as $sub) {
        $slug = (string)$sub['slug'];
        $name = trim((string)$sub['name']);
        $categoryId = (int)$sub['category_id'];
        if ($name === '' || $categoryId <= 0 || !preg_match('/^item-[a-f0-9]{6}$/', $slug)) {
            continue;
        }

        $newSlug = uniqueSlug($db, 'subcategories', $name, $categoryId);
        if ($newSlug !== $slug) {
            $stmt = $db->prepare('UPDATE subcategories SET slug = ? WHERE id = ?');
            $stmt->execute([$newSlug, (int)$sub['id']]);
        }
    }
}

function seedDefaultContent(PDO $db): void
{
    $now = dbNow();

    $settings = [
        'site_name' => 'Untitled Imageboard',
        'site_tagline' => 'Celestial low-glow community board',
        'site_theme' => 'celestial-matrix',
        'site_favicon_url' => '',
        'site_favicon_storage' => '',
        'site_favicon_mime' => 'image/x-icon',
        'brand_display_mode' => 'name_only',
        'site_assets_version' => '1',
        'legal_email' => 'owner@example.com',
        'site_meta_description' => 'Community imageboard with secure media uploads and moderation.',
        'about_page_content' => 'Write your About page in Admin Settings.',
        'dmca_page_content' => 'Write your DMCA policy in Admin Settings.',
        'legal_page_content' => 'Write your Legal page in Admin Settings.',
        'terms_page_content' => 'Write your Terms of Service in Admin Settings.',
        'privacy_page_content' => 'Write your Privacy Policy in Admin Settings.',
        'legal_notice_page_content' => 'Write your Legal Notice in Admin Settings.',
        'post_disclaimer_enabled' => '0',
        'post_disclaimer_title' => 'Content Disclaimer',
        'post_disclaimer_body' => "The content of this section may include mature language or imagery and may not be suitable for all audiences.\n\nBy continuing, you confirm you are legally allowed to access this content, and you agree to use this website at your own risk and in compliance with the site rules.",
        'pagination_per_page' => '20',
        'board_default_max_files' => '10',
        'board_default_max_file_mb' => '20',
        'board_default_max_file_bytes' => '20971520',
        'board_default_cooldown_seconds' => '10',
        'board_default_max_posts_per_hour' => '200',
        'board_default_max_comments_per_hour' => '500',
        'board_default_captcha_mode' => 'low_trust',
        'auto_approve_new_posts' => '1',
        'anti_spam_posts_per_10m' => '8',
        'anti_spam_comments_per_10m' => '20',
        'anti_spam_contact_per_hour' => '5',
        'low_trust_age_hours' => '72',
        'low_trust_min_activity' => '5',
        'email_verification_enabled' => '0',
        'password_reset_enabled' => '0',
        'contact_forward_enabled' => '0',
        'robots_custom_rules' => '',
        'sitemap_auto_ping_enabled' => '0',
        'sitemap_ping_services' => 'https://www.bing.com/ping?sitemap={sitemap_url}',
        'smtp_host' => '',
        'smtp_port' => '587',
        'smtp_username' => '',
        'smtp_password_enc' => '',
        'smtp_security' => 'tls',
        'smtp_from_email' => '',
    ];

    $stmt = $db->prepare('INSERT OR IGNORE INTO settings (setting_key, setting_value, updated_at) VALUES (:k, :v, :u)');
    foreach ($settings as $k => $v) {
        $stmt->execute([':k' => $k, ':v' => $v, ':u' => $now]);
    }
}

function settingGet(string $key, string $default = ''): string
{
    static $cache = [];

    if (array_key_exists($key, $cache)) {
        return $cache[$key];
    }

    $stmt = appDb()->prepare('SELECT setting_value FROM settings WHERE setting_key = ? LIMIT 1');
    $stmt->execute([$key]);
    $value = $stmt->fetchColumn();
    $cache[$key] = $value !== false ? (string)$value : $default;

    return $cache[$key];
}

function settingSet(string $key, string $value): void
{
    $now = dbNow();
    $stmt = appDb()->prepare('INSERT INTO settings (setting_key, setting_value, updated_at) VALUES (?, ?, ?) ON CONFLICT(setting_key) DO UPDATE SET setting_value = excluded.setting_value, updated_at = excluded.updated_at');
    $stmt->execute([$key, $value, $now]);
}

function logAudit(?int $actorUserId, string $action, string $details = ''): void
{
    $stmt = appDb()->prepare('INSERT INTO audit_logs (actor_user_id, action, details, created_at) VALUES (?, ?, ?, ?)');
    $stmt->execute([$actorUserId, $action, $details, dbNow()]);
}

function slugify(string $text): string
{
    $original = trim($text);
    $candidate = $original;

    if (function_exists('iconv')) {
        $converted = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $original);
        if ($converted !== false && trim((string)$converted) !== '') {
            $candidate = (string)$converted;
        }
    }

    $text = strtolower(trim($candidate));
    $text = preg_replace('/[^a-z0-9]+/', '-', $text) ?? '';
    $text = trim($text, '-');

    if ($text === '') {
        $unicode = mb_strtolower($original, 'UTF-8');
        $unicode = preg_replace('/[^\p{L}\p{N}]+/u', '-', $unicode) ?? '';
        $text = trim($unicode, '-');
    }

    if ($text === '') {
        $text = 'item-' . bin2hex(random_bytes(3));
    }

    return $text;
}

function uniqueSlug(PDO $db, string $table, string $slug, ?int $categoryId = null): string
{
    $base = slugify($slug);
    $candidate = $base;
    $counter = 1;

    while (true) {
        if ($table === 'subcategories' && $categoryId !== null) {
            $stmt = $db->prepare('SELECT 1 FROM subcategories WHERE category_id = ? AND slug = ? LIMIT 1');
            $stmt->execute([$categoryId, $candidate]);
        } else {
            $stmt = $db->prepare("SELECT 1 FROM {$table} WHERE slug = ? LIMIT 1");
            $stmt->execute([$candidate]);
        }

        if (!$stmt->fetchColumn()) {
            return $candidate;
        }

        $counter++;
        $candidate = $base . '-' . $counter;
    }
}
