<?php

declare(strict_types=1);

require_once dirname(__DIR__) . '/includes/database.php';

if (appIsInstalled()) {
    http_response_code(404);
    echo '<h1>404 Not Found</h1>';
    exit;
}

if (session_status() !== PHP_SESSION_ACTIVE) {
    session_name('W22SETUP');
    session_start();
}

if (empty($_SESSION['setup_csrf'])) {
    $_SESSION['setup_csrf'] = bin2hex(random_bytes(32));
}

function setupClientIp(): string
{
    $keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
    foreach ($keys as $key) {
        if (empty($_SERVER[$key])) {
            continue;
        }

        $candidate = trim(explode(',', (string)$_SERVER[$key])[0]);
        if (filter_var($candidate, FILTER_VALIDATE_IP)) {
            return $candidate;
        }
    }

    return '0.0.0.0';
}

function setupHashPassword(string $password): string
{
    if (defined('PASSWORD_ARGON2ID')) {
        return password_hash($password, PASSWORD_ARGON2ID);
    }

    return password_hash($password, PASSWORD_BCRYPT);
}

$error = '';

if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET')) === 'POST') {
    $token = (string)($_POST['csrf_token'] ?? '');
    if ($token === '' || !hash_equals((string)$_SESSION['setup_csrf'], $token)) {
        $error = 'Invalid CSRF token.';
    } else {
        $siteName = trim((string)($_POST['site_name'] ?? 'Untitled Imageboard'));
        $siteTagline = trim((string)($_POST['site_tagline'] ?? 'Celestial low-glow community board'));
        $legalEmail = trim((string)($_POST['legal_email'] ?? 'owner@example.com'));
        $adminUser = trim((string)($_POST['admin_username'] ?? ''));
        $adminEmail = trim((string)($_POST['admin_email'] ?? ''));
        $adminPass = (string)($_POST['admin_password'] ?? '');
        $adminPassConfirm = (string)($_POST['admin_password_confirm'] ?? '');
        $addCurrentIp = isset($_POST['add_current_ip']);

        if ($siteName === '') {
            $error = 'Site name is required.';
        } elseif (!preg_match('/^\w{3,30}$/', $adminUser)) {
            $error = 'Admin username must be 3-30 chars and only letters/numbers/_.';
        } elseif (!filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
            $error = 'Admin email is invalid.';
        } elseif (strlen($adminPass) < 10) {
            $error = 'Admin password must be at least 10 characters.';
        } elseif (!hash_equals($adminPass, $adminPassConfirm)) {
            $error = 'Admin passwords do not match.';
        } elseif (!filter_var($legalEmail, FILTER_VALIDATE_EMAIL)) {
            $error = 'Legal email is invalid.';
        } else {
            try {
                $db = appDb();
                appInstallSchema($db);

                $appSecret = bin2hex(random_bytes(48));
                $allowlistToolToken = bin2hex(random_bytes(24));
                $configBody = "<?php\nreturn [\n"
                    . "    'APP_SECRET' => " . var_export($appSecret, true) . ",\n"
                    . "    'ALLOWLIST_TOOL_TOKEN' => " . var_export($allowlistToolToken, true) . ",\n"
                    . "    'FORCE_HTTPS' => '0',\n"
                    . "    'MAX_UPLOAD_BYTES' => 20971520,\n"
                    . "    'COOKIE_DOMAIN' => '',\n"
                    . "];\n";

                if (!is_dir(DATA_PATH)) {
                    mkdir(DATA_PATH, 0755, true);
                }

                file_put_contents(DATA_PATH . '/config.php', $configBody, LOCK_EX);

                $existsStmt = $db->prepare('SELECT id FROM users WHERE username = ? OR email = ? LIMIT 1');
                $existsStmt->execute([$adminUser, $adminEmail]);
                if ($existsStmt->fetch()) {
                    throw new RuntimeException('Admin username or email already exists in DB.');
                }

                $db->prepare('INSERT INTO users (username, email, password_hash, role, is_banned, created_at) VALUES (?, ?, ?, "admin", 0, ?)')
                    ->execute([$adminUser, $adminEmail, setupHashPassword($adminPass), dbNow()]);

                settingSet('site_name', $siteName);
                settingSet('site_tagline', $siteTagline);
                settingSet('legal_email', $legalEmail);

                if ($addCurrentIp) {
                    $ip = setupClientIp();
                    $db->prepare('INSERT INTO admin_allowed_ips (ip_address, label, added_at) VALUES (?, ?, ?) ON CONFLICT(ip_address) DO UPDATE SET label = excluded.label, added_at = excluded.added_at')
                        ->execute([$ip, 'Setup', dbNow()]);
                }

                file_put_contents(DATA_PATH . '/.installed.lock', dbNow() . PHP_EOL, LOCK_EX);

                $_SESSION['allowlist_token_notice'] = $allowlistToolToken;
                header('Location: /admin?installed=1');
                exit;
            } catch (Throwable $exception) {
                $error = 'Setup failed: ' . $exception->getMessage();
            }
        }
    }
}

$currentIp = setupClientIp();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Website22 Setup</title>
    <style>
        body { font-family: Inter, Arial, sans-serif; background: #0a0f1f; color: #e7ecff; margin: 0; padding: 24px; }
        .card { max-width: 760px; margin: 0 auto; background: #101833; border: 1px solid #2d3a6a; border-radius: 14px; padding: 20px; }
        h1 { margin-top: 0; }
        .grid { display: grid; gap: 12px; }
        label { display: grid; gap: 6px; font-weight: 600; }
        input, textarea { width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #30406f; background: #0a1129; color: #f2f5ff; }
        button { margin-top: 6px; padding: 10px 16px; border: 0; border-radius: 9px; background: linear-gradient(135deg,#52c6ff,#7f86ff); color: #071022; font-weight: 700; cursor: pointer; }
        .error { background: #3a1624; border: 1px solid #7f2a48; color: #ffbdd1; padding: 10px; border-radius: 8px; margin-bottom: 12px; }
        .muted { color: #aeb8de; font-size: 0.95rem; }
    </style>
</head>
<body>
<div class="card">
    <h1>Website22 First-Time Setup</h1>
    <p class="muted">After setup: <strong>delete or restrict /public/setup.php</strong>. Admin route is guarded by IP allowlist and login.</p>

    <?php if ($error !== ''): ?>
        <div class="error"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></div>
    <?php endif; ?>

    <form method="post" class="grid">
        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars((string)$_SESSION['setup_csrf'], ENT_QUOTES, 'UTF-8'); ?>">

        <h2>Site</h2>
        <label>Site name
            <input name="site_name" required value="<?php echo htmlspecialchars((string)($_POST['site_name'] ?? 'Untitled Imageboard'), ENT_QUOTES, 'UTF-8'); ?>">
        </label>
        <label>Tagline
            <input name="site_tagline" value="<?php echo htmlspecialchars((string)($_POST['site_tagline'] ?? 'Celestial low-glow community board'), ENT_QUOTES, 'UTF-8'); ?>">
        </label>
        <label>Legal/DMCA contact email
            <input type="email" name="legal_email" required value="<?php echo htmlspecialchars((string)($_POST['legal_email'] ?? 'owner@example.com'), ENT_QUOTES, 'UTF-8'); ?>">
        </label>

        <h2>Admin account</h2>
        <label>Admin username
            <input name="admin_username" required minlength="3" maxlength="30" value="<?php echo htmlspecialchars((string)($_POST['admin_username'] ?? ''), ENT_QUOTES, 'UTF-8'); ?>">
        </label>
        <label>Admin email
            <input type="email" name="admin_email" required value="<?php echo htmlspecialchars((string)($_POST['admin_email'] ?? ''), ENT_QUOTES, 'UTF-8'); ?>">
        </label>
        <label>Admin password (min 10 chars)
            <input type="password" name="admin_password" required minlength="10">
        </label>
        <label>Confirm password
            <input type="password" name="admin_password_confirm" required minlength="10">
        </label>

        <label>
            <input type="checkbox" name="add_current_ip" checked>
            Add my current IP (<code><?php echo htmlspecialchars($currentIp, ENT_QUOTES, 'UTF-8'); ?></code>) to admin allowlist
        </label>

        <button type="submit">Install Website</button>
    </form>
</div>
</body>
</html>
